import axios from 'axios'
import { ElNotification, ElMessageBox, ElMessage, ElLoading } from 'element-plus'
import { getToken } from '@/utils/auth'
import errorCode from '@/utils/errorCode'
import { tansParams, blobValidate } from '@/utils/ruoyi'
import cache from '@/plugins/cache'
import { saveAs } from 'file-saver'
import useUserStore from '@/store/modules/user'
import { fsClientAuth } from '@/api'
import router from '@/router'

let downloadLoadingInstance;
// 是否显示重新登录
export let isRelogin = { show: false };

// 促销系统后台 baseURL
export const promotionBaseURL = import.meta.env.VITE_APP_PROMOTION

// 错误消息队列和状态管理
let errorMessageQueue = [];
let isShowingError = false;
let lastErrorTime = 0;
const ERROR_DEBOUNCE_TIME = 1000; // 1秒内的错误视为同一批

// 显示错误消息（单例模式）
function showErrorMessage(message, type = 'error') {
  const now = Date.now();

  // 如果正在显示错误消息，或者距离上次错误时间很短，将消息加入队列
  if (isShowingError || (now - lastErrorTime < ERROR_DEBOUNCE_TIME)) {
    // 检查队列中是否已有相同消息，避免重复
    const existingIndex = errorMessageQueue.findIndex(item => item.message === message);
    if (existingIndex === -1) {
      errorMessageQueue.push({ message, type });
    }
    return;
  }

  // 标记为正在显示
  isShowingError = true;
  lastErrorTime = now;

  // 显示当前错误消息
  ElMessage({
    message: message,
    type: type,
    duration: 3000, // 适当延长显示时间
    onClose: () => {
      isShowingError = false;

      // 检查队列中是否有待显示的消息
      if (errorMessageQueue.length > 0) {
        // 延迟一段时间再显示下一条，避免消息切换太快
        setTimeout(() => {
          const nextError = errorMessageQueue.shift();
          if (nextError) {
            showErrorMessage(nextError.message, nextError.type);
          }
        }, 500);
      }
    }
  });
}

// 清空错误消息队列（可选，在某些场景下可能需要）
export function clearErrorMessageQueue() {
  errorMessageQueue = [];
  isShowingError = false;
}

axios.defaults.headers['Content-Type'] = 'application/json;charset=utf-8'
// 创建axios实例
const service = axios.create({
  // axios中请求配置有baseURL选项，表示请求URL公共部分
  baseURL: import.meta.env.VITE_APP_BASE_API,
  // 超时
  timeout: 20000
})

// request拦截器
service.interceptors.request.use(config => {
  // 是否需要设置 token
  const isToken = (config.headers || {}).isToken === false
  // 是否需要防止数据重复提交
  const isRepeatSubmit = (config.headers || {}).repeatSubmit === false
  if (getToken() && !isToken) {
    if (config.baseURL === promotionBaseURL || config.baseURL.startsWith(promotionBaseURL)) {
      // 促销后台不能带 Bearer 前缀
      config.headers['Authorization'] = getToken()
    } else {
      config.headers['Authorization'] = 'Bearer ' + getToken()
    }
  }
  // get请求映射params参数
  if (config.method === 'get' && config.params) {
    let url = config.url + '?' + tansParams(config.params);
    url = url.slice(0, -1);
    config.params = {};
    config.url = url;
  }
  if (!isRepeatSubmit && (config.method === 'post' || config.method === 'put')) {
    const requestObj = {
      url: config.url,
      data: typeof config.data === 'object' ? JSON.stringify(config.data) : config.data,
      time: new Date().getTime()
    }
    const requestSize = Object.keys(JSON.stringify(requestObj)).length; // 请求数据大小
    const limitSize = 5 * 1024 * 1024; // 限制存放数据5M
    if (requestSize >= limitSize) {
      console.warn(`[${config.url}]: ` + '请求数据大小超出允许的5M限制，无法进行防重复提交验证。')
      return config;
    }
    const sessionObj = cache.session.getJSON('sessionObj')
    if (sessionObj === undefined || sessionObj === null || sessionObj === '') {
      cache.session.setJSON('sessionObj', requestObj)
    } else {
      const s_url = sessionObj.url;                // 请求地址
      const s_data = sessionObj.data;              // 请求数据
      const s_time = sessionObj.time;              // 请求时间
      const interval = 1000;                       // 间隔时间(ms)，小于此时间视为重复提交
      if (s_data === requestObj.data && requestObj.time - s_time < interval && s_url === requestObj.url) {
        const message = '数据正在处理，请勿重复提交';
        console.warn(`[${s_url}]: ` + message)
        // return Promise.reject(new Error(message))
      } else {
        cache.session.setJSON('sessionObj', requestObj)
      }
    }
  }
  return config
}, error => {
  Promise.reject(error)
})

// 响应拦截器
service.interceptors.response.use(async res => {
  // 未设置状态码则默认成功状态
  const code = res.data.code || 200;
  // 获取错误信息
  const msg = errorCode[code] || res.data.msg || errorCode['default']
  // 二进制数据则直接返回
  if (res.request.responseType === 'blob' || res.request.responseType === 'arraybuffer') {
    return res.data
  }
  if (code === 401) {
    // ... existing 401 handling code ...
  } else if (code === 403) {
    showErrorMessage(msg, 'error')
    return Promise.reject(new Error(msg))
  } else if (code === 500) {
    showErrorMessage(msg, 'error')
    return Promise.reject(new Error(msg))
  } else if (code === 601) {
    showErrorMessage(msg, 'warning')
    return Promise.reject(new Error(msg))
  } else if (code !== 200) {
    showErrorMessage(msg, 'error')
    return Promise.reject('error')
  } else {
    return Promise.resolve(res.data)
  }
},
  error => {
    let { message } = error;
    if (message == "Network Error") {
      message = "后端接口连接异常";
    } else if (message.includes("timeout")) {
      message = "系统接口请求超时";
    } else if (message.includes("Request failed with status code")) {
      message = "系统接口" + message.substr(message.length - 3) + "异常";
    }
    showErrorMessage(message, 'error')
    return Promise.reject(error)
  }
)

// 通用下载方法
export function download(url, params, filename, config) {
  downloadLoadingInstance = ElLoading.service({ text: "正在下载数据，请稍候", background: "rgba(0, 0, 0, 0.7)", })
  return service.post(url, params, {
    transformRequest: [(params) => { return tansParams(params) }],
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    responseType: 'blob',
    ...config
  }).then(async (data) => {
    const isBlob = blobValidate(data);
    if (isBlob) {
      const blob = new Blob([data])
      saveAs(blob, filename)
    } else {
      const resText = await data.text();
      const rspObj = JSON.parse(resText);
      const errMsg = errorCode[rspObj.code] || rspObj.msg || errorCode['default']
      ElMessage.error(errMsg);
    }
    downloadLoadingInstance.close();
  }).catch((r) => {
    console.error(r)
    ElMessage.error('下载文件出现错误，请联系管理员！')
    downloadLoadingInstance.close();
  })
}

export default service
