package com.sfa.system.domain.dao.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.sfa.common.core.constant.UserConstants;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.StringUtils;
import com.sfa.common.core.utils.bean.BeanUtils;
import com.sfa.common.core.web.domain.PageInfo;
import com.sfa.common.core.web.page.TableSupport;
import com.sfa.system.domain.dao.ISysPostDao;
import com.sfa.system.domain.entity.SysPost;
import com.sfa.system.domain.mapper.SysPostMapper;
import com.sfa.system.domain.mapper.SysUserMapper;
import com.sfa.system.domain.mapper.SysUserPostMapper;
import com.sfa.system.pojo.response.SysPostDto;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 岗位信息 服务层处理
 * 
 * @author ruoyi
 */
@Service
public class SysPostDaoImpl implements ISysPostDao
{
    @Autowired
    private SysPostMapper postMapper;
    @Autowired
    private SysUserPostMapper userPostMapper;
    @Autowired
    private SysUserMapper sysUserMapper;

    /**
     * 查询岗位信息集合
     * 
     * @param post 岗位信息
     * @return 岗位信息集合
     */
    @Override
    public PageInfo selectPostList(SysPost post)
    {
        LambdaQueryWrapper<SysPost> qw = buildWrapper(post);
        Page<SysPost> pageR = postMapper.selectPage(TableSupport.pageI(), qw);
        PageInfo<SysPost> pageInfo = new PageInfo<>(pageR);
        return pageInfo;
    }

    private LambdaQueryWrapper<SysPost> buildWrapper(SysPost post) {
        LambdaQueryWrapper<SysPost> qw = new LambdaQueryWrapper<>();
        if (StringUtils.isNotBlank(post.getStatus())){
            qw.eq(SysPost::getStatus,post.getStatus());
        }
        if (StringUtils.isNotBlank(post.getPostCode())){
            qw.like(SysPost::getPostCode,post.getPostCode());
        }
        if (StringUtils.isNotBlank(post.getPostName())){
            qw.like(SysPost::getPostName,post.getPostName());
        }
        return qw;
    }

    /**
     * 查询所有岗位
     * 
     * @return 岗位列表
     */
    @Override
    public List<SysPost> selectPostAll()
    {
        return postMapper.selectPostAll();
    }

    /**
     * 通过岗位ID查询岗位信息
     * 
     * @param postId 岗位ID
     * @return 角色对象信息
     */
    @Override
    public SysPost selectPostById(Long postId)
    {
        return postMapper.selectPostById(postId);
    }

    /**
     * 根据用户ID获取岗位选择框列表
     * 
     * @param userId 用户ID
     * @return 选中岗位ID列表
     */
    @Override
    public List<Long> selectPostListByUserId(Long userId)
    {
        return postMapper.selectPostListByUserId(userId);
    }

    /**
     * 校验岗位名称是否唯一
     * 
     * @param post 岗位信息
     * @return 结果
     */
    @Override
    public boolean checkPostNameUnique(SysPost post)
    {
        Long postId = StringUtils.isNull(post.getPostId()) ? -1L : post.getPostId();
        SysPost info = postMapper.checkPostNameUnique(post.getPostName());
        if (StringUtils.isNotNull(info) && info.getPostId().longValue() != postId.longValue())
        {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }

    /**
     * 校验岗位编码是否唯一
     * 
     * @param post 岗位信息
     * @return 结果
     */
    @Override
    public boolean checkPostCodeUnique(SysPost post)
    {
        Long postId = StringUtils.isNull(post.getPostId()) ? -1L : post.getPostId();
        SysPost info = postMapper.checkPostCodeUnique(post.getPostCode());
        if (StringUtils.isNotNull(info) && info.getPostId().longValue() != postId.longValue())
        {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }

    /**
     * 通过岗位ID查询岗位使用数量
     * 
     * @param postId 岗位ID
     * @return 结果
     */
    @Override
    public int countUserPostById(Long postId)
    {
        return userPostMapper.countUserPostById(postId);
    }

    /**
     * 删除岗位信息
     * 
     * @param postId 岗位ID
     * @return 结果
     */
    @Override
    public int deletePostById(Long postId)
    {
        return postMapper.deletePostById(postId);
    }

    /**
     * 批量删除岗位信息
     * 
     * @param postIds 需要删除的岗位ID
     * @return 结果
     */
    @Override
    public int deletePostByIds(Long[] postIds)
    {
        for (Long postId : postIds)
        {
            SysPost post = selectPostById(postId);
            if (countUserPostById(postId) > 0)
            {
                throw new ServiceException(String.format("%1$s已分配,不能删除", post.getPostName()));
            }
        }
        return postMapper.deletePostByIds(postIds);
    }

    /**
     * 新增保存岗位信息
     * 
     * @param post 岗位信息
     * @return 结果
     */
    @Override
    public int insertPost(SysPost post)
    {
        return postMapper.insert(post);
    }

    /**
     * 修改保存岗位信息
     * 
     * @param post 岗位信息
     * @return 结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public int updatePost(SysPost post)
    {
        // 修改岗位、user绑定的岗位信息
        postMapper.updateById(post);
        sysUserMapper.updateUserPostByPostId(post.getPostId(),post.getPostName(),post.getRuleId(),post.getRuleName());
        return 1;
    }

    @Override
    public SysPostDto selectPostByPostCode(String postCode) {
        SysPost sysPost = postMapper.selectPostByPostCode(postCode);
        return BeanUtils.transitionDto(sysPost,SysPostDto.class);
    }

    @Override
    public SysPostDto selectByPostId(Long postId) {
        SysPost sysPost = postMapper.selectById(postId);
        return BeanUtils.transitionDto(sysPost,SysPostDto.class);
    }

}
