package com.sfa.system.domain.dao.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.sfa.common.core.enums.ECode;
import com.sfa.common.core.exception.CheckedException;
import com.sfa.common.core.exception.WXLSQLException;
import com.sfa.common.core.web.domain.PageInfo;
import com.sfa.common.core.web.page.TableSupport;
import com.sfa.system.domain.dao.IMdmKqRuleDao;
import com.sfa.system.domain.entity.MdmKqRule;
import com.sfa.system.domain.mapper.MdmKqRuleMapper;
import com.sfa.system.domain.wq.MdmKqRuleWq;
import com.sfa.system.pojo.request.MdmKqRuleVo;
import com.sfa.system.api.pojo.response.MdmKqRuleDto;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author : liqiulin
 * @date : 2024-10-29 14
 * @describe :
 */
@Service
public class MdmKqRuleDaoImpl implements IMdmKqRuleDao {
    @Autowired
    private MdmKqRuleMapper mdmKqRuleMapper;

    /**
     * 查询考勤规则
     *
     * @param ruleId 考勤规则主键
     * @return 考勤规则
     */
    @Override
    public MdmKqRuleDto selectMdmKqRuleByRuleId(Long ruleId) {
        MdmKqRule mdmKqRule = mdmKqRuleMapper.selectById(ruleId);
        return transitionDto(mdmKqRule);
    }

    @Override
    public MdmKqRuleDto selectMdmKqRuleByRuleNum(String ruleNum) {
        MdmKqRule mdmKqRule = mdmKqRuleMapper.selectByRuleNum(ruleNum);
        return transitionDto(mdmKqRule);
    }

    /**
     * 查询考勤规则列表
     *
     * @param mdmKqRuleWq 考勤规则查询条件
     * @return 考勤规则
     */
    @Override
    public PageInfo selectMdmKqRuleList(MdmKqRuleWq mdmKqRuleWq) {
        LambdaQueryWrapper<MdmKqRule> qw = buildWrapper(mdmKqRuleWq);
        Page<MdmKqRule> pageR = mdmKqRuleMapper.selectPage(TableSupport.pageI(), qw);
        PageInfo<MdmKqRule> pageInfo = new PageInfo<>(pageR);
        return pageInfo;
    }

    /**
     * 新增考勤规则
     *
     * @param mdmKqRuleVo 考勤规则
     * @return 结果
     */
    @Override
    public int insertMdmKqRule(MdmKqRuleVo mdmKqRuleVo) {
        LambdaQueryWrapper<MdmKqRule> qw = buildWrapper(new MdmKqRuleWq().setRuleName(mdmKqRuleVo.getRuleName()));
        MdmKqRule mdmKqRule = mdmKqRuleMapper.selectOne(qw);
        if (!Objects.isNull(mdmKqRule)){
            throw new CheckedException(ECode.DB_TABLE_UNIQUE_EXIST, "考勤名称");
        }

        try {
            MdmKqRule mdo = new MdmKqRule();
            BeanUtils.copyProperties(mdmKqRuleVo, mdo);
            return mdmKqRuleMapper.insert(mdo);
        } catch (Exception e) {
            String eMsg = e.getCause().getMessage();
            if (eMsg.contains("for key 'mdm_kq_rule_unique'")) {
                throw new WXLSQLException(ECode.DB_TABLE_UNIQUE_EXIST, "考勤编码");
            }
            return -1;
        }
    }

    /**
     * 修改考勤规则
     *
     * @param mdmKqRuleVo 考勤规则
     * @return 结果
     */
    @Override
    public int updateMdmKqRule(MdmKqRuleVo mdmKqRuleVo) {
        LambdaQueryWrapper<MdmKqRule> qw = new LambdaQueryWrapper<>();
        qw.eq(MdmKqRule::getRuleId,mdmKqRuleVo.getRuleId())
                .eq(MdmKqRule::getFlag,mdmKqRuleVo.getFlag());
        MdmKqRule mdmKqRuleDo = mdmKqRuleMapper.selectOne(qw);
        if (Objects.isNull(mdmKqRuleDo)){
            throw new CheckedException(ECode.DB_TABLE_FLAG_NOTEXIST);
        }
        MdmKqRule mdo = new MdmKqRule();
        BeanUtils.copyProperties(mdmKqRuleVo, mdo);
        mdo.setFlag(mdo.getFlag()+1);
        return mdmKqRuleMapper.updateById(mdo);
    }

    /**
     * 批量删除考勤规则
     *
     * @param ruleIds 需要删除的考勤规则主键
     * @return 结果
     */
    @Override
    public int deleteMdmKqRuleByRuleIds(Long[] ruleIds) {
        return mdmKqRuleMapper.deleteMdmKqRuleByRuleIds(ruleIds);
    }

    /**
     * 删除考勤规则信息
     *
     * @param ruleId 考勤规则主键
     * @return 结果
     */
    @Override
    public int deleteMdmKqRuleByRuleId(Long ruleId) {
        return mdmKqRuleMapper.deleteMdmKqRuleByRuleId(ruleId);
    }



    private LambdaQueryWrapper<MdmKqRule> buildWrapper(MdmKqRuleWq mdmKqRuleWq) {
        LambdaQueryWrapper<MdmKqRule> queryWrapper = new LambdaQueryWrapper<>();
        if (!StringUtils.isBlank(mdmKqRuleWq.getRuleName())){
            queryWrapper.eq(MdmKqRule::getRuleName,mdmKqRuleWq.getRuleName());
        }
        if (!StringUtils.isBlank(mdmKqRuleWq.getRuleNameLike())){
            queryWrapper.like(MdmKqRule::getRuleName,mdmKqRuleWq.getRuleNameLike());
        }
        if (!StringUtils.isBlank(mdmKqRuleWq.getCreateByLike())){
            queryWrapper.like(MdmKqRule::getCreateBy,mdmKqRuleWq.getCreateByLike());
        }
        queryWrapper.eq(MdmKqRule::getDelFlag,"0");
        return queryWrapper;
    }

    /**
     * DO to DTO (单个对象)
     *
     * @param mdos DO对象List
     * @return DTO对象
     */
    private List<MdmKqRuleDto> transitionDtos(List<MdmKqRule> mdos) {
        if (CollectionUtils.isEmpty(mdos)) {
            return new ArrayList<>();
        }

        List<MdmKqRuleDto> dtos = new ArrayList<>(mdos.size() * 2);
        for (MdmKqRule mdo : mdos) {
            dtos.add(transitionDto(mdo));
        }
        return dtos;
    }

    private MdmKqRuleDto transitionDto(MdmKqRule mdo) {
        MdmKqRuleDto dto = null;
        if (Objects.isNull(mdo)) {
            return dto;
        }
        dto = new MdmKqRuleDto();
        BeanUtils.copyProperties(mdo, dto);
        return dto;
    }

}
