package com.sfa.system.controller.query;

import cn.hutool.core.util.ObjectUtil;
import com.sfa.common.core.domain.R;
import com.sfa.common.core.utils.StringUtils;
import com.sfa.common.core.utils.poi.ExcelUtil;
import com.sfa.common.core.web.controller.BaseController;
import com.sfa.common.core.web.domain.AjaxResult;
import com.sfa.common.core.web.domain.PageInfo;
import com.sfa.common.log.annotation.Log;
import com.sfa.common.log.enums.BusinessType;
import com.sfa.common.security.annotation.InnerAuth;
import com.sfa.common.security.annotation.RequiresPermissions;
import com.sfa.common.security.utils.SecurityUtils;
import com.sfa.system.api.domain.SysDept;
import com.sfa.system.api.domain.SysRole;
import com.sfa.system.api.domain.SysUser;
import com.sfa.system.api.model.LoginUser;
import com.sfa.system.api.pojo.request.UserVo;
import com.sfa.system.api.pojo.response.UserDto;
import com.sfa.system.domain.dao.ISysPostDao;
import com.sfa.system.domain.dao.ISysUserDao;
import com.sfa.system.domain.dao.ISysConfigDao;
import com.sfa.system.domain.dao.ISysDeptDao;
import com.sfa.system.domain.dao.ISysPermissionDao;
import com.sfa.system.domain.dao.ISysRoleDao;
import com.sfa.system.pojo.response.SysUserProDto;
import com.sfa.system.service.query.IUserQueryService;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用户信息
 *
 * @author ruoyi
 */
@RestController
@RequestMapping("/user")
public class SysUserQueryController extends BaseController {
    @Autowired
    private ISysUserDao userDao;

    @Autowired
    private IUserQueryService userQueryService;

    @Autowired
    private ISysRoleDao sysRoleDao;

    @Autowired
    private ISysDeptDao sysDeptDao;

    @Autowired
    private ISysPostDao sysPostDao;

    @Autowired
    private ISysPermissionDao sysPermissionDao;

    @Autowired
    private ISysConfigDao sysConfigDao;

    /**
     * 获取用户列表
     */
    @RequiresPermissions("system:user:list")
    @GetMapping("/list")
    public PageInfo list(UserVo userVo) {
        return userQueryService.selectUserPage(userVo);
    }

    @Log(title = "用户管理", businessType = BusinessType.EXPORT)
    @RequiresPermissions("system:user:export")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysUser user) {
        List<SysUser> list = userDao.selectUserList(user);
        ExcelUtil<SysUser> util = new ExcelUtil<SysUser>(SysUser.class);
        util.exportExcel(response, list, "用户数据");
    }

    @Log(title = "用户管理", businessType = BusinessType.IMPORT)
    @RequiresPermissions("system:user:import")
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception {
        ExcelUtil<SysUser> util = new ExcelUtil<SysUser>(SysUser.class);
        List<SysUser> userList = util.importExcel(file.getInputStream());
        String operName = SecurityUtils.getUsername();
        String message = userDao.importUser(userList, updateSupport, operName);
        return success(message);
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response) throws IOException {
        ExcelUtil<SysUser> util = new ExcelUtil<SysUser>(SysUser.class);
        util.importTemplateExcel(response, "用户数据");
    }

    /**
     * 获取当前用户信息
     */
    @InnerAuth
    @GetMapping("/info/{username}")
    public R<LoginUser> info(@PathVariable("username") String username) {
        SysUser sysUser = userDao.selectUserByUserName(username);
        if (StringUtils.isNull(sysUser)) {
            return R.fail("用户名或密码错误");
        }
        // 角色集合
        Set<String> roles = sysPermissionDao.getRolePermission(sysUser);
        // 权限集合
        Set<String> permissions = sysPermissionDao.getMenuPermission(sysUser);
        LoginUser sysUserVo = new LoginUser();
        sysUserVo.setSysUser(sysUser);
        sysUserVo.setRoles(roles);
        sysUserVo.setPermissions(permissions);
        return R.ok(sysUserVo);
    }

    @InnerAuth
    @GetMapping("/userInfoById")
    public R<SysUser> userInfo(Long userId) {
        SysUser user =new SysUser(userId);
        List<SysUser> list = userDao.selectUserList(user);
        if(ObjectUtil.isEmpty(list)){
            return R.fail("用户名或密码错误");
        }
        SysUser sysUser = list.get(0);
        return R.ok(sysUser);

    }

    /**
     * 注册用户信息
     */
    @InnerAuth
    @PostMapping("/register")
    public R<Boolean> register(@RequestBody SysUser sysUser) {
        String username = sysUser.getUserName();
        if (!("true".equals(sysConfigDao.selectConfigByKey("sys.account.registerUser")))) {
            return R.fail("当前系统没有开启注册功能！");
        }
        if (!userDao.checkUserNameUnique(sysUser)) {
            return R.fail("保存用户'" + username + "'失败，注册账号已存在");
        }
        return R.ok(userDao.registerUser(sysUser));
    }

    /**
     * 记录用户登录IP地址和登录时间
     */
    @InnerAuth
    @PutMapping("/recordlogin")
    public R<Boolean> recordlogin(@RequestBody SysUser sysUser) {
        return R.ok(userDao.updateUserProfile(sysUser));
    }

    /**
     * 登录后获取用户信息
     *
     * @return 用户信息
     */
    @GetMapping("/getInfo")
    public AjaxResult getInfo() {
        SysUser user = userDao.selectUserById(SecurityUtils.getUserId());
        UserDto userDto = new UserDto();
        BeanUtils.copyProperties(user, userDto);
        SysUserProDto sysUserProDto = userQueryService.selectUserPromotion(userDto);
        userDto.setIsOpenPromotion(sysUserProDto != null);
        if (userDto.getIsOpenPromotion()) {
            userDto.setPrivilegeId(sysUserProDto.getPrivilegeId());
            userDto.setProStatus(sysUserProDto.getStatus());
        }
        // 角色集合
        Set<String> roles = sysPermissionDao.getRolePermission(user);
        // 权限集合
        Set<String> permissions = sysPermissionDao.getMenuPermission(user);
        AjaxResult ajax = AjaxResult.success();
        ajax.put("user", userDto);
        ajax.put("roles", roles);
        ajax.put("permissions", permissions);
        return ajax;
    }

    /**
     * 管理后台-用户列表-查看用户详情
     * 根据用户编号获取详细信息
     */
    @RequiresPermissions("system:user:query")
    @GetMapping(value = {"/", "/{userId}"})
    public AjaxResult getInfo(@PathVariable(value = "userId", required = false) Long userId) {
        userDao.checkUserDataScope(userId);
        AjaxResult ajax = AjaxResult.success();
        List<SysRole> roles = sysRoleDao.selectRoleAll();
        ajax.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        ajax.put("posts", sysPostDao.selectPostAll());
        if (StringUtils.isNotNull(userId)) {
            SysUser sysUser = userDao.selectUserById(userId);
            UserDto userDto = new UserDto();
            BeanUtils.copyProperties(sysUser, userDto);
            SysUserProDto sysUserProDto = userQueryService.selectUserPromotion(userDto);
            userDto.setIsOpenPromotion(sysUserProDto != null);
            if (userDto.getIsOpenPromotion()) {
                userDto.setPrivilegeId(sysUserProDto.getPrivilegeId());
                userDto.setProStatus(sysUserProDto.getStatus());
            }

            ajax.put(AjaxResult.DATA_TAG, userDto);
//            ajax.put("postIds", sysPostDao.selectPostListByUserId(userId));
//            ajax.put("roleIds", sysUser.getRoles().stream().map(SysRole::getRoleId).collect(Collectors.toList()));
        }
        return ajax;
    }

    /**
     * 新增用户
     @RequiresPermissions("system:user:add")
     @Log(title = "用户管理", businessType = BusinessType.INSERT)
     @PostMapping public AjaxResult add(@Validated @RequestBody UserVo userVo)
     {
     sysDeptDao.checkDeptDataScope(userVo.getDeptId());
     sysRoleDao.checkRoleDataScope(userVo.getRoleIds());
     if (!userDao.checkUserNameUnique(user))
     {
     return error("新增用户'" + user.getUserName() + "'失败，登录账号已存在");
     }
     else if (StringUtils.isNotEmpty(user.getPhonenumber()) && !userDao.checkPhoneUnique(user))
     {
     return error("新增用户'" + user.getUserName() + "'失败，手机号码已存在");
     }
     else if (StringUtils.isNotEmpty(user.getEmail()) && !userDao.checkEmailUnique(user))
     {
     return error("新增用户'" + user.getUserName() + "'失败，邮箱账号已存在");
     }
     userVo.setCreateBy(SecurityUtils.getUsername());
     userVo.setPassword(SecurityUtils.encryptPassword(userVo.getPassword()));
     return toAjax(userDao.insertUser(userVo));
     }
     */


    /**
     * 修改用户

     @RequiresPermissions("system:user:edit")
     @Log(title = "用户管理", businessType = BusinessType.UPDATE)
     @PutMapping public AjaxResult edit(@Validated @RequestBody SysUser user)
     {
     userDao.checkUserAllowed(user);
     userDao.checkUserDataScope(user.getUserId());
     sysDeptDao.checkDeptDataScope(user.getDeptId());
     sysRoleDao.checkRoleDataScope(user.getRoleIds());
     if (!userDao.checkUserNameUnique(user))
     {
     return error("修改用户'" + user.getUserName() + "'失败，登录账号已存在");
     }
     else if (StringUtils.isNotEmpty(user.getPhonenumber()) && !userDao.checkPhoneUnique(user))
     {
     return error("修改用户'" + user.getUserName() + "'失败，手机号码已存在");
     }
     else if (StringUtils.isNotEmpty(user.getEmail()) && !userDao.checkEmailUnique(user))
     {
     return error("修改用户'" + user.getUserName() + "'失败，邮箱账号已存在");
     }
     user.setUpdateBy(SecurityUtils.getUsername());
     return toAjax(userDao.updateUser(user));
     }
     */


    /**
     * 删除用户
     */
    @RequiresPermissions("system:user:remove")
    @Log(title = "用户管理", businessType = BusinessType.DELETE)
    @DeleteMapping("/{userIds}")
    public AjaxResult remove(@PathVariable Long[] userIds) {
        if (ArrayUtils.contains(userIds, SecurityUtils.getUserId())) {
            return error("当前用户不能删除");
        }
        return toAjax(userDao.deleteUserByIds(userIds));
    }

    /**
     * 重置密码
     */
    @RequiresPermissions("system:user:edit")
    @Log(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/resetPwd")
    public AjaxResult resetPwd(@RequestBody SysUser user) {
        userDao.checkUserAllowed(user.getUserId());
        userDao.checkUserDataScope(user.getUserId());
        user.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
        user.setUpdateBy(SecurityUtils.getUsername());
        return toAjax(userDao.resetPwd(user));
    }

    /**
     * 状态修改
     */
    @RequiresPermissions("system:user:edit")
    @Log(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/changeStatus")
    public AjaxResult changeStatus(@RequestBody SysUser user) {
        userDao.checkUserAllowed(user.getUserId());
        userDao.checkUserDataScope(user.getUserId());
        user.setUpdateBy(SecurityUtils.getUsername());
        return toAjax(userDao.updateUserStatus(user));
    }

    /**
     * 根据用户编号获取授权角色
     */
    @RequiresPermissions("system:user:query")
    @GetMapping("/authRole/{userId}")
    public AjaxResult authRole(@PathVariable("userId") Long userId) {
        AjaxResult ajax = AjaxResult.success();
        SysUser user = userDao.selectUserById(userId);
        List<SysRole> roles = sysRoleDao.selectRolesByUserId(userId);
        ajax.put("user", user);
        ajax.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        return ajax;
    }

    /**
     * 用户授权角色
     */
    @RequiresPermissions("system:user:edit")
    @Log(title = "用户管理", businessType = BusinessType.GRANT)
    @PutMapping("/authRole")
    public AjaxResult insertAuthRole(Long userId, Long[] roleIds) {
        userDao.checkUserDataScope(userId);
        sysRoleDao.checkRoleDataScope(roleIds);
        userDao.insertUserAuth(userId, roleIds);
        return success();
    }

    /**
     * 获取部门树列表
     */
    @RequiresPermissions("system:user:list")
    @GetMapping("/deptTree")
    public AjaxResult deptTree(SysDept dept) {
        return success(sysDeptDao.selectDeptTreeList(dept));
    }
}
