package com.sfa.operation.util.excel;

import cn.hutool.core.bean.BeanUtil;
import com.sfa.operation.config.ExportColumnConfig;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-05 16:56
 * @Description: AP导出excel工具类
 */
@Slf4j
public class ExcelUtils {

    private static final Logger logger = LoggerFactory.getLogger(ExcelUtils.class);
    private static final ConcurrentHashMap<String, CellStyle> CACHE_CELL_STYLE = new ConcurrentHashMap<>();

    /**
     * 将Excel字节数组导出为文件（写入Http响应流）
     *
     * @param excelBytes     生成的Excel字节数组
     * @param response       Http响应对象
     * @param fileNamePrefix 导出文件名前缀（如"常规陈列"）
     * @throws Exception 导出异常
     */
    public static void exportExcelBytesToResponse(byte[] excelBytes, HttpServletResponse response, String fileNamePrefix) throws Exception {
        if (excelBytes == null || excelBytes.length == 0) {
            throw new Exception("导出Excel失败：生成的Excel字节数组为空");
        }
        if (StringUtils.isBlank(fileNamePrefix)) {
            fileNamePrefix = "导出文件";
        }

        // 1. 设置响应头（解决中文乱码、指定文件类型）
        String fileName = String.format("%s_%s.xlsx", fileNamePrefix, new SimpleDateFormat("yyyyMMdd").format(new Date()));
        // URLEncoder编码兼容所有浏览器
        String encodedFileName = URLEncoder.encode(fileName, String.valueOf(StandardCharsets.UTF_8));

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setHeader("Content-Disposition", "attachment;filename=" + encodedFileName);
        // 禁止缓存
        response.setHeader("Pragma", "no-cache");
        response.setHeader("Cache-Control", "no-store, no-cache, must-revalidate");
        response.setDateHeader("Expires", 0);
        // 设置文件大小（可选，提升下载体验）
        response.setContentLength(excelBytes.length);

        // 2. 将字节数组写入响应流
        try (OutputStream outputStream = response.getOutputStream()) {
            outputStream.write(excelBytes);
            outputStream.flush();
        } catch (Exception e) {
            logger.error("写入Excel响应流失败：{}", e.getMessage(), e);
            throw new Exception("导出Excel失败：" + e.getMessage());
        }
    }

    /**
     * 生成Excel文件
     *
     * @param exportColumnConfigList 表头配置
     * @param dataList               数据列表
     * @param sheetName              sheetName
     * @return byte[] excel 数组
     * @throws IOException
     */
    public static byte[] generateExcelBytes(List<ExportColumnConfig> exportColumnConfigList, List<?> dataList, String sheetName) throws IOException {
        //数据校验
        if (CollectionUtils.isEmpty(exportColumnConfigList)) {
            throw new IOException("导出Excel失败：没有配置导出列");
        }
        if (CollectionUtils.isEmpty(dataList)) {
            throw new IOException("导出Excel失败：没有获取到相关数据，请检查条件后重试");
        }
        if (sheetName == null || sheetName.trim().isEmpty()) {
            throw new IOException("导出Excel失败：没有配置sheetName");
        }

        // 根据sheetName创建工作表
        try ( // 创建工作簿
              Workbook workbook = createExcelWorkbook(dataList);
              // 创建文件字节数组
              ByteArrayOutputStream outputStream = new ByteArrayOutputStream();) {
            try {
                Sheet sheet = workbook.createSheet(sheetName);
                // 创建表头
                createHeader(sheet, exportColumnConfigList);
                // 创建数据行
                createDataRows(sheet, dataList, exportColumnConfigList);

                // 处理需要验证的列
                for (int i = 0; i < exportColumnConfigList.size(); i++) {
                    ExportColumnConfig config = exportColumnConfigList.get(i);
                    // 给列配置只读验证
                    if (isColumnProtected(config)) {
                        // 为该列所有数据行添加只读验证
                        // 从第1行开始（跳过表头）
                        addReadOnlyValidation(sheet, i, 1, dataList.size());
                    }
                }
                workbook.write(outputStream);
                return outputStream.toByteArray();
            } finally {
                //如果创建的是 sxssf 清理临时文件
                if (workbook instanceof SXSSFWorkbook) {
                    ((SXSSFWorkbook) workbook).dispose();
                }
                workbook.close();
            }

        } catch (IOException e) {
            logger.error("生成Excel文件失败：{}", e.getMessage(), e);
            throw new IOException("生成Excel文件失败：" + e.getMessage());
        }
    }


    /**
     * 创建工作簿
     *
     * @param dataList 数据列表
     * @return 工作簿
     */
    private static Workbook createExcelWorkbook(List<?> dataList) {
        if (dataList.size() > 1000) {
            SXSSFWorkbook sxssfWorkbook = new SXSSFWorkbook(100);
            sxssfWorkbook.setCompressTempFiles(true);
            return sxssfWorkbook;
        } else {
            return new XSSFWorkbook();
        }
    }

    /**
     * 创建表头
     *
     * @param sheet                  工作表
     * @param exportColumnConfigList 表头配置
     */
    public static void createHeader(Sheet sheet, List<ExportColumnConfig> exportColumnConfigList) {
        Workbook workbook = sheet.getWorkbook();
        Row headerRow = sheet.createRow(0);
        // 调整表头行高（适配换行）
        headerRow.setHeightInPoints(40);

        for (int i = 0; i < exportColumnConfigList.size(); i++) {
            ExportColumnConfig config = exportColumnConfigList.get(i);
            Cell cell = headerRow.createCell(i);
            String originalHeader = config.getHeaderName() == null ? "" : config.getHeaderName();
            // 处理表头文本：拆分括号内容+换行
            String processedHeader = processHeaderText(originalHeader);
            cell.setCellValue(processedHeader);
            // 使用 HEADER_STYLE 作为表头的基础样式
            CellStyle headerStyle = cacheStyle(workbook, ExcelStyleUtils.ExcelStyle.HEADER_STYLE);
            cell.setCellStyle(headerStyle);
            cell.getCellStyle().setWrapText(true);
            // 表头强制锁定（不可修改）
            cell.getCellStyle().setLocked(true);
            // 设置初始列宽
            String headerText = config.getHeaderName();
            if (headerText != null) {
                int maxWidth = calculateTextWidth(processedHeader);
                // 至少20个字符宽度
                sheet.setColumnWidth(i, Math.max(maxWidth, 20 * 256));
            }
        }
        // 为表头行添加只读验证（防止用户修改表头）
        for (int i = 0; i < exportColumnConfigList.size(); i++) {
            ExportColumnConfig exportColumnConfig = exportColumnConfigList.get(i);
            if (exportColumnConfig.getValidationPromptTitle() != null && exportColumnConfig.getValidationPromptMsg() != null){
                addHeaderPromptValidation(sheet, i, exportColumnConfig);
            }
                // 只保护第0行（表头行）
                addReadOnlyValidation(sheet, i, 0, 0);

        }
    }


    /**
     * 创建数据行
     *
     * @param sheet                  工作表
     * @param dataList               数据列表
     * @param exportColumnConfigList 表头配置
     */

    public static void createDataRows(Sheet sheet, List<?> dataList, List<ExportColumnConfig> exportColumnConfigList) {
        Workbook workbook = sheet.getWorkbook();
        for (int i = 0; i < dataList.size(); i++) {
            // 创建数据行
            Row dataRow = sheet.createRow(i + 1);
            // 获取数据
            Object data = dataList.get(i);
            // 渲染数据列
            for (int j = 0; j < exportColumnConfigList.size(); j++) {
                // 获取数据列的配置
                ExportColumnConfig exportColumnConfig = exportColumnConfigList.get(j);
                Cell cell = dataRow.createCell(j);

                // 处理条件样式
                CellStyle style;
                if (exportColumnConfig.isConditionalStyling() && j > 0) {
                    // 检查前一列是否有值
                    Cell previousCell = dataRow.getCell(j - 1);
                    boolean hasPreviousValue = false;

                    if (previousCell != null) {
                        switch (previousCell.getCellType()) {
                            case STRING:
                                hasPreviousValue = previousCell.getStringCellValue() != null &&
                                        !previousCell.getStringCellValue().trim().isEmpty();
                                break;
                            case NUMERIC:
                                hasPreviousValue = true;
                                break;
                            case BOOLEAN:
                                hasPreviousValue = true;
                                break;
                            default:
                        }
                    }
                    // 根据前一列是否有值来设置样式
                    if (hasPreviousValue) {
                        style = cacheStyle(workbook, ExcelStyleUtils.ExcelStyle.LIGHT_BLUE_BG);
                    } else {
                        style = cacheStyle(workbook, ExcelStyleUtils.ExcelStyle.DEFAULT_STYLE);
                    }
                } else {
                    // 使用配置的默认样式
                    style = cacheStyle(workbook, exportColumnConfig.getStyle());
                }

                cell.setCellStyle(style);
                Object value = getFieldValueByOgnl(exportColumnConfig.getFieldName(), data);
                // 设置单元格值并应用格式
                setCellValueWithFormat(cell, value, exportColumnConfig.getFormat());
            }
        }
    }


    /**
     * 添加表头提示
     *
     * @param sheet       工作表
     * @param columnIndex 列索引
     * @param config      表头配置
     */
    private static void addHeaderPromptValidation(Sheet sheet, int columnIndex, ExportColumnConfig config) {
        DataValidationHelper helper = sheet.getDataValidationHelper();
        // 创建总是为真的约束
        DataValidationConstraint constraint = helper.createCustomConstraint("FALSE");

        // 仅对表头行（第0行）应用
        CellRangeAddressList addressList = new CellRangeAddressList(0, 0, columnIndex, columnIndex);

        DataValidation validation = helper.createValidation(constraint, addressList);

        // 设置提示信息
        validation.setShowPromptBox(true);
        validation.createPromptBox(config.getValidationPromptTitle(), config.getValidationPromptMsg());
        validation.setShowErrorBox(true);
        validation.createErrorBox("禁止修改", "单元格内容禁止修改！");
        validation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        validation.setEmptyCellAllowed(false);
        validation.setSuppressDropDownArrow(false);

        sheet.addValidationData(validation);
    }

    /**
     * 根据字段名获取字段值
     *
     * @param fieldName 字段名
     * @param data      数据对象
     * @return 字段值
     */
    private static Object getFieldValueByOgnl(String fieldName, Object data) {
        // 避免空指针访问
        if (data == null || fieldName == null || fieldName.trim().isEmpty()) {
            return "";
        }
        try {
            // 使用 BeanUtils 获取属性值
            return BeanUtil.getProperty(data, fieldName);
        } catch (Exception e) {
            logger.error("获取字段值失败：{}", fieldName, e);
            // 空值兜底，避免崩溃
            return "";
        }
    }

    /**
     * 设置单元格值并应用格式（修复：日期/数字格式处理）
     *
     * @param cell   单元格
     * @param value  数据值
     * @param format 格式（如yyyy-MM-dd、0.00）
     */
    private static void setCellValueWithFormat(Cell cell, Object value, String format) {
        if (value == null) {
            cell.setCellValue("");
            return;
        }

        // 特殊处理：如果格式为 "@" 或者字段名包含某些关键词，则强制作为文本处理
        if ("@".equals(format)) {
            cell.setCellValue(value.toString());
            return;
        }

        // 1. 日期类型
        if (value instanceof Date) {
            if (format != null && !format.isEmpty()) {
                SimpleDateFormat sdf = new SimpleDateFormat(format);
                cell.setCellValue(sdf.format((Date) value));
            } else {
                cell.setCellValue((Date) value);
            }
            return;
        }

        // 2. 数值类型（整数/小数）：设置为数值单元格，保证数值校验生效
        try {
            double numericValue = Double.parseDouble(value.toString());
            cell.setCellValue(numericValue);
            // 数值格式（如保留2位小数）
            if (format != null && !format.isEmpty()) {
                DataFormat formatInstance = cell.getSheet().getWorkbook().createDataFormat();
                CellStyle numericStyle = cell.getSheet().getWorkbook().createCellStyle();
                numericStyle.setDataFormat(formatInstance.getFormat(format));
                cell.setCellStyle(numericStyle);
            }
            return;
        } catch (NumberFormatException e) {
            // 非数值类型，走字符串逻辑
        }

        // 3. 字符串类型
        cell.setCellValue(value.toString());
    }

    /**
     * 计算文本宽度（支持换行）
     *
     * @param text 文本内容
     * @return 宽度值
     */
    private static int calculateTextWidth(String text) {
        if (text == null || text.isEmpty()) {
            return 10 * 256;
        }

        String[] lines = text.split("\n");
        int maxLineLength = 0;
        for (String line : lines) {
            maxLineLength = Math.max(maxLineLength, line.length());
        }

        // 每个字符大约占用256单位宽度，增加一些缓冲
        return maxLineLength * 256 + 1000;
    }


    /**
     * 缓存样式避免内存溢出
     *
     * @param workbook 工作簿
     * @param style    目标样式
     * @return 缓存的样式
     */
    private static CellStyle cacheStyle(Workbook workbook, ExcelStyleUtils.ExcelStyle style) {
        ExcelStyleUtils.ExcelStyle targetStyle = style == null ? ExcelStyleUtils.ExcelStyle.DEFAULT_STYLE : style;
        // 使用更稳定的缓存键
        String styleKey = System.identityHashCode(workbook) + "_" + targetStyle.name();
        if (!CACHE_CELL_STYLE.containsKey(styleKey)) {
            CellStyle cellStyle = ExcelStyleUtils.getStyle(workbook, targetStyle);
            CACHE_CELL_STYLE.put(styleKey, cellStyle);
            return cellStyle;
        }
        return CACHE_CELL_STYLE.get(styleKey);
    }

    /**
     * 处理表头文本：拆分括号内容+换行
     *
     * @param originalText 原始文本
     * @return 处理后的文本
     */
    private static String processHeaderText(String originalText) {
        if (originalText == null || originalText.isEmpty()) {
            return "";
        }
        // 替换中文括号及其内容为换行形式
        String result ;
        if (originalText.contains("（") && originalText.contains("）")) {
            // 中文括号替换为换行
            result = originalText.replaceAll("（(.*?)）", "\n$1");
        } else if (originalText.contains("(") && originalText.contains(")")){
            // 英文括号替换为换行
            result = originalText.replaceAll("\\((.*?)\\)", "\n$1");
        }else if (originalText.contains("-")){
            // 横线替换为换行（内容）
            result = originalText.replaceAll("-(.+)", "\n（$1）");
        }else if (originalText.contains("_")){
            result = originalText.replaceAll("_(.+)", "\n（$1）");
        }else {
            result = originalText;
        }

        return result;
    }

    /**
     * 为特定单元格区域添加只读数据验证
     *
     * @param sheet       工作表
     * @param columnIndex 列索引
     * @param startRow    起始行
     * @param endRow      结束行
     */
    private static void addReadOnlyValidation(Sheet sheet, int columnIndex, int startRow, int endRow) {
        DataValidationHelper helper = sheet.getDataValidationHelper();

        // 创建自定义公式验证，始终返回false（禁止输入）
        DataValidationConstraint constraint = helper.createCustomConstraint("FALSE");

        CellRangeAddressList addressList = new CellRangeAddressList(startRow, endRow, columnIndex, columnIndex);
        DataValidation validation = helper.createValidation(constraint, addressList);

        validation.setShowErrorBox(true);
        validation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        validation.createErrorBox("禁止修改", "此单元格内容不可修改");
        validation.setEmptyCellAllowed(false);
        validation.setSuppressDropDownArrow(false);

        sheet.addValidationData(validation);
    }

    /**
     * 判断列是否需要保护
     *
     * @param config 列配置
     * @return 是否需要保护
     */
    private static boolean isColumnProtected(ExportColumnConfig config) {
        return config.getStyle() == ExcelStyleUtils.ExcelStyle.UNMODIFIABLE || config.getStyle() == ExcelStyleUtils.ExcelStyle.CHANGE_TEXT_STYLE;
    }

}

