package com.sfa.operation.service.sales.export.impl;

import com.alibaba.fastjson2.JSONObject;
import com.sfa.common.core.domain.R;
import com.sfa.operation.factory.ApImportExcelStrategyFactory;
import com.sfa.operation.pojo.sales.request.ImportApExcelRequest;
import com.sfa.operation.service.sales.export.IImportExcelService;
import com.sfa.operation.strategy.IImportApExcelStrategy;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-12 13:11
 * @Description: 导入excel服务实现类
 */
@Slf4j
@Service
public class ImportExcelServiceImpl implements IImportExcelService {

    @Autowired
    private ApImportExcelStrategyFactory apImportExcelStrategyFactory;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    public static final String REDIS_KEY_PREFIX = "import_excel_ap:";
    /**
     * 店内执行上上报 -导入方法
     *
     * @param request 导入excel请求参数
     * @return 导入结果
     */
    @Override
    public R importApExcel(ImportApExcelRequest request) {
        // 参数校验
        if (request == null || request.getImportApFilePath() == null || request.getImportApFilePath().trim().isEmpty()){
            return R.fail("导入文件路径不能为空!");
        }
        if (request.getImportApType() == null || request.getImportApType().trim().isEmpty()){
            return R.fail("导入类型不能为空!");
        }

        // 获取策略
        IImportApExcelStrategy strategy = apImportExcelStrategyFactory.getStrategy(request.getImportApType());
        if (strategy == null){
            return R.fail("未找到对应的导入策略!");
        }

        // 执行导入的数据验证（根据不同的策略独自设计验证）
        Map<String, Object> result = strategy.execute(request.getImportApFilePath());
        Integer failCount = (Integer) result.get("failCount");

        //failCount>0 时 返回错误信息

        if (failCount > 0 ) {
            log.error("导入失败，失败条数：{}",failCount);
            return R.fail(0, result);
        } else {
            String uuid = (String) result.getOrDefault("uuid", "");
            String redisKey = REDIS_KEY_PREFIX + uuid;

            //将数据保存躁redis中
            stringRedisTemplate.opsForValue().set(redisKey, JSONObject.toJSONString(result), 30, TimeUnit.MINUTES);
            log.info("数据保存至redis中，redisKey={},result={}", redisKey, JSONObject.toJSONString(result));
        }

        return R.ok(result);
    }

    /**
     * 前端点击确认后 更新数据
     * @param request 导入excel请求参数
     * @return 导入结果
     */
    @Override
    public R updateApEntity(ImportApExcelRequest request) {
         if (request.getImportApType() == null || request.getImportApType().trim().isEmpty()){
            return R.fail("导入类型不能为空!");
        }
         if (request.getUuid() == null || request.getUuid().trim().isEmpty()){
            return R.fail("导入数据标识不能为空!");
        }

        // 获取策略
        IImportApExcelStrategy strategy = apImportExcelStrategyFactory.getStrategy(request.getImportApType());
        if (strategy == null){
            return R.fail("未找到对应的导入策略!");
        }
        //从redis获取数据
        String redisKey =  REDIS_KEY_PREFIX+ request.getUuid();
        String redisValue = stringRedisTemplate.opsForValue().get(redisKey);

        //解析jsonToDtoList
        List list = strategy.getTransactionJsonToObject(redisValue);

        //批量更新
        String result = strategy.updateDisplay(list);

        if ("更新失败".equals(result)) {
            return R.fail(result);
        }

        return R.ok(result);
    }
}
