package com.sfa.operation.util.excel;

import com.sfa.operation.config.ExportColumnConfig;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddressList;

import java.util.List;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-08 12:08
 * @Description: 配置导出的EXCEL样式
 */

public class ExcelStyleUtils {
    public enum ExcelStyle {
        // 不可修改
        UNMODIFIABLE,
        // 浅蓝色
        LIGHT_BLUE_BG,
        // 浅灰色
        LIGHT_GREY_BG,
        //默认
        DEFAULT_STYLE,
        RED_STYLE,
        HEADER_STYLE
    }

    /**
     * 获取表格样式
     *
     * @param workbook 工作表格
     * @param style    样式
     * @return 已经设置好的样式
     */
    /**
     * 获取表格样式
     *
     * @param workbook 工作表格
     * @param style    样式
     * @return 已经设置好的样式
     */
    public static CellStyle getStyle(Workbook workbook, ExcelStyle style) {
        CellStyle cellStyle = workbook.createCellStyle();
        Font font = workbook.createFont();

        // 设置通用样式属性
        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        cellStyle.setAlignment(HorizontalAlignment.CENTER);
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyle.setBorderBottom(BorderStyle.THIN);
        cellStyle.setBorderTop(BorderStyle.THIN);
        cellStyle.setBorderLeft(BorderStyle.THIN);
        cellStyle.setBorderRight(BorderStyle.THIN);

        switch (style) {
            // 不可修改
            case UNMODIFIABLE:
                cellStyle.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.getIndex());
                cellStyle.setLocked(true);
                font.setBold(true);
                cellStyle.setFont(font);
                break;
            // 浅蓝色
            case LIGHT_BLUE_BG:
                cellStyle.setFillForegroundColor(IndexedColors.PALE_BLUE.getIndex());
                cellStyle.setLocked(false);
                break;
            // 浅灰色
            case LIGHT_GREY_BG:
                cellStyle.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.getIndex());
                cellStyle.setLocked(true);
                break;
            // 红色样式
            case RED_STYLE:
                font.setColor(IndexedColors.RED.getIndex());
                font.setFontName("微软雅黑");
                font.setFontHeightInPoints((short) 11);
                cellStyle.setFont(font);
                cellStyle.setFillForegroundColor(IndexedColors.WHITE.getIndex());
                cellStyle.setLocked(false);
                break;
            // 表头样式
            case HEADER_STYLE:
                font.setBold(true);
                font.setFontName("微软雅黑");
                font.setFontHeightInPoints((short) 12);
                cellStyle.setFont(font);
                cellStyle.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.getIndex());
                cellStyle.setLocked(false);
                break;
            // 默认
            case DEFAULT_STYLE:
            default:
                cellStyle.setFillForegroundColor(IndexedColors.WHITE.getIndex());
                cellStyle.setLocked(false);
                break;
        }
        return cellStyle;
    }

    /**
     * 匹配不可修改样式的列
     *
     * @param headerName             表头名称
     * @param unmodifiableHeaderList 不可修改的列名
     * @return 匹配到的样式
     */
    public static ExcelStyle matchUnmodifiedStyle(String headerName, List<String> unmodifiableHeaderList) {
        return isHeaderInList(headerName, unmodifiableHeaderList) ? ExcelStyle.UNMODIFIABLE : ExcelStyle.DEFAULT_STYLE;
    }

    /**
     * 匹配浅蓝色样式的列
     *
     * @param headerName          表头名称
     * @param lightBlueHeaderList 浅蓝色列名
     * @return 匹配到的样式
     */
    public static ExcelStyle matchLightBlueStyle(String headerName, List<String> lightBlueHeaderList) {
        return isHeaderInList(headerName, lightBlueHeaderList) ? ExcelStyle.LIGHT_BLUE_BG : ExcelStyle.DEFAULT_STYLE;
    }

    /**
     * 匹配浅灰色样式的列
     *
     * @param headerName          表头名称
     * @param lightGreyHeaderList 浅灰色列名
     * @return 匹配到的样式
     */
    public static ExcelStyle matchLightGreyStyle(String headerName, List<String> lightGreyHeaderList) {

        return isHeaderInList(headerName, lightGreyHeaderList) ? ExcelStyle.LIGHT_GREY_BG : ExcelStyle.DEFAULT_STYLE;
    }

    /**
     * 判断表头是否在列表中
     *
     * @param headerName 表头名称
     * @param headerList 表头列表
     * @return 是否在列表中
     */
    public static boolean isHeaderInList(String headerName, List<String> headerList) {
        if (headerList == null || CollectionUtils.isEmpty(headerList)) {
            return false;
        }
        if (headerName == null || headerName.trim().isEmpty()) {
            return false;
        }
        if (!headerList.contains(headerName)) {
            return false;
        }
        return headerList.contains(headerName);
    }

    /**
     * 添加列数据验证
     *
     * @param sheet      工作表
     * @param columnIndex 列索引
     * @param config     列配置
     */
    public static void addColumnDataValidation(Sheet sheet, int columnIndex, ExportColumnConfig config) {
        // 空值校验：Sheet或配置为空，直接返回（避免空指针）
        if (sheet == null || config == null) {
            return;
        }
        // 获取核心对象：工作簿、数据验证助手
        Workbook workbook = sheet.getWorkbook();
        DataValidationHelper helper = sheet.getDataValidationHelper();
        // 声明约束对象（后续根据验证类型赋值）
        DataValidationConstraint constraint = null;

        // 数字验证分支
        if (config.isNumberValidation()) {
            // 获取数字验证的最值配置
            Double min = config.getValidationNumberMin();
            Double max = config.getValidationNumberMax();

            // 校验最值配置：至少配置一个才生效
            if (min == null && max == null) {
                return;
            }

            // 选择数字类型：整数/小数
            // DECIMAL=2（小数），INTEGER=1（整数）
            int numericType = config.isValidationNumberAllowDecimal()
                    ? DataValidationConstraint.ValidationType.DECIMAL
                    : DataValidationConstraint.ValidationType.INTEGER;

            // 转换最值为字符串（POI要求的参数格式）
            String minStr = (min == null) ? "" : min.toString();
            String maxStr = (max == null) ? "" : max.toString();

            // 根据最值配置，构建不同的数字约束
            if (min != null && max != null) {
                // 介于min和max之间（OperatorType.BETWEEN = 1）
                constraint = helper.createNumericConstraint(
                        numericType,
                        DataValidationConstraint.OperatorType.BETWEEN,
                        minStr,
                        maxStr
                );
            } else if (min != null) {
                // 大于等于min（OperatorType.GREATER_OR_EQUAL = 6）
                constraint = helper.createNumericConstraint(
                        numericType,
                        DataValidationConstraint.OperatorType.GREATER_OR_EQUAL,
                        minStr,
                        ""
                );
            } else {
                // 小于等于max（OperatorType.LESS_OR_EQUAL = 7）
                constraint = helper.createNumericConstraint(
                        numericType,
                        DataValidationConstraint.OperatorType.LESS_OR_EQUAL,
                        "",
                        maxStr
                );
            }
        }

        // 列表验证分支（原有逻辑）
        else if (config.getValidationValidOptions() != null && !config.getValidationValidOptions().isEmpty()) {
            // 获取列表合法值
            List<String> validOptions = config.getValidationValidOptions();
            // 构建列表约束（仅允许输入列表中的值）
            constraint = helper.createExplicitListConstraint(
                    validOptions.toArray(new String[0])
            );
        }

        // 无有效约束，直接返回
        if (constraint == null) {
            return;
        }

        // 配置验证生效的行范围（默认从第1行开始，跳过表头）
        CellRangeAddressList addressList = new CellRangeAddressList(
                config.getValidationStartRow(),
                config.getValidationEndRow(),
                columnIndex,
                columnIndex
        );

        // 创建数据验证规则（绑定约束+生效范围）
        DataValidation validation = helper.createValidation(constraint, addressList);

        // 配置错误提示（非法输入时弹窗）
        if (config.getValidationErrorTitle() != null && config.getValidationErrorMsg() != null) {
            validation.setShowErrorBox(true);
            validation.setErrorStyle(config.getValidationErrorStyle());
            // 设置错误弹窗的标题和内容
            validation.createErrorBox(config.getValidationErrorTitle(), config.getValidationErrorMsg());
        }

        // 配置输入提示（鼠标选中单元格时的引导提示）
        if (config.getValidationPromptTitle() != null && config.getValidationPromptMsg() != null) {
            validation.setShowPromptBox(true);
            // 设置输入提示的标题和内容
            validation.createPromptBox(config.getValidationPromptTitle(), config.getValidationPromptMsg());
        }

        // 将验证规则添加到Sheet，使其生效
        sheet.addValidationData(validation);
    }
}
