package com.sfa.operation.service.sales.export.impl;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.sfa.common.core.domain.R;
import com.sfa.operation.factory.ApImportExcelStrategyFactory;
import com.sfa.operation.pojo.sales.request.ImportApExcelRequest;
import com.sfa.operation.service.sales.export.IImportExcelService;
import com.sfa.operation.strategy.IImportApExcelStrategy;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-12 13:11
 * @Description: 导入excel服务实现类
 */
@Slf4j
@Service
public class ImportExcelServiceImpl implements IImportExcelService {

    @Autowired
    private ApImportExcelStrategyFactory apImportExcelStrategyFactory;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    public static final String REDIS_KEY_PREFIX = "import_excel_ap:";
    /**
     * 店内执行上上报 -导入方法
     *
     * @param request 导入excel请求参数
     * @return 导入结果
     */
    @Override
    public R importApExcel(ImportApExcelRequest request) {
        // 参数校验
        if (request == null || request.getImportApFilePath() == null || request.getImportApFilePath().trim().isEmpty()){
            return R.fail(910,"文件上传失败，请重新上传！");
        }
        if (request.getImportApType() == null || request.getImportApType().trim().isEmpty()){
            return R.fail(910,"文件上传失败，请重新上传！");
        }

        // 获取策略
        IImportApExcelStrategy strategy = apImportExcelStrategyFactory.getStrategy(request.getImportApType());
        if (strategy == null){
            log.error("未找到对应的导入策略!");
            return R.fail(910,"文件上传失败，请重新上传！");
        }

        // 执行导入的数据验证（根据不同的策略独自设计验证）
        Map<String, Object> result = strategy.execute(request.getImportApFilePath());
        Integer failCount = (Integer) result.get("failCount");

        //failCount>0 时 返回错误信息

        if (failCount > 0 ) {
            log.error("导入失败，失败条数：{}",failCount);
            return R.fail(0, result);
        } else {
            String uuid = (String) result.getOrDefault("uuid", "");
            String redisKey = REDIS_KEY_PREFIX + uuid;

            //将数据保存躁redis中
            stringRedisTemplate.opsForValue().set(redisKey, JSONObject.toJSONString(result), 30, TimeUnit.MINUTES);
            log.info("数据保存至redis中，redisKey={},result={}", redisKey, JSONObject.toJSONString(result));
        }

        return R.ok(result);
    }

    /**
     * 前端点击确认后 更新数据
     * @param request 导入excel请求参数
     * @return 导入结果
     */
    @Override
    public R updateApEntity(ImportApExcelRequest request) {
        if (request.getImportApType() == null || request.getImportApType().trim().isEmpty()) {
            return R.fail(910,"数据更新失败，请重新导入!");
        }
        if (request.getUuid() == null || request.getUuid().trim().isEmpty()) {
            return R.fail(910,"数据更新失败，请重新导入!");
        }

        // 获取策略
        IImportApExcelStrategy strategy = apImportExcelStrategyFactory.getStrategy(request.getImportApType());
        if (strategy == null) {
            return R.fail(910,"数据更新失败，请重新导入!");
        }

        // 从redis获取数据
        String redisKey = REDIS_KEY_PREFIX + request.getUuid();
        String redisDataJson = stringRedisTemplate.opsForValue().get(redisKey);
        log.info("从Redis获取数据，redisKey={}, redisDataJson={}", redisKey, redisDataJson);

        // 1校验Redis数据是否存在
        if (redisDataJson == null || redisDataJson.trim().isEmpty()) {
            log.error("Redis中无有效数据，key={}", redisKey);
            return R.fail(910,"数据更新失败，请重新导入!");
        }

        // 解析Redis数据为JSONObject，仅提取table数组（核心修改点，适配JDK8）
        List<?> tableList;
        try {
            // 先解析整个Redis数据为JSONObject
            JSONObject redisDataObj = JSONObject.parseObject(redisDataJson);
            // 提取table字段对应的JSON数组
            JSONArray tableArray = redisDataObj.getJSONArray("table");
            if (tableArray == null) {
                tableList = Collections.emptyList();
            } else {
                // 根据策略指定的DTO类型解析table数组
                Class<?> dtoClass = strategy.getTargetDtoClass();
                // 解析为策略对应的DTO类型列表（而非Object）
                tableList = tableArray.toList(dtoClass);
            }
        } catch (Exception e) {
            log.error("解析Redis中的table数组失败，redisDataJson={}", redisDataJson, e);
            return R.fail(910,"数据更新失败，请重新导入!");
        }

        // 校验table数组非空
        if (CollectionUtils.isEmpty(tableList)) {
            log.error("解析出的table数组为空，redisDataJson={}", redisDataJson);
            return R.fail("更新数据为空，请重新导入!");
        }

        for (Object o : tableList) {
            System.out.println( o.toString());
        }

        // 批量更新（传入table数组，适配不同策略处理）
        String updateResult = strategy.updateDisplay(tableList);


        log.info("批量更新结果：{}", updateResult);

        if ("更新失败".equals(updateResult)) {
            return R.fail(910,updateResult);
        }

        return R.ok(updateResult);
    }
}
