package com.sfa.operation.util.excel;

import cn.hutool.core.bean.BeanUtil;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.auth.sts.AssumeRoleRequest;
import com.aliyuncs.auth.sts.AssumeRoleResponse;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.sfa.operation.config.ExportColumnConfig;
import com.sfa.operation.config.OssConfigProperties;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.math.BigDecimal;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-05 16:56
 * @Description: AP导出excel工具类
 */
@Slf4j
@Component
public class ExcelUtils {
    //缓存样式
    private static final ConcurrentHashMap<String, CellStyle> CACHE_CELL_STYLE = new ConcurrentHashMap<>();
    //表头行(默认为0)
    private static final int HEADER_ROW = 0;

    private static OssConfigProperties staticOssConfig;

    //OSS配置
    @Autowired
    private OssConfigProperties ossConfig;

    @PostConstruct
    public void init() {
        staticOssConfig = this.ossConfig;
    }

    /* ============================================================导入部分=========================================================== */

    public static InputStream getLocalFileInputStream(String filePath) throws Exception {
        // 参数校验
        if (StringUtils.isBlank(filePath)) {
            throw new IllegalArgumentException("本地文件路径不能为空");
        }

        File localFile = new File(filePath);
        // 检查文件是否存在
        if (!localFile.exists()) {
            String errorMsg = String.format("本地文件不存在：%s", filePath);
            log.error(errorMsg);
            throw new Exception(errorMsg);
        }
        // 检查是否是文件（而非目录）
        if (!localFile.isFile()) {
            String errorMsg = String.format("路径不是有效文件：%s", filePath);
            log.error(errorMsg);
            throw new Exception(errorMsg);
        }
        // 检查文件可读权限
        if (!localFile.canRead()) {
            String errorMsg = String.format("本地文件不可读取（权限不足）：%s", filePath);
            log.error(errorMsg);
            throw new Exception(errorMsg);
        }

        try {
            // 创建文件输入流并返回
            FileInputStream fis = new FileInputStream(localFile);
            log.info("成功获取本地文件输入流：{}", filePath);
            return fis;
        } catch (IOException e) {
            String errorMsg = String.format("读取本地文件失败：%s，错误：%s", filePath, e.getMessage());
            log.error(errorMsg, e);
            throw new Exception(errorMsg, e);
        }
    }

    /**
     * 获取OSS文件输入流
     * @param ossFileUrl OSS文件URL
     * @return OSS文件输入流
     * @throws Exception 获取OSS文件输入流失败
     */
    public static InputStream getOssFileInputStream(String ossFileUrl) throws Exception {
        // 参数校验
        if (StringUtils.isBlank(ossFileUrl)) {
            throw new IllegalArgumentException("OSS文件URL不能为空");
        }
        OssConfigProperties.Oss oss = staticOssConfig.getOss();
        if (oss == null || StringUtils.isBlank(oss.getRegionId()) || StringUtils.isBlank(oss.getBucketName())) {
            throw new IllegalStateException("OSS基础配置未初始化（regionId/bucketName为空）");
        }

        // 解析OSS URL：提取文件路径（objectKey）
        String objectKey = parseObjectKeyFromUrl(ossFileUrl, oss.getWebJsLink());
        if (StringUtils.isBlank(objectKey)) {
            throw new Exception("解析OSS文件路径失败：" + ossFileUrl);
        }

        //获取STS临时凭证
        AssumeRoleResponse.Credentials stsCredentials = getStsCredentials();

        //用STS临时凭证创建OSS客户端
        OSS ossClient = null;
        try {
            // Endpoint
            ossClient = new OSSClientBuilder().build(
                    "https://oss-" + oss.getRegionId() + ".aliyuncs.com",
                    // STS临时AK
                    stsCredentials.getAccessKeyId(),
                    // STS临时SK
                    stsCredentials.getAccessKeySecret(),
                    // STS Token
                    stsCredentials.getSecurityToken()
            );

            // 校验文件是否存在
            if (!ossClient.doesObjectExist(oss.getBucketName(), objectKey)) {
                throw new Exception(String.format("OSS文件不存在：bucket=%s, path=%s", oss.getBucketName(), objectKey));
            }

            // 获取输入流(调用时注意关闭流)
            return ossClient.getObject(new GetObjectRequest(oss.getBucketName(), objectKey)).getObjectContent();
        } catch (Exception e) {
            log.error("获取OSS文件失败：url={}", ossFileUrl, e);
            throw new Exception("OSS文件读取失败：" + e.getMessage(), e);
        } finally {
            if (ossClient != null) {
                // 关闭客户端
                ossClient.shutdown();
            }
        }
    }



    /**
     * 导入数据
     * @param inputStream 导入文件的输入流
     * @param fileName 导入文件名
     * @param columnConfigs 导入列配置
     * @param errorMap 错误信息
     * @param dtoClass 目标DTO类型
     * @return 导入数据
     * @param <T> 目标DTO泛型
     */
public static <T> List<T> readApExcelWithColumnConfig(InputStream inputStream,
                                                      String fileName,
                                                      List<ExportColumnConfig> columnConfigs,
                                                      Map<String, List<T>> errorMap,
                                                      Class<T> dtoClass) {
    log.info("开始导入数据（复用列配置校验），目标DTO类型：{}", dtoClass.getSimpleName());

    // 初始化空错误列表（基础方法需要）
    List<String> baseErrorList = new ArrayList<>();
    // 调用基础导入方法，传入自定义行解析器
    List<T> dtoList = readApExcel(inputStream, fileName, baseErrorList, (row, rowNum) -> {
        // 初始化当前行数据和错误信息
        List<T> rowData = new ArrayList<>();
        StringBuilder rowErrorMsg = new StringBuilder();
        T dto = null;

        // 反射创建DTO实例
        try {
            dto = dtoClass.getDeclaredConstructor().newInstance();
        } catch (Exception e) {
            log.error("创建DTO实例失败，类型：{}，错误：{}", dtoClass.getName(), e.getMessage(), e);
            return null;
        }

        // 逐列解析+校验（复用ExportColumnConfig规则）
        for (int colIndex = 0; colIndex < columnConfigs.size(); colIndex++) {
            ExportColumnConfig config = columnConfigs.get(colIndex);
            String fieldName = config.getFieldName();
            Cell cell = row.getCell(colIndex);
            String cellValue = getCellStringValue(cell);
            // 记录原始行数据，用于错误Map
            rowData.add((T) cellValue);

            // 字段赋值（兼容数值/字符串类型）
            try {
                if (StringUtils.isNotBlank(cellValue)) {
                    if (config.isNumberValidation()) {
                        // 数值类型字段：转换为BigDecimal赋值
                        Double numericValue = Double.parseDouble(cellValue);
                        BeanUtil.setProperty(dto, fieldName, new BigDecimal(numericValue));
                    } else {
                        // 字符串类型字段：直接赋值
                        BeanUtil.setProperty(dto, fieldName, cellValue);
                    }
                }
            } catch (Exception e) {
                log.warn("字段赋值失败，字段名：{}，值：{}，错误：{}", fieldName, cellValue, e.getMessage());
            }

            // 原有校验规则（数值范围/枚举值）
            if (config.isNumberValidation() && StringUtils.isNotBlank(cellValue)) {
                // 数值范围校验
                try {
                    BigDecimal value = new BigDecimal(cellValue);
                    if (value.compareTo(new BigDecimal(config.getValidationNumberMin())) < 0
                            || value.compareTo(new BigDecimal(config.getValidationNumberMax())) > 0) {
                        rowErrorMsg.append(config.getValidationErrorMsg()).append("；");
                    }
                } catch (Exception e) {
                    rowErrorMsg.append(config.getValidationErrorMsg()).append("；");
                }
            } else if (CollectionUtils.isNotEmpty(config.getValidationValidOptions()) && StringUtils.isNotBlank(cellValue)) {
                // 枚举值校验
                if (!config.getValidationValidOptions().contains(cellValue)) {
                    rowErrorMsg.append(config.getValidationErrorMsg()).append("；");
                }
            }

        }

        // 收集错误
        if (rowErrorMsg.length() > 0) {
            String errorKey = rowErrorMsg.toString();
            errorMap.computeIfAbsent(errorKey, k -> new ArrayList<>()).addAll(rowData);
            // 给DTO设置错误信息（需DTO包含errorMsg字段）
            try {
                BeanUtil.setProperty(dto, "errorMsg", errorKey);
            } catch (Exception e) {
                log.warn("DTO[{}]无errorMsg字段，跳过错误信息赋值", dtoClass.getSimpleName());
            }
        } else {
            // 即使没有错误，也要确保errorMsg字段有默认值
            try {
                BeanUtil.setProperty(dto, "errorMsg", "");
            } catch (Exception e) {
                log.debug("DTO[{}]无errorMsg字段或设置默认值失败", dtoClass.getSimpleName());
            }
        }

        // 返回DTO（即使有错误也返回）
        return dto;
    });

    // 打印基础错误（如文件格式、空Sheet）
    if (CollectionUtils.isNotEmpty(baseErrorList)) {
        log.warn("导入基础校验错误：{}", String.join("；", baseErrorList));
        throw new RuntimeException(String.join("；", baseErrorList));
    }
    return dtoList;
}




    /**
     * 基础AP Excel导入方法（通用行解析，无业务耦合）
     * @param inputStream Excel文件流
     * @param fileName 文件名（校验格式）
     * @param errorList 基础错误列表（文件格式、空Sheet等）
     * @param rowParser 行解析器（自定义每行解析逻辑）
     * @param <T> 解析结果类型
     * @return 解析后的列表
     */
    private static <T> List<T> readApExcel(InputStream inputStream,
                                          String fileName,
                                          List<String> errorList,
                                          BiFunction<Row, Integer, T> rowParser) {
        log.info("开始基础Excel导入解析，文件名：{}", fileName);

        List<T> dataList = new ArrayList<>();
        Workbook workbook = null;

        try {
            // 1. 校验文件格式
            if (fileName == null || (!fileName.endsWith(".xls") && !fileName.endsWith(".xlsx"))) {
                errorList.add("文件格式错误，仅支持.xls和.xlsx");
                return dataList;
            }

            // 2. 创建Workbook
            if (fileName.endsWith(".xls")) {
                workbook = new HSSFWorkbook(inputStream);
            } else {
                workbook = new XSSFWorkbook(inputStream);
            }

            // 3. 校验Sheet
            Sheet sheet = workbook.getSheetAt(0);
            if (sheet == null) {
                errorList.add("获取sheet为空，导入失败，请检查文件！");
                return dataList;
            }
            int lastRowNum = sheet.getLastRowNum();
            if (lastRowNum <= HEADER_ROW) {
                errorList.add("获取数据行数小于等于表头行数，文件内无内容，导入失败，请检查文件！");
                return dataList;
            }

            // 4. 逐行解析
            for (int rowIndex = HEADER_ROW + 1; rowIndex <= lastRowNum; rowIndex++) {
                Row row = sheet.getRow(rowIndex);
                if (row == null) {
                    // 调整：移除行号，仅保留纯错误提示
                    errorList.add("数据行为空，跳过处理");
                    continue;
                }
                T rowData = rowParser.apply(row, rowIndex + 1);
                if (rowData != null) {
                    dataList.add(rowData);
                }
            }
        } catch (IOException e) {
            log.error("导入Excel文件失败：{}", e.getMessage(), e);
            errorList.add("文件读取失败：" + e.getMessage());
        } finally {
            // 5. 关闭流
            try {
                if (workbook != null) {
                    workbook.close();
                }
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (IOException e) {
                log.error("Excel流关闭异常", e);
            }
        }

        return dataList;
    }

    /* ============================================================导出部分=========================================================== */

    /**
     * 将Excel字节数组导出为文件（写入Http响应流）
     *
     * @param excelBytes     生成的Excel字节数组
     * @param response       Http响应对象
     * @param fileNamePrefix 导出文件名前缀（如"常规陈列"）
     * @throws Exception 导出异常
     */
    public static void exportExcelBytesToResponse(byte[] excelBytes, HttpServletResponse response, String fileNamePrefix) throws Exception {
        if (excelBytes == null || excelBytes.length == 0) {
            throw new Exception("导出Excel失败：生成的Excel字节数组为空");
        }
        if (StringUtils.isBlank(fileNamePrefix)) {
            fileNamePrefix = "导出文件";
        }

        // 1. 设置响应头（解决中文乱码、指定文件类型）
        String fileName = String.format("%s_%s.xlsx", fileNamePrefix, new SimpleDateFormat("yyyyMMdd").format(new Date()));
        // URLEncoder编码兼容所有浏览器
        String encodedFileName = URLEncoder.encode(fileName, String.valueOf(StandardCharsets.UTF_8));

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setHeader("Content-Disposition", "attachment;filename=" + encodedFileName);
        // 禁止缓存
        response.setHeader("Pragma", "no-cache");
        response.setHeader("Cache-Control", "no-store, no-cache, must-revalidate");
        response.setDateHeader("Expires", 0);
        // 设置文件大小（可选，提升下载体验）
        response.setContentLength(excelBytes.length);

        // 2. 将字节数组写入响应流
        try (OutputStream outputStream = response.getOutputStream()) {
            outputStream.write(excelBytes);
            outputStream.flush();
        } catch (Exception e) {
            log.error("写入Excel响应流失败：{}", e.getMessage(), e);
            throw new Exception("导出Excel失败：" + e.getMessage());
        }
    }

    /**
     * 生成Excel字节流
     *
     * @param exportColumnConfigList 表头配置
     * @param dataList               数据列表
     * @param sheetName              sheetName
     * @return byte[] excel 数组
     * @throws IOException
     */
    public static byte[] generateExcelBytes(List<ExportColumnConfig> exportColumnConfigList, List<?> dataList, String sheetName) throws IOException {
        //数据校验
        if (CollectionUtils.isEmpty(exportColumnConfigList)) {
            throw new IOException("导出Excel失败：没有配置导出列");
        }
        if (CollectionUtils.isEmpty(dataList)) {
            throw new IOException("导出Excel失败：没有获取到相关数据，请检查条件后重试");
        }
        if (sheetName == null || sheetName.trim().isEmpty()) {
            throw new IOException("导出Excel失败：没有配置sheetName");
        }

        // 根据sheetName创建工作表
        try ( // 创建工作簿
              Workbook workbook = createExcelWorkbook(dataList);
              // 创建文件字节数组
              ByteArrayOutputStream outputStream = new ByteArrayOutputStream();) {
            try {
                Sheet sheet = workbook.createSheet(sheetName);
                // 创建表头
                createHeader(sheet, exportColumnConfigList);
                // 创建数据行
                createDataRows(sheet, dataList, exportColumnConfigList);

                // 处理需要验证的列
                for (int i = 0; i < exportColumnConfigList.size(); i++) {
                    ExportColumnConfig config = exportColumnConfigList.get(i);
                    // 给列配置数据验证(跳过表头)
                    ExcelStyleUtils.addColumnDataValidation(sheet, i, config);
                    if (isColumnProtected(config)) {
                        // 为该列所有数据行添加只读验证
                        // 从第1行开始（跳过表头）
                        addReadOnlyValidation(sheet, i, 1, dataList.size());
                    }
                }
                workbook.write(outputStream);
                return outputStream.toByteArray();
            } finally {
                //如果创建的是 sxssf 清理临时文件
                if (workbook instanceof SXSSFWorkbook) {
                    ((SXSSFWorkbook) workbook).dispose();
                }
                workbook.close();
            }

        } catch (IOException e) {
            log.error("生成Excel文件失败：{}", e.getMessage(), e);
            throw new IOException("生成Excel文件失败：" + e.getMessage());
        }
    }


    /* ============================================================导出工具方法=========================================================== */

    /**
     * 创建工作簿
     *
     * @param dataList 数据列表
     * @return 工作簿
     */
    private static Workbook createExcelWorkbook(List<?> dataList) {
        if (dataList.size() > 1000) {
            SXSSFWorkbook sxssfWorkbook = new SXSSFWorkbook(100);
            sxssfWorkbook.setCompressTempFiles(true);
            return sxssfWorkbook;
        } else {
            return new XSSFWorkbook();
        }
    }

    /**
     * 创建表头
     *
     * @param sheet                  工作表
     * @param exportColumnConfigList 表头配置
     */
    public static void createHeader(Sheet sheet, List<ExportColumnConfig> exportColumnConfigList) {
        Workbook workbook = sheet.getWorkbook();
        Row headerRow = sheet.createRow(0);
        // 调整表头行高（适配换行）
        headerRow.setHeightInPoints(40);

        for (int i = 0; i < exportColumnConfigList.size(); i++) {
            ExportColumnConfig config = exportColumnConfigList.get(i);
            Cell cell = headerRow.createCell(i);
            String originalHeader = config.getHeaderName() == null ? "" : config.getHeaderName();
            // 处理表头文本：拆分括号内容+换行
            String processedHeader = processHeaderText(originalHeader);
            cell.setCellValue(processedHeader);
            // 使用 HEADER_STYLE 作为表头的基础样式
            CellStyle headerStyle = cacheStyle(workbook, ExcelStyleUtils.ExcelStyle.HEADER_STYLE);
            cell.setCellStyle(headerStyle);
            cell.getCellStyle().setWrapText(true);
            // 表头强制锁定（不可修改）
            cell.getCellStyle().setLocked(true);
            // 设置初始列宽
            String headerText = config.getHeaderName();
            if (headerText != null) {
                int maxWidth = calculateTextWidth(processedHeader);
                // 至少20个字符宽度
                sheet.setColumnWidth(i, Math.max(maxWidth, 20 * 256));
            }
        }
        // 为表头行添加只读验证（防止用户修改表头）
        for (int i = 0; i < exportColumnConfigList.size(); i++) {
            ExportColumnConfig exportColumnConfig = exportColumnConfigList.get(i);
            if (exportColumnConfig.getValidationPromptTitle() != null && exportColumnConfig.getValidationPromptMsg() != null) {
                addHeaderPromptValidation(sheet, i, exportColumnConfig);
            }
            // 只保护第0行（表头行）
            addReadOnlyValidation(sheet, i, HEADER_ROW, HEADER_ROW);

        }
    }


    /**
     * 创建数据行
     *
     * @param sheet                  工作表
     * @param dataList               数据列表
     * @param exportColumnConfigList 表头配置
     */

    public static void createDataRows(Sheet sheet, List<?> dataList, List<ExportColumnConfig> exportColumnConfigList) {
        Workbook workbook = sheet.getWorkbook();
        for (int i = 0; i < dataList.size(); i++) {
            // 创建数据行
            Row dataRow = sheet.createRow(i + 1);
            // 获取数据
            Object data = dataList.get(i);
            // 渲染数据列
            for (int j = 0; j < exportColumnConfigList.size(); j++) {
                // 获取数据列的配置
                ExportColumnConfig exportColumnConfig = exportColumnConfigList.get(j);
                Cell cell = dataRow.createCell(j);

                // 处理条件样式
                CellStyle style;
                if (exportColumnConfig.isConditionalStyling() && j > 0) {
                    // 检查前一列是否有值
                    Cell previousCell = dataRow.getCell(j - 1);
                    boolean hasPreviousValue = false;

                    if (previousCell != null) {
                        switch (previousCell.getCellType()) {
                            case STRING:
                                hasPreviousValue = previousCell.getStringCellValue() != null &&
                                        !previousCell.getStringCellValue().trim().isEmpty();
                                break;
                            case NUMERIC:
                                hasPreviousValue = true;
                                break;
                            case BOOLEAN:
                                hasPreviousValue = true;
                                break;
                            default:
                        }
                    }
                    // 根据前一列是否有值来设置样式
                    if (hasPreviousValue) {
                        style = cacheStyle(workbook, ExcelStyleUtils.ExcelStyle.LIGHT_BLUE_BG);
                    } else {
                        style = cacheStyle(workbook, ExcelStyleUtils.ExcelStyle.DEFAULT_STYLE);
                    }
                } else {
                    // 使用配置的默认样式
                    style = cacheStyle(workbook, exportColumnConfig.getStyle());
                }

                cell.setCellStyle(style);
                Object value = getFieldValueByOgnl(exportColumnConfig.getFieldName(), data);
                // 设置单元格值并应用格式
                setCellValueWithFormat(cell, value, exportColumnConfig.getFormat());
            }
        }
    }


    /**
     * 添加表头提示
     *
     * @param sheet       工作表
     * @param columnIndex 列索引
     * @param config      表头配置
     */
    private static void addHeaderPromptValidation(Sheet sheet, int columnIndex, ExportColumnConfig config) {
        DataValidationHelper helper = sheet.getDataValidationHelper();
        // 创建总是为真的约束
        DataValidationConstraint constraint = helper.createCustomConstraint("FALSE");

        // 仅对表头行（第0行）应用
        CellRangeAddressList addressList = new CellRangeAddressList(HEADER_ROW, HEADER_ROW, columnIndex, columnIndex);

        DataValidation validation = helper.createValidation(constraint, addressList);

        // 设置提示信息
        validation.setShowPromptBox(true);
        validation.createPromptBox(config.getValidationPromptTitle(), config.getValidationPromptMsg());
        validation.setShowErrorBox(true);
        validation.createErrorBox("禁止修改", "单元格内容禁止修改！");
        validation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        validation.setEmptyCellAllowed(false);
        validation.setSuppressDropDownArrow(false);

        sheet.addValidationData(validation);
    }

    /**
     * 根据字段名获取字段值
     *
     * @param fieldName 字段名
     * @param data      数据对象
     * @return 字段值
     */
    private static Object getFieldValueByOgnl(String fieldName, Object data) {
        // 避免空指针访问
        if (data == null || fieldName == null || fieldName.trim().isEmpty()) {
            return "";
        }
        try {
            // 使用 BeanUtils 获取属性值
            return BeanUtil.getProperty(data, fieldName);
        } catch (Exception e) {
            log.error("获取字段值失败：{}", fieldName, e);
            // 空值兜底，避免崩溃
            return "";
        }
    }

    /**
     * 设置单元格值并应用格式（修复：日期/数字格式处理）
     *
     * @param cell   单元格
     * @param value  数据值
     * @param format 格式（如yyyy-MM-dd、0.00）
     */
    private static void setCellValueWithFormat(Cell cell, Object value, String format) {
        if (value == null) {
            cell.setCellValue("");
            return;
        }

        // 特殊处理：如果格式为 "@" 或者字段名包含某些关键词，则强制作为文本处理
        if ("@".equals(format)) {
            cell.setCellValue(value.toString());
            return;
        }

        // 1. 日期类型
        if (value instanceof Date) {
            if (format != null && !format.isEmpty()) {
                SimpleDateFormat sdf = new SimpleDateFormat(format);
                cell.setCellValue(sdf.format((Date) value));
            } else {
                cell.setCellValue((Date) value);
            }
            return;
        }

        // 2. 数值类型（整数/小数）：设置为数值单元格，保证数值校验生效
        try {
            double numericValue = Double.parseDouble(value.toString());
            cell.setCellValue(numericValue);
            // 数值格式（如保留2位小数）
            if (format != null && !format.isEmpty()) {
                DataFormat formatInstance = cell.getSheet().getWorkbook().createDataFormat();
                CellStyle numericStyle = cell.getSheet().getWorkbook().createCellStyle();
                numericStyle.setDataFormat(formatInstance.getFormat(format));
                cell.setCellStyle(numericStyle);
            }
            return;
        } catch (NumberFormatException e) {
            // 非数值类型，走字符串逻辑
        }

        // 3. 字符串类型
        cell.setCellValue(value.toString());
    }

    /**
     * 计算文本宽度（支持换行）
     *
     * @param text 文本内容
     * @return 宽度值
     */
    private static int calculateTextWidth(String text) {
        if (text == null || text.isEmpty()) {
            return 10 * 256;
        }

        String[] lines = text.split("\n");
        int maxLineLength = 0;
        for (String line : lines) {
            maxLineLength = Math.max(maxLineLength, line.length());
        }

        // 每个字符大约占用256单位宽度，增加一些缓冲
        return maxLineLength * 256 + 1000;
    }


    /**
     * 缓存样式避免内存溢出
     *
     * @param workbook 工作簿
     * @param style    目标样式
     * @return 缓存的样式
     */
    private static CellStyle cacheStyle(Workbook workbook, ExcelStyleUtils.ExcelStyle style) {
        ExcelStyleUtils.ExcelStyle targetStyle = style == null ? ExcelStyleUtils.ExcelStyle.DEFAULT_STYLE : style;
        // 使用更稳定的缓存键
        String styleKey = System.identityHashCode(workbook) + "_" + targetStyle.name();
        if (!CACHE_CELL_STYLE.containsKey(styleKey)) {
            CellStyle cellStyle = ExcelStyleUtils.getStyle(workbook, targetStyle);
            CACHE_CELL_STYLE.put(styleKey, cellStyle);
            return cellStyle;
        }
        return CACHE_CELL_STYLE.get(styleKey);
    }

    /**
     * 处理表头文本：拆分括号内容+换行
     *
     * @param originalText 原始文本
     * @return 处理后的文本
     */
    private static String processHeaderText(String originalText) {
        if (originalText == null || originalText.isEmpty()) {
            return "";
        }
        // 替换中文括号及其内容为换行形式
        String result;
        if (originalText.contains("（") && originalText.contains("）")) {
            // 中文括号替换为换行
            result = originalText.replaceAll("（(.*?)）", "\n$1");
        } else if (originalText.contains("(") && originalText.contains(")")) {
            // 英文括号替换为换行
            result = originalText.replaceAll("\\((.*?)\\)", "\n$1");
        } else if (originalText.contains("-")) {
            // 横线替换为换行（内容）
            result = originalText.replaceAll("-(.+)", "\n（$1）");
        } else if (originalText.contains("_")) {
            result = originalText.replaceAll("_(.+)", "\n（$1）");
        } else {
            result = originalText;
        }

        return result;
    }

    /**
     * 为特定单元格区域添加只读数据验证
     *
     * @param sheet       工作表
     * @param columnIndex 列索引
     * @param startRow    起始行
     * @param endRow      结束行
     */
    private static void addReadOnlyValidation(Sheet sheet, int columnIndex, int startRow, int endRow) {
        DataValidationHelper helper = sheet.getDataValidationHelper();

        // 创建自定义公式验证，始终返回false（禁止输入）
        DataValidationConstraint constraint = helper.createCustomConstraint("FALSE");

        CellRangeAddressList addressList = new CellRangeAddressList(startRow, endRow, columnIndex, columnIndex);
        DataValidation validation = helper.createValidation(constraint, addressList);

        validation.setShowErrorBox(true);
        validation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        validation.createErrorBox("禁止修改", "此单元格内容不可修改");
        validation.setEmptyCellAllowed(false);
        validation.setSuppressDropDownArrow(false);

        sheet.addValidationData(validation);
    }

    /**
     * 判断列是否需要保护
     *
     * @param config 列配置
     * @return 是否需要保护
     */
    private static boolean isColumnProtected(ExportColumnConfig config) {
        return config.getStyle() == ExcelStyleUtils.ExcelStyle.UNMODIFIABLE || config.getStyle() == ExcelStyleUtils.ExcelStyle.CHANGE_TEXT_STYLE;
    }

    /* ================================================导入工具方法=================================================== */

    /**
     * 获取单元格内容
     *
     * @param cell 单元格
     * @return 单元格内容
     */
    public static String getCellStringValue(Cell cell) {
        if (cell == null) {
            return "";
        }

        CellType cellType = cell.getCellType();
        switch (cellType) {
            case STRING:
                return cell.getStringCellValue().trim();
            case NUMERIC:
                // 处理日期/数字
                if (DateUtil.isCellDateFormatted(cell)) {
                    return cell.getDateCellValue().toString();
                } else {
                    // 避免科学计数法，去除末尾的.0
                    return String.valueOf(cell.getNumericCellValue()).replaceAll("\\.0*$", "");
                }
            case BOOLEAN:
                return String.valueOf(cell.getBooleanCellValue());
            case FORMULA:
                try {
                    return String.valueOf(cell.getNumericCellValue()).replaceAll("\\.0*$", "");
                } catch (IllegalStateException e) {
                    return cell.getStringCellValue().trim();
                }
            default:
                return "";
        }
    }

    /**
     * 判断单元格是否有值（与导出时的条件样式判断逻辑完全一致）
     * @param cell 单元格
     * @return true=有值，false=无值
     */
    private static boolean isCellHasValue(Cell cell) {
        if (cell == null) {
            return false;
        }
        switch (cell.getCellType()) {
            case STRING:
                return StringUtils.isNotBlank(cell.getStringCellValue().trim());
            case NUMERIC:
                // 数值类型无论值是多少都算有值
                return true;
            case BOOLEAN:
                // 布尔类型算有值
                return true;
            default:
                return false;
        }
    }

    /**
     *  获取STS临时凭证
     * @return STS临时凭证
     * @throws ClientException
     */
    private static AssumeRoleResponse.Credentials getStsCredentials() throws ClientException {
        OssConfigProperties.Oss oss = staticOssConfig.getOss();
        // 构建STS客户端
        IClientProfile profile = DefaultProfile.getProfile(
                oss.getRegionId(),
                staticOssConfig.getAccessKeyId(),
                staticOssConfig.getAccessKeySecret()
        );
        DefaultAcsClient client = new DefaultAcsClient(profile);

        // 构建STS请求
        AssumeRoleRequest request = new AssumeRoleRequest();
        // POST方式发送请求
        request.setMethod(MethodType.POST);
        // 你的STS角色ARN
        request.setRoleArn(oss.getStsRoleArm());

        request.setRoleSessionName(oss.getSessionName());
        // 获取STS响应
        AssumeRoleResponse response = client.getAcsResponse(request);
        AssumeRoleResponse.Credentials credentials = response.getCredentials();
        if (credentials == null) {
            throw new ClientException("STS临时凭证获取失败");
        }
        log.info("STS临时凭证获取成功，过期时间：{}", credentials.getExpiration());
        return credentials;
    }

    /**
     * 从OSS文件URL中解析出ObjectKey
     * @param ossFileUrl OSS文件URL
     * @param ossDomain OSS域名
     * @return ObjectKey
     */
    private static String parseObjectKeyFromUrl(String ossFileUrl, String ossDomain) {
        if (StringUtils.isBlank(ossFileUrl) || StringUtils.isBlank(ossDomain)) {
            return null;
        }
        String domainPrefix = "//" + ossDomain + "/";
        if (ossFileUrl.contains(domainPrefix)) {
            return ossFileUrl.split(domainPrefix)[1];
        }
        // 兼容直接传入objectKey的场景（如path/file.xlsx）
        if (!ossFileUrl.startsWith("http")) {
            return ossFileUrl;
        }
        return null;
    }

    /**
     * 从OSS文件URL中解析出文件名
     * @param ossUrl OSS文件URL
     * @return 文件名
     */
    public static String parseFileNameFromOssUrl(String ossUrl) {
        //参数校验
        if (StringUtils.isBlank(ossUrl) || !ossUrl.contains("oss-") || !ossUrl.contains(".aliyuncs.com")) {
            log.warn("非有效阿里云OSS URL，使用通用默认文件名：{}", ossUrl);
            // 通用默认名，不绑定具体业务
            return "Excel导入文件";
        }

        // 解析文件名：按"/"分割取URL最后一段
        String[] urlParts = ossUrl.split("/");
        String fileName = urlParts[urlParts.length - 1];

        // - 解析结果为空 → 通用前缀+时间戳+默认后缀
        // - 无Excel后缀 → 补充.xlsx（适配Excel导入通用场景）
        if (StringUtils.isBlank(fileName)) {
            fileName = "Excel导入文件_" + System.currentTimeMillis() + ".xlsx";
        } else if (!fileName.endsWith(".xlsx") && !fileName.endsWith(".xls")) {
            // 保留原文件名前缀 + 补充Excel后缀（更友好）
            fileName = fileName + ".xlsx";
        }

        log.info("从OSS URL解析通用文件名完成：{} → {}", ossUrl, fileName);
        return fileName;
    }



}

