package com.sfa.operation.strategy.impl.imports;

import com.alibaba.nacos.shaded.com.google.gson.reflect.TypeToken;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.sfa.operation.config.ExportColumnConfig;
import com.sfa.operation.domain.sales.entity.SalesApDisplay;
import com.sfa.operation.pojo.sales.excel.SalesApDisplayImportExcelDto;
import com.sfa.operation.pojo.sales.vo.SalesApDisplayVo;
import com.sfa.operation.service.sales.impl.ApDisplayCoreServiceImpl;
import com.sfa.operation.service.sales.impl.ApDisplayQueryServiceImpl;
import com.sfa.operation.strategy.IImportApExcelStrategy;
import com.sfa.operation.strategy.impl.exports.NormalDisplayExportStrategyImpl;
import com.sfa.operation.util.excel.ExcelUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-11 17:31
 * @Description: AP导入excel(填报)控制类
 */

@Slf4j
@Component("normalDisplayImportStrategy")
public class NormalDisplayImportStrategyImpl implements IImportApExcelStrategy<SalesApDisplayImportExcelDto> {

    @Autowired
    private NormalDisplayExportStrategyImpl normalDisplayExportStrategy;
    @Autowired
    private ApDisplayQueryServiceImpl salesApQueryDisplayService;
    @Autowired
    private ApDisplayCoreServiceImpl salesApDisplayCoreService;

    @Override
    public List<ExportColumnConfig> getImportColumnConfig() {
        return normalDisplayExportStrategy.getExportColumnConfig();
    }


    @Override
    public List<SalesApDisplayImportExcelDto> parseExcelToDtoList(InputStream inputStream, String filePathUrl,
                                                                  Map<String, List<SalesApDisplayImportExcelDto>> errorMap) {
        // 基础校验
        if (inputStream == null || StringUtils.isBlank(filePathUrl)) {
            SalesApDisplayImportExcelDto errorDto = buildSimpleErrorDto("文件流/URL为空");
            errorMap.put("参数异常", Collections.singletonList(errorDto));
            // 修改：返回含错误DTO的列表
            return Collections.singletonList(errorDto);
        }

        // 复用ExcelUtils解析（内置列校验/错误收集）
        String fileName = ExcelUtils.parseFileNameFromOssUrl(filePathUrl);

//        String fileName = ExcelUtils.extractFileNameFromUrl(filePathUrl);
        List<ExportColumnConfig> columnConfigs = getImportColumnConfig();
        // 基础校验
        return ExcelUtils.readApExcelWithColumnConfig(inputStream, fileName, columnConfigs, errorMap, SalesApDisplayImportExcelDto.class);
    }

    @Override
    public void preprocessData(List<SalesApDisplayImportExcelDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }

        dtoList.forEach(dto -> {
            // 修改：初始化errorMsg为空字符串，避免null
            if (dto.getErrorMsg() == null) {
                dto.setErrorMsg("");
            }

            StringBuilder errorMsg = new StringBuilder();
            if (dto.getSadId() == null) {
                errorMsg.append("序号不能为空；");
            }

            // 合并错误信息
            if (errorMsg.length() > 0) {
                String newError = errorMsg.toString().replaceAll("；$", "");
                dto.setErrorMsg(StringUtils.isBlank(dto.getErrorMsg())
                        ? newError
                        : dto.getErrorMsg() + "；" + newError);
            }
        });
    }

    @Override
    public int queryAndBatchOperate(List<SalesApDisplayImportExcelDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return 0;
        }

        // 过滤有效数据（无预处理错误）
        List<SalesApDisplayImportExcelDto> validDtoList = dtoList.stream()
                .filter(d -> StringUtils.isBlank(d.getErrorMsg()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(validDtoList)) {
            return 0;
        }

        // 先进行查询验证，确保所有数据都能在数据库中找到对应记录
        int successCount = 0;

        List<SalesApDisplay> salesApDisplayList = queryData(validDtoList);
        Map<String, SalesApDisplay> displayMap = new HashMap<>();
        for (SalesApDisplay display : salesApDisplayList) {
            String matchKey = buildMatchKey(
                    display.getSadId(),
                    display.getRegionName(),
                    display.getDealerName(),
                    display.getLineName()
            );
            displayMap.put(matchKey, display);
        }
        for (SalesApDisplayImportExcelDto dto : validDtoList) {
            String dtoMatchKey = buildMatchKey(
                    dto.getSadId(),
                    dto.getRegionName(),
                    dto.getDealerName(),
                    dto.getLineName()
            );

            if (displayMap.containsKey(dtoMatchKey)) {
                successCount++;
            } else {
                // 错误提示仅显示「序号为{sadId}的数据不存在」
                String errorMsg = dto.getSadId() != null
                        ? "序号为" + dto.getSadId() + "的数据不存在"
                        : "序号为空的数据不存在"; // sadId为空时的兜底提示

                // 保留原有错误信息拼接逻辑（如有其他错误，追加当前提示）
                dto.setErrorMsg(StringUtils.isNotBlank(dto.getErrorMsg())
                        ? dto.getErrorMsg() + "；" + errorMsg
                        : errorMsg);
            }
        }

        return successCount;
    }



    @Override
    public List<SalesApDisplay>buildUpdateEntityList(List<SalesApDisplayImportExcelDto> dtoList) {
        // 空值快速返回
        if (CollectionUtils.isEmpty(dtoList)) {
            return Collections.emptyList();
        }

        List<SalesApDisplay> updateEntityList = new ArrayList<>();

        // 批量查询所有DTO对应的计划数据，避免循环内重复查询
        List<SalesApDisplay> salesApDisplayList = queryData(dtoList);
        Map<String, SalesApDisplay> displayMap = new HashMap<>();
        for (SalesApDisplay display : salesApDisplayList) {
            String matchKey = buildMatchKey(
                    display.getSadId(),
                    display.getRegionName(),
                    display.getDealerName(),
                    display.getLineName()
            );
            displayMap.put(matchKey, display);
        }

        for (SalesApDisplayImportExcelDto dto : dtoList) {
            SalesApDisplayVo salesApDisplayVo = new SalesApDisplayVo();
            BeanUtils.copyProperties(dto, salesApDisplayVo);

            // 从Map中获取当前DTO对应的计划数据
            SalesApDisplay salesApDisplay = displayMap.get(buildMatchKey(
                    dto.getSadId(),
                    dto.getRegionName(),
                    dto.getDealerName(),
                    dto.getLineName()
            ));
            if (salesApDisplay == null) {
                log.warn("序号为{}的DTO无匹配计划数据，跳过处理", dto.getSadId());
                continue;
            }

            // 主货架执行状态计算
            if (salesApDisplay.getPlannedMainShelfType() != null && salesApDisplay.getPlannedMainShelfQty() != null
                    && salesApDisplayVo.getActualMainShelfType() != null && salesApDisplayVo.getActualMainShelfQty() != null) {
                boolean mainShelfTypeMatch = salesApDisplayVo.getActualMainShelfType().equals(salesApDisplay.getPlannedMainShelfType());
                boolean mainShelfQtySufficient = salesApDisplayVo.getActualMainShelfQty() >= salesApDisplay.getPlannedMainShelfQty();
                salesApDisplayVo.setActualMainShelfExecuted((mainShelfTypeMatch && mainShelfQtySufficient) ? "执行" : "未执行");
            }

            // 端架执行状态计算
            if (salesApDisplayVo.getActualEndCapQty() != null && salesApDisplay.getPlannedEndCapQty() != null) {
                salesApDisplayVo.setActualEndCapExecuted(
                        salesApDisplayVo.getActualEndCapQty() >= salesApDisplay.getPlannedEndCapQty() ? "执行" : "未执行"
                );
            }

            // 地堆执行状态计算
            if (salesApDisplay.getPlannedFloorStackArea() != null && salesApDisplay.getPlannedFloorStackQty() != null
                    && salesApDisplayVo.getActualFloorStackArea() != null && salesApDisplayVo.getActualFloorStackQty() != null) {
                boolean areaSufficient = salesApDisplayVo.getActualFloorStackArea() >= salesApDisplay.getPlannedFloorStackArea();
                boolean qtySufficient = salesApDisplayVo.getActualFloorStackQty() >= salesApDisplay.getPlannedFloorStackQty();
                salesApDisplayVo.setActualFloorStackExecuted((areaSufficient && qtySufficient) ? "执行" : "未执行");
            }

            // 多点陈列执行状态
            if (salesApDisplay.getPlannedMultiDisplay() != null && salesApDisplayVo.getActualMultiDisplay() != null) {
                salesApDisplayVo.setActualMultiDisplayExecuted(
                        StringUtils.equals("执行与计划一致", salesApDisplayVo.getActualMultiDisplay()) ? "执行" : "未执行"
                );
            }

            // 挂条执行状态字段赋值错误
            if (salesApDisplay.getPlannedHangingStripQuantityForm() != null && salesApDisplayVo.getActualHangingStripQuantityForm() != null) {
                salesApDisplayVo.setHangingStripExecuted(
                        StringUtils.equals("执行与计划一致", salesApDisplayVo.getActualHangingStripQuantityForm()) ? "执行" : "未执行"
                );
            }

            // VO转实体逻辑
            SalesApDisplay updatedEntity = new SalesApDisplay();
            BeanUtils.copyProperties(salesApDisplayVo, updatedEntity);
            updateEntityList.add(updatedEntity);
        }

        return updateEntityList;
    }

    @Override
    public Map<String, Object> execute(String filePathUrl) {
        Map<String, Object> resultMap = new HashMap<>(5);
        InputStream inputStream = null;

        try {
            // 获取文件流（修改：文件流为空时构建错误DTO）
            inputStream = ExcelUtils.getOssFileInputStream(filePathUrl);
            if (inputStream == null) {
                SalesApDisplayImportExcelDto errorDto = buildSimpleErrorDto("文件流获取失败：OSS文件不存在或权限不足");
                List<SalesApDisplayImportExcelDto> errorList = Collections.singletonList(errorDto);

                resultMap.put("uuid", UUID.randomUUID().toString());
                resultMap.put("table", errorList);
                resultMap.put("successCount", 0);
                resultMap.put("failCount", errorList.size());
                resultMap.put("errorMsg", "文件流获取失败：OSS文件不存在或权限不足");
                return resultMap;
            }

            // 解析Excel
            Map<String, List<SalesApDisplayImportExcelDto>> tempErrorMap = new HashMap<>();
            List<SalesApDisplayImportExcelDto> dtoList = parseExcelToDtoList(inputStream, filePathUrl, tempErrorMap);

            // 数据预处理
            preprocessData(dtoList);

            //批量验证/操作）
            int successCount = queryAndBatchOperate(dtoList);

            // 5封装返回结果
            resultMap.put("uuid", UUID.randomUUID().toString());
            resultMap.put("table", dtoList);
            resultMap.put("successCount", successCount);
            resultMap.put("failCount", dtoList.size() - successCount);
            resultMap.put("errorMsg", "");

        } catch (Exception e) {
            log.error("常规陈列导入失败，filePathUrl={}", filePathUrl, e);
            // 修改：异常时构建错误DTO，存入table
            SalesApDisplayImportExcelDto errorDto = buildSimpleErrorDto("导入失败：" + e.getMessage());
            List<SalesApDisplayImportExcelDto> errorList = Collections.singletonList(errorDto);

            resultMap.put("uuid", UUID.randomUUID().toString());
            resultMap.put("table", errorList);
            resultMap.put("successCount", 0);
            resultMap.put("failCount", errorList.size());
            resultMap.put("errorMsg", errorList.get(0).getErrorMsg());
        } finally {
            // 关闭流（不变）
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (Exception e) {
                    log.error("关闭文件流失败", e);
                }
            }
        }
        return resultMap;
    }


    @Override
    public  List<SalesApDisplay> queryData(List<SalesApDisplayImportExcelDto> dto) {

        // 查询数据库 此处查询正常只会返回一个对象
        return salesApQueryDisplayService.queryByCondition(dto);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String updateDisplay(List<SalesApDisplayImportExcelDto> dtoList) {

        if (dtoList == null || dtoList.isEmpty()) {
            return "更新失败";
        }
        List<SalesApDisplay> salesApDisplayList = buildUpdateEntityList(dtoList);
        if (salesApDisplayList.isEmpty()) {
            return "更新失败";
        }
        int i = salesApDisplayCoreService.batchUpdate(salesApDisplayList);
        return i > 0 ? "更新成功" : "更新失败";
    }



    @Override
    public List<SalesApDisplayImportExcelDto> getTransactionJsonToObject(String json) {
        if (StringUtils.isBlank(json)) {
            return Collections.emptyList();
        }

        try {
            JsonObject jsonObject = JsonParser.parseString(json).getAsJsonObject();
            // 获取table字段
            if (jsonObject.has("table") && !jsonObject.get("table").isJsonNull()) {
                Gson gson = new Gson();
                String tableJson = jsonObject.get("table").toString();

                return gson.fromJson(tableJson, new TypeToken<List<SalesApDisplayImportExcelDto>>(){}.getType());
            }
            return Collections.emptyList();
        } catch (Exception e) {
            log.error("解析JSON失败: {}", json, e);
            return Collections.emptyList();
        }
    }


    /**
     * 统一构建组合匹配键（保证DTO和查询结果的键规则一致）
     */
    private String buildMatchKey(Long sadId, String regionName, String dealerName, String lineName) {
        return (sadId == null ? "" : sadId) + "_"
                + (regionName == null ? "" : regionName.trim()) + "_"
                + (dealerName == null ? "" : dealerName.trim()) + "_"
                + (lineName == null ? "" : lineName.trim());
    }

    /**
     * 构建错误DTO（仅用于参数异常场景）
     */
    public SalesApDisplayImportExcelDto buildSimpleErrorDto(String errorMsg) {
        SalesApDisplayImportExcelDto dto = new SalesApDisplayImportExcelDto();
        dto.setErrorMsg(errorMsg);
        return dto;
    }
}