package com.sfa.operation.factory;

import com.sfa.operation.enums.ImportApType;
import com.sfa.operation.strategy.IImportApExcelStrategy;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * @Author: DouXinYu
 * @Date: 2025-12-10 16:23
 * @Description: 店内执行上报 - 填报- 导入策略工厂类
 */
@Slf4j
@Component
public class ApImportExcelStrategyFactory {
    private final Map<String, IImportApExcelStrategy> importApExcelStrategyMap;

    //构造器注入
    @Autowired
    public ApImportExcelStrategyFactory(Map<String, IImportApExcelStrategy> importApExcelStrategyMap) {
        this.importApExcelStrategyMap = importApExcelStrategyMap;
    }

    /**
     * 获取策略
     * @param importApType 点击导入的类型
     * @return 对应策略实现类
     */
    public IImportApExcelStrategy getStrategy(String importApType) {
        log.info("店内执行填报导入策略注入开始，目标类型为：{}", importApType);

        // 参数校验
        if (importApType == null || importApType.trim().isEmpty()){
            log.error("店内执行填报导入策略注入失败，失败原因：传入的导入类型为空!");
            return null;
        }

        //转换为枚举类型
        ImportApType targetType;
        try {
            targetType = ImportApType.valueOf(importApType);
        } catch (IllegalArgumentException e) {
            log.error("店内执行填报导入策略注入失败，失败原因：传入的导入类型不存在!目标类型为：{}", importApType);
            return null;
        }

        // 从枚举中获取策略Bean名称，再从Map中查找策略
        String beanName = targetType.getImportStrategy();
        IImportApExcelStrategy strategy = importApExcelStrategyMap.get(beanName);
        if (strategy == null) {
            log.error("店内执行填报导入策略注入失败，失败原因：未找到对应的导入策略!目标策略Bean名称为：{}", beanName);
            return null;
        }

        log.info("店内执行填报导入策略注入成功，目标策略Bean名称为：{}", beanName);
        return strategy;
    }
}
