package com.sfa.operation.aspect;

import com.sfa.common.core.constant.RoleConstants;
import com.sfa.common.core.enums.ECode;
import com.sfa.common.core.exception.CheckedException;
import com.sfa.common.security.annotation.SalesPermissionCheck;
import com.sfa.common.security.utils.SecurityUtils;
import com.sfa.operation.config.ConstantValue;
import com.sfa.operation.pojo.sales.request.SalesApRequest;
import com.sfa.operation.service.qc.IQinceMarketEmployeeService;
import com.sfa.system.api.domain.SysRole;
import com.sfa.system.api.model.LoginUser;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

/**
 * @author : liqiulin
 * @date : 2025-12-10 11
 * @describe :
 */
@Aspect
@Component
public class SalesPermissionAspect {
    @Autowired
    private IQinceMarketEmployeeService qinceMarketEmployeeService;
    @Autowired
    private ConstantValue constantValue;


    @Around("@annotation(salesPermissionCheck)")
    public Object checkPermission(ProceedingJoinPoint joinPoint, SalesPermissionCheck salesPermissionCheck) throws Throwable {
        // 获取请求参数
        Object[] args = joinPoint.getArgs();
        SalesApRequest salesApRequest = null;

        for (Object arg : args) {
            if (arg instanceof SalesApRequest) {
                salesApRequest = (SalesApRequest) arg;
                break;
            }
        }

        if (salesApRequest != null) {
            if (salesPermissionCheck.value() == SalesPermissionCheck.CheckType.USER) {
                checkPermission(salesApRequest);
            }
            if (salesPermissionCheck.value() == SalesPermissionCheck.CheckType.DEPT) {
                checkPermissionDept(salesApRequest);
            }
        }
        return joinPoint.proceed();
    }

    /**
     * 判断登录人权限
     * 判断是否存在人客关系，存在：根据负责经销商查询数据；不存在：判断是否是销售部人员。是：根据部门查询；不是：返回全部数据
     */
    private void checkPermission(SalesApRequest salesApRequest) {
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser == null || loginUser.getSysUser() == null) {
            throw new CheckedException(ECode.QC_MARKET_EMP_ERROR);
        }
        
        // 安全获取 ancestors 和 roles
        String ancestors = "";
        if (loginUser.getSysUser().getDept() != null) {
            ancestors = loginUser.getSysUser().getDept().getAncestors();
            if (ancestors == null) {
                ancestors = "";
            }
        }
        
        List<SysRole> roles = loginUser.getSysUser().getRoles();
        if (roles == null) {
            roles = new ArrayList<>();
        }

        // 不是区域销售部的人员，可以查看所有
        if (!ancestors.contains(constantValue.deptYX) && roles.stream().filter(role -> RoleConstants.OPERATION_CENTER_AP_MANAGER.equals(role.getRoleKey())).findFirst().isPresent()) {
            return;
        }

        // 是区域销售部的人员：1、销售查看人客关系；2、大区TM查看当前部门下所有数据
        // 判断是否是大区TM
        String empNo = loginUser.getUsername();
        Optional<SysRole> first = roles.stream().filter(role -> RoleConstants.DISTRICT_AP_MANAGER.equals(role.getRoleKey())).findFirst();
        if (first.isPresent()) {
            ArrayList<String> deptNames = new ArrayList<>();
            deptNames.add(loginUser.getSysUser().getDept().getDeptNameSale());
            if ("000889".equals(empNo)){
                deptNames.add("北京特区");
            }
            return;
        }
        // 不是大区TM，判断是否是销售人员(人客关系)
        List<String> dealerCodes = qinceMarketEmployeeService.checkPermission(empNo);
        if (dealerCodes.isEmpty()) {
            throw new CheckedException(ECode.QC_MARKET_EMP_ERROR);
        }
//        salesApRequest.setNickName(loginUser.getNickName());
        salesApRequest.setDealerCodes(dealerCodes);
    }


    private void checkPermissionDept(SalesApRequest salesApRequest) {
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser == null || loginUser.getSysUser() == null) {
            return;
        }
        
        // 安全获取 ancestors 和 roles
        String ancestors = "";
        if (loginUser.getSysUser().getDept() != null) {
            ancestors = loginUser.getSysUser().getDept().getAncestors();
            if (ancestors == null) {
                ancestors = "";
            }
        }
        
        List<SysRole> roles = loginUser.getSysUser().getRoles();
        if (roles == null) {
            roles = new ArrayList<>();
        }

        // 不是区域销售部的人员，可以查看所有
        if (!ancestors.contains(constantValue.deptYX) && roles.stream().filter(role -> RoleConstants.OPERATION_CENTER_AP_MANAGER.equals(role.getRoleKey())).findFirst().isPresent()) {
            return;
        }

        // 是区域销售部的人员按部门查询
        String empNo = loginUser.getUsername();
        if (loginUser.getSysUser().getDept() == null) {
            return;
        }
        
        String deptName = loginUser.getSysUser().getDept().getDeptNameSale();
        if (deptName == null) {
            deptName = "";
        }
        
        salesApRequest.setDeptNames("000889".equals(empNo) ? Arrays.asList(deptName, "北京特区") : Arrays.asList(deptName));
    }
}
