package com.sfa.operation.util.excel;

import com.baomidou.mybatisplus.annotation.FieldStrategy;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.annotation.TableField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * 实体更新字段校验工具
 */
public class EntityUpdateCheckUtil {
    private static final Logger log = LoggerFactory.getLogger(EntityUpdateCheckUtil.class);

    /**
     * 校验实体是否有「非主键的有效更新字段」
     * @param entity 待校验实体
     * @return true=有有效字段，false=无有效字段
     */
    public static <T> boolean hasValidUpdateField(T entity) {
        if (entity == null) {
            return false;
        }

        Class<?> clazz = entity.getClass();
        Field[] fields = clazz.getDeclaredFields();
        boolean hasUpdateField = false;

        for (Field field : fields) {
            field.setAccessible(true);
            String fieldName = field.getName();

            try {
                Object fieldValue = field.get(entity);
                // 跳过主键字段（带@TableId注解）
                if (field.isAnnotationPresent(TableId.class)) {
                    continue;
                }
                // 跳过不参与更新的字段（如@TableField(updateStrategy = FieldStrategy.NEVER)）
                TableField tableField = field.getAnnotation(TableField.class);
                if (tableField != null && tableField.updateStrategy().equals(FieldStrategy.NEVER)) {
                    continue;
                }
                // 存在非空的更新字段
                if (fieldValue != null) {
                    hasUpdateField = true;
                    break;
                }
            } catch (IllegalAccessException e) {
                log.error("反射校验实体字段失败：fieldName={}", fieldName, e);
            }
        }
        return hasUpdateField;
    }

    public static <T> List<List<T>> splitList(List<T> list, int batchSize) {
        List<List<T>> batches = new ArrayList<>();
        for (int i = 0; i < list.size(); i += batchSize) {
            int end = Math.min(i + batchSize, list.size());
            batches.add(list.subList(i, end));
        }
        return batches;
    }
}