package com.sfa.job.service.order.impl;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.sfa.common.core.enums.ECode;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.DateUtils;
import com.sfa.job.domain.order.dao.IOrdersSentDao;
import com.sfa.job.enums.KyeEnum;
import com.sfa.job.pojo.response.OrdersSentDto;
import com.sfa.job.service.order.IOrdersSentQueryService;
import com.sfa.job.util.KyeUtil;
import com.sfa.job.util.QinCeUtils;
import com.sfa.job.util.aliyun.OssUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author : liqiulin
 * @date : 2025-07-10 16
 * @describe :
 */
@Slf4j
@Service
public class OrdersSentQueryServiceImpl implements IOrdersSentQueryService {

    @Autowired
    private IOrdersSentDao ordersSentDao;
    @Autowired
    private QinCeUtils qinCeUtils;
    @Autowired
    private KyeUtil kyeUtil;
    @Autowired
    private OssUtil ossUtil;

    @Override
    public void ordersSentToQince() {
        // 查询所有未push到勤策的发货单
        List<OrdersSentDto> sents = ordersSentDao.findByPushqcStatus();
        log.info("查询未推送到勤策的发货单：{}", sents.size());
        for (OrdersSentDto sent : sents) {
            // 修改勤策物流https地址
            pushQc(sent);
        }
    }

    @Override
    public OrdersSentDto getSent(String sentNo) {
        return ordersSentDao.getSent(sentNo);
    }

    @Override
    public void getOrderSentReceipt() {
        List<OrdersSentDto> sents = ordersSentDao.getNotReceiptSent();
        if (CollectionUtils.isEmpty(sents)) {
            log.info("今日暂无，需回单单据");
            return;
        }
        // 按物流公司分类
        Map<String, List<OrdersSentDto>> transportMap = sents.stream().collect(Collectors.groupingBy(OrdersSentDto::getTransport));
        getOrderSentReceiptBy109(transportMap.get("109"));
    }

    private void getOrderSentReceiptBy109(List<OrdersSentDto> ordersSentDtos) {
        if (CollectionUtils.isEmpty(ordersSentDtos)) {
            log.info("今日[109-跨越]暂无，需回单单据");
            return;
        }
        // 填写Object完整路径（不包含Bucket名称），例：[path]/[文件名.扩展名]
        String ossON = "wuliu/" + DateUtils.getYearMonth() + "/跨越/";
        // 将ordersSentDtos按长度分割，每10个分割
        List<List<OrdersSentDto>> split = ordersSentDtos.stream().collect(Collectors.groupingBy(it -> ordersSentDtos.indexOf(it) / 10)).values().stream().collect(Collectors.toList());
        for (List<OrdersSentDto> os : split) {
            List<String> exNos = os.stream().map(it -> it.getExpressNo()).collect(Collectors.toList());
            JSONArray waybillBaseInfo = kyeUtil.getWaybillBaseInfo(exNos);
            if (CollectionUtils.isEmpty(waybillBaseInfo)){
                continue;
            }

            for (Object o : waybillBaseInfo) {
                disReceipt((JSONObject) o,ossON);
            }
             // todo
            break;
        }

    }

    private void disReceipt(JSONObject infoJson,String ossON){
        // 判断物流单是否已签收，未签收数据不处理
        Date operateEndTime = infoJson.getDate("operateEndTime");
        if (operateEndTime == null){
            return;
        }
        String expressNo = infoJson.getString("waybillNumber");
        String receiptFlag = infoJson.getString("receiptFlag");
        OrdersSentDto ordersSentDto = new OrdersSentDto();
        ordersSentDto.setExpressNo(expressNo);
        ordersSentDto.setOperateEndDatetime(operateEndTime);
        ordersSentDto.setReceiptFlag(KyeEnum.KyeReceiptEnum.getDesc(receiptFlag));

        /**
         * 回单类型为回单原件、回单照片时查询照片，否则跳过
         */
        if (!KyeEnum.KyeReceiptEnum.hasPhoto(receiptFlag)){
            ordersSentDao.updateByExpressNo(ordersSentDto);
            return;
        }

        JSONArray objects = kyeUtil.queryWaybillPicture(expressNo);
        if (CollectionUtils.isEmpty(objects)){
            ordersSentDao.updateByExpressNo(ordersSentDto);
            return;
        }
        List<String> urls = new ArrayList<>();
        for (Object object : objects) {
            JSONObject jsonObject = (JSONObject) object;
            String url = jsonObject.getString("url");
            String originalName = jsonObject.getString("originalName");
            String ossUrl = ossUtil.uploadPhotoByte(url, ossON + originalName);
            urls.add(ossUrl);
        }
        ordersSentDto.setReceiptPhoto(JSONObject.toJSONString(urls));
        ordersSentDao.updateByExpressNo(ordersSentDto);
    }

    private void pushQc(OrdersSentDto sent){
        try {
            log.info("start push qc sent no [{}] 物流轨迹", sent.getAhSentNo());
            String htmlPath = "";
            String transport = sent.getTransport();
            switch (transport) {
                case "134":
                case "109":
                    htmlPath = qinCeUtils.wlHtmlPath + sent.getAhSentNo();
                    break;
                case "132":
                    htmlPath = qinCeUtils.wlRrsPath + sent.getExpressNo();
                    break;
                case "104":
                    htmlPath = qinCeUtils.wlWlyPath + sent.getDdNo();
                    sent.setTransportName("丰诺");
                    break;
                default:
                    throw new ServiceException(ECode.SENT_ISNULL_ERROR);
            }

            // 勤策中发货单按安徽匹配
            Map<String, Object> params = qinCeUtils.modifySentDefinedValParams(sent.getAhSentNo(), htmlPath,sent.getTransportName());
            String url = qinCeUtils.builderUrl(QinCeUtils.MODIFY_SENT_DEFINED_VAL, params);
            qinCeUtils.postQC(url, params);
            // 推送成功
            ordersSentDao.updatePushqcByAhSentNo(sent.getAhSentNo());
            log.info("end push qc sent no [{}] 物流轨迹", sent.getAhSentNo());
        }catch (Exception e) {
            log.error("勤策推送物流地址失败，物流信息：{}", JSONObject.toJSONString(sent));
        }
    }

}
