package com.sfa.job.domain.qince.dao;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.sfa.job.domain.qince.entity.QinceUserStatistic;
import com.sfa.job.domain.qince.mapper.QinceUserStatisticMapper;
import com.sfa.job.pojo.qince.response.QinceUserStatisticDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 勤策考勤用户统计 DAO 实现类（纯 DTO 对外，内部转换 Entity 调用 Mapper）
 */
@Slf4j
@Service
@DS("bi")
public class QinceUserStatisticDaoImpl implements IQinceUserStatisticDao {

    @Autowired
    private QinceUserStatisticMapper qinceUserStatisticMapper;

    // 配置项
    private static final int BATCH_SIZE = 200;
    // 线程安全的日期格式化器（适配 Date 类型，避免多线程问题）
    private static final ThreadLocal<SimpleDateFormat> DATE_SDF = ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd"));

    // ===================== 内部工具：DTO ↔ Entity 转换（隔离对外接口与持久化层） =====================
    /**
     * DTO 转换为 Entity（用于调用 Mapper 进行持久化）
     */
    private QinceUserStatistic dtoToEntity(QinceUserStatisticDTO dto) {
        if (dto == null) {
            return null;
        }
        QinceUserStatistic entity = new QinceUserStatistic();
        // BeanUtils 拷贝同名同类型字段（Date/Long/String 均兼容）
        BeanUtils.copyProperties(dto, entity);
        return entity;
    }

    /**
     * Entity 转换为 DTO（用于隐藏 Entity，对外返回纯净 DTO）
     */
    private QinceUserStatisticDTO entityToDto(QinceUserStatistic entity) {
        if (entity == null) {
            return null;
        }
        QinceUserStatisticDTO dto = new QinceUserStatisticDTO();
        // BeanUtils 拷贝同名同类型字段
        BeanUtils.copyProperties(entity, dto);
        return dto;
    }

    /**
     * DTO 列表 转换为 Entity 列表（批量操作适配）
     */
    private List<QinceUserStatistic> dtoListToEntityList(List<QinceUserStatisticDTO> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return new ArrayList<>(0);
        }
        return dtoList.stream()
                .map(this::dtoToEntity)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * Entity 列表 转换为 DTO 列表（批量操作适配）
     */
    private List<QinceUserStatisticDTO> entityListToDtoList(List<QinceUserStatistic> entityList) {
        if (CollectionUtils.isEmpty(entityList)) {
            return new ArrayList<>(0);
        }
        return entityList.stream()
                .map(this::entityToDto)
                .collect(Collectors.toList());
    }

    // ===================== DAO 接口实现：纯 DTO 对外，内部转换调用 Mapper =====================
    @Override
    public Map<String, QinceUserStatisticDTO> buildExistAttendanceMap(List<Long> userIdList, List<LocalDate> dateList) {
        if (CollectionUtils.isEmpty(userIdList) || CollectionUtils.isEmpty(dateList)) {
            return new HashMap<>(0);
        }

        // 1. 构建 Mapper 查询条件（仅操作 Entity，按需查询字段提升性能）
        LambdaQueryWrapper<QinceUserStatistic> queryWrapper = new LambdaQueryWrapper<QinceUserStatistic>()
                .in(QinceUserStatistic::getQcUserId, userIdList)
                .in(QinceUserStatistic::getAttDate, dateList)
                .select(QinceUserStatistic::getId, QinceUserStatistic::getQcUserId,
                        QinceUserStatistic::getAttDate);

        // 2. Mapper 操作 Entity，查询数据库（唯一直接操作 Entity 的地方）
        List<QinceUserStatistic> existEntityList = qinceUserStatisticMapper.selectList(queryWrapper);

        // 3. 转换 Entity 为 DTO，构建唯一键映射（对外隐藏 Entity，返回纯 DTO）
        Map<String, QinceUserStatisticDTO> existDtoMap = new HashMap<>(existEntityList.size());
        for (QinceUserStatistic entity : existEntityList) {
            QinceUserStatisticDTO dto = entityToDto(entity);
            String dateStr = DATE_SDF.get().format(entity.getAttDate());
            String uniqueKey = entity.getQcUserId() + "_" + dateStr;
            existDtoMap.put(uniqueKey, dto);
        }

        return existDtoMap;
    }

    @Override
    public int batchInsert(List<QinceUserStatisticDTO> dtoList) {
        int count = 0;
        if (CollectionUtils.isEmpty(dtoList)) {
            return count;
        }

        // 1. 分批处理，避免 SQL 语句过长（保护数据库，防止超出最大允许长度）
        for (int i = 0; i < dtoList.size(); i += BATCH_SIZE) {
            int end = Math.min(i + BATCH_SIZE, dtoList.size());
            List<QinceUserStatisticDTO> batchDtoList = dtoList.subList(i, end);

            // 2. 内部转换：DTO 列表 → Entity 列表（仅 Mapper 能处理 Entity）
            List<QinceUserStatistic> batchEntityList = dtoListToEntityList(batchDtoList);

            // 3. Mapper 执行批量插入（当前为循环单条，可优化为 MyBatis 批量 SQL）
            for (QinceUserStatistic entity : batchEntityList) {
                qinceUserStatisticMapper.insert(entity);
                count++;
            }
        }

        log.info("DAO层批量新增{}条考勤记录（DTO → Entity 转换后持久化）", count);
        return count;
    }

    @Override
    public int batchUpdate(List<QinceUserStatisticDTO> dtoList) {
        int count = 0;
        if (CollectionUtils.isEmpty(dtoList)) {
            return count;
        }

        // 分批处理，提升执行效率
        for (int i = 0; i < dtoList.size(); i += BATCH_SIZE) {
            int end = Math.min(i + BATCH_SIZE, dtoList.size());
            List<QinceUserStatisticDTO> batchDtoList = dtoList.subList(i, end);

            // 内部转换：DTO 列表 → Entity 列表（需包含主键 ID，否则更新失败）
            List<QinceUserStatistic> batchEntityList = dtoListToEntityList(batchDtoList);

            // Mapper 执行批量更新（基于主键 ID，当前为循环单条，可优化为 MyBatis 批量 SQL）
            for (QinceUserStatistic entity : batchEntityList) {
                if (entity.getId() != null) {
                    qinceUserStatisticMapper.updateById(entity);
                    count++;
                } else {
                    log.warn("考勤记录无主键 ID，跳过更新：qcUserId={}", entity.getQcUserId());
                }
            }
        }

        log.info("DAO层批量更新{}条考勤记录（DTO → Entity 转换后持久化）", count);
        return count;
    }
}