package com.sfa.job.util;

import cn.hutool.core.date.DatePattern;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.sfa.common.core.enums.ECode;
import com.sfa.common.core.exception.ServiceException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.time.LocalDateTime;
import java.time.OffsetTime;
import java.util.*;

/**
 * @author : liqiulin
 * @date : 2025-07-08 16
 * @describe :
 */
@Slf4j
@Component
public class JdtcUtil {
    /**
     * =================== JDTC API - config ===================
     */
    @Value("${jdtc.url}")
    private String jdTCUrl;
    @Value("${jdtc.accesstoken}")
    private String jdTCToken;
    @Value("${jdtc.wl_appkey}")
    private String wlAppkey;
    @Value("${jdtc.wl_appsecret}")
    private String wlAppSecret;
    @Value("${jdtc.pin}")
    private String pin;
    @Value("${jdtc.customer_code}")
    private String customerCode;

    /**
     * =================== JDTC API - path ===================
     */
    private final String ORDER_TRACE_QUERY_PATH = "/TransferCenterService/order/trace/query/v1";
    private final String ORDER_QUERY_PATH = "/TransferCenterService/order/query/v1";
    private final String DOMAIN = "TransferCenterService";
    private final String HEX_CHARACTERS = "0123456789ABCDEF";
    private final String ALGORITHM = "md5-salt";
    public JSONArray getOrderTrace(String orderNo) {
        try {
            String body = "[{\"pin\":\""+pin+"\",\"purchaseOrderNo\":\""+orderNo+"\"}]";
            Map<String, String> urlParams = getUrlParams(ORDER_TRACE_QUERY_PATH, body);
            String req = HttpUtil.createPost(jdTCUrl + ORDER_TRACE_QUERY_PATH + "?" + httpBuildQuery(urlParams)).addHeaders(getHeaders()).body(body).execute().body();

            JSONObject reqJson = JSONObject.parseObject(req);
            if (!reqJson.getString("code").equals("1000")){
                log.error("请求京东TC 物流轨迹 接口返回异常：{}",reqJson);
                throw new ServiceException(ECode.JINGDONG_TC_ORDER_TRACE_API_ERROR);
            }
            JSONObject data = reqJson.getJSONObject("data");
            return data.containsKey("purchaseTraceList") ? data.getJSONArray("purchaseTraceList") : new JSONArray();
        } catch (GeneralSecurityException | UnsupportedEncodingException e) {
            throw new ServiceException(ECode.JINGDONG_TC_ORDER_TRACE_QUERY_ERROR);
        }
    }
    public JSONArray orderQuery(List<String> waybillCodes) throws Exception{
        try {
            Map<String,Object> params = new HashMap<>();
            params.put("customerCode", customerCode);
            params.put("pin", pin);
            params.put("waybillCodes", waybillCodes);
            String body = JSONObject.toJSONString(Arrays.asList(params));

            Map<String, String> urlParams = getUrlParams(ORDER_QUERY_PATH, body);
            String req = HttpUtil.createPost(jdTCUrl + ORDER_QUERY_PATH + "?" + httpBuildQuery(urlParams)).addHeaders(getHeaders()).body(body).execute().body();

            JSONObject reqJson = JSONObject.parseObject(req);
            if (!reqJson.getString("code").equals("1000")){
                log.error("请求京东TC 到仓订单查询 接口返回异常：{}",reqJson);
                throw new ServiceException(ECode.JINGDONG_TC_ORDER_QUERY_API_ERROR);
            }
            JSONObject data = reqJson.getJSONObject("data");
            return data.containsKey("details") ? data.getJSONArray("details") : new JSONArray();
        } catch (GeneralSecurityException | UnsupportedEncodingException e) {
            throw new ServiceException(ECode.JINGDONG_TC_ORDER_QUERY_API_ERROR);
        }
    }

    private String sign(String algorithm, byte[] data, byte[] secret) throws GeneralSecurityException {
        if (Objects.equals(algorithm, "md5-salt")) {
            return bytesToHex(MessageDigest.getInstance("md5").digest(data));
        } else if (Objects.equals(algorithm, "HMacMD5")) {
            Mac mac = Mac.getInstance(algorithm);
            mac.init(new SecretKeySpec(secret, algorithm));
            return Base64.getEncoder().encodeToString(mac.doFinal(data));
        } else if (Objects.equals(algorithm, "HMacSHA1")) {
            Mac mac = Mac.getInstance(algorithm);
            mac.init(new SecretKeySpec(secret, algorithm));
            return Base64.getEncoder().encodeToString(mac.doFinal(data));
        } else if (Objects.equals(algorithm, "HMacSHA256")) {
            Mac mac = Mac.getInstance(algorithm);
            mac.init(new SecretKeySpec(secret, algorithm));
            return Base64.getEncoder().encodeToString(mac.doFinal(data));
        } else if (Objects.equals(algorithm, "HMacSHA512")) {
            Mac mac = Mac.getInstance(algorithm);
            mac.init(new SecretKeySpec(secret, algorithm));
            return Base64.getEncoder().encodeToString(mac.doFinal(data));
        }
        throw new GeneralSecurityException("Algorithm " + algorithm + " not supported yet");
    }

    private String bytesToHex(byte[] bytes) {
        StringBuilder stringBuilder = new StringBuilder(bytes.length * 2);
        for (byte b : bytes) {
            stringBuilder.append(HEX_CHARACTERS.charAt((b >>> 4) & 0x0F));
            stringBuilder.append(HEX_CHARACTERS.charAt(b & 0x0F));
        }
        return stringBuilder.toString();
    }

    private String httpBuildQuery(Map<String, String> query) throws UnsupportedEncodingException {
        StringBuilder stringBuilder = new StringBuilder();
        boolean first = true;
        for (Map.Entry<String, String> entry : query.entrySet()) {
            if (!first) {
                stringBuilder.append("&");
            } else {
                first = false;
            }
            stringBuilder.append(entry.getKey()).append("=").append(URLEncoder.encode(entry.getValue(), StandardCharsets.UTF_8.name()));
        }
        return stringBuilder.toString();
    }

    private Map<String, String> getUrlParams(String method,String body) throws GeneralSecurityException{
        String timestamp = DatePattern.NORM_DATETIME_FORMATTER.format(LocalDateTime.now());
        String content = String.join("", new String[]{
                wlAppSecret,
                "access_token", jdTCToken,
                "app_key", wlAppkey,
                "method", method,
                "param_json", body,
                "timestamp", timestamp,
                "v", "2.0",
                wlAppSecret
        });
        Map<String, String> urlParams = new HashMap<>();
        urlParams.put("LOP-DN", DOMAIN);
        urlParams.put("access_token", jdTCToken);
        urlParams.put("app_key", wlAppkey);
        urlParams.put("timestamp", timestamp);
        urlParams.put("v", "2.0");
        urlParams.put("sign", sign(ALGORITHM, content.getBytes(StandardCharsets.UTF_8), wlAppSecret.getBytes(StandardCharsets.UTF_8)));
        urlParams.put("algorithm", ALGORITHM);
        return urlParams;
    }

    private Map<String, String> getHeaders(){
        Map<String, String> headers = new HashMap<>();
        // lop-tz代表时区，为接口调用当地的时区；删去后默认为东八区
        headers.put("lop-tz", String.valueOf(OffsetTime.now().getOffset().getTotalSeconds() / 3600));
        // 用于开放平台识别客户调用API方式，客户无需修改
        headers.put("User-Agent", "lop-http/java");
        headers.put("content-type", "application/json;charset=utf-8");
        return headers;
    }
}
