package com.sfa.job.service.order.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.URLUtil;
import cn.hutool.http.HttpStatus;
import cn.hutool.json.JSONUtil;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.qimencloud.api.DefaultQimenCloudClient;
import com.qimencloud.api.QimenCloudClient;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.DateUtils;
import com.sfa.job.config.WdtQimenConfig;
import com.sfa.job.domain.order.dao.*;
import com.sfa.job.domain.order.entity.*;
import com.sfa.job.pojo.order.request.WdtSalesTradeQueryQueryWithDetailVo;
import com.sfa.job.pojo.order.response.WdtQimenBodyDto;
import com.sfa.job.pojo.order.response.WdtQimenDataDto;
import com.sfa.job.pojo.order.response.WdtQimenResponseDto;
import com.sfa.job.pojo.order.response.WdtSalesTradeQueryQueryWithDetailDto;
import com.sfa.job.pojo.response.CollectOrderLogDto;
import com.sfa.job.pojo.response.SyncOrderDetailDto;
import com.sfa.job.service.order.WdtQimenOrderSyncService;
import com.sfa.job.util.WdtQimenUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;

/**
 * 旺店通订单Service业务层处理
 *
 * @author lvbencai
 * @date 2025年01月07日17:00:16
 */
@DS("bi")
@Slf4j
@Service
public class WdtQimenOrderSyncServiceImpl implements WdtQimenOrderSyncService {
    @Autowired
    WdtQimenOrderDetailDao orderDetailDao;
    @Autowired
    WdtQimenOrderDao orderDao;

    @Autowired
    FinianceBaseProductDao baseProductDao;
    @Autowired
    FinanceBaseZbjTypeDao baseZbjTypeDao;
    @Autowired
    CollectOrderLogDao orderLogInfoDao;
    @Autowired
    CollectErrorLogDao errorInfoDao;
    @Autowired
    IProductDao productDao;

    @Autowired
    WdtQimenConfig wdtQimenConfig;


    /**
     * 多线程调用此方法
     *
     * @param startTime
     * @param endTime
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public SyncOrderDetailDto syncWdtQimenOrder(Date startTime, Date endTime, Integer syncType, Long pageSize) {
        SyncOrderDetailDto detailDto = new SyncOrderDetailDto();
        String batchNo = syncType + DateUtils.dateTimeNow() + Thread.currentThread().getId();
        WdtSalesTradeQueryQueryWithDetailVo request = new WdtSalesTradeQueryQueryWithDetailVo();
        String method = request.getApiMethodName();
        List<WdtQimenOrder> orderResult = new ArrayList<>();
        try {
            detailDto.setBatchNo(batchNo);
            detailDto.setStartTime(startTime);
            detailDto.setEndTime(endTime);
            detailDto.setPageNo(-1);
            CollectOrderLogDto collectOrderLog = orderLogInfoDao.selectOrderSyncLatest(syncType, method);

            // 调用新方法设置时间范围
            detailDto.setupTimeRange( 60, collectOrderLog);
            startTime = detailDto.getStartTime();
            endTime = detailDto.getEndTime();

            // 调用查询旺店通接口api 获取最新日期前的2小时
            Date currentLatest = DateUtils.addHours(new Date(), -2);

            if (currentLatest.compareTo(startTime) < 0) {
                detailDto.setMessage(String.format("开始时间%s不能大于当前时间%s", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(currentLatest)));
                log.debug(String.format("开始时间%s不能大于当前时间%s", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(currentLatest)));
                return detailDto;
            }
            if (!detailDto.validateStartTime()) {
                return detailDto;
            }
            log.info("查询订单参数：开始时间{},结束时间{},当前页:{}", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime), detailDto.getPageNo());
            String serverUrl = wdtQimenConfig.getServerUrl();
            String appKey = wdtQimenConfig.getAppKey();
            String appSecret = wdtQimenConfig.getAppSecret();
            String wdtAppSecret = wdtQimenConfig.getWdtAppSecret();

            String wdtSalt = wdtAppSecret.split(":")[1];
            String wdtSecret = wdtAppSecret.split(":")[0];
            String wdtAppKey = wdtQimenConfig.getWdtAppKey();
            ;
            String targetAppSecret = wdtQimenConfig.getTargetAppSecret();
            ;
            String wdtSid = wdtQimenConfig.getWdtSid();
            ;
            Long pageNo = 1L;
            Long totalPage = 1L;
            Integer totalCount = 1;
            do {
                // 查询只到数据查询完毕
                QimenCloudClient client = new DefaultQimenCloudClient(serverUrl, appKey, appSecret);
                WdtSalesTradeQueryQueryWithDetailVo.Params params = new WdtSalesTradeQueryQueryWithDetailVo.Params();
                params.setStartTime(DateUtil.formatDateTime(startTime));
                params.setEndime(DateUtil.formatDateTime(endTime));
                // 旺店通的页码从0开始，奇门从1开始
                params.setTimeType(1);
                request.setParams(params);

                WdtSalesTradeQueryQueryWithDetailVo.Pager pager = new WdtSalesTradeQueryQueryWithDetailVo.Pager();
                pager.setPageNo(pageNo);
                pager.setPageSize(pageSize);
                request.setPager(pager);

                // 查询数据
                String dateTime = DateUtil.formatDateTime(new Date());
                request.setDatetime(dateTime);
                request.putOtherTextParam("wdt3_customer_id", wdtSid);
                request.setTargetAppKey(targetAppSecret);
                request.setWdtAppkey(wdtAppKey);
                request.setWdtSalt(wdtSalt);
                request.setWdtSign(WdtQimenUtil.getQimenCustomWdtSign(request, wdtSecret));

                WdtSalesTradeQueryQueryWithDetailDto response = client.execute(request);
                if(ObjectUtils.isNotEmpty(response) && response.getFlag().equals("failure")){
                    log.error("奇门加载数据异常："+response.getMessage()+response.getSubMessage()+ URLUtil.decode(response.getRequestUrl()));
                    throw new ServiceException("奇门加载数据异常："+response.getMessage()+response.getSubMessage());
                }
                // 转换返回的json字符串结果
                String body = response.getBody();
                WdtQimenBodyDto wdtQimenBodyDto = JSONUtil.toBean(body, WdtQimenBodyDto.class);
                WdtQimenResponseDto wdtQimenResponseDto = wdtQimenBodyDto.getResponse();
                WdtQimenDataDto wdtQimenDataDto = wdtQimenResponseDto.getData();

                totalCount = wdtQimenDataDto.getTotalCount();
                totalPage = totalCount / pager.getPageSize()+1;

                List<WdtQimenOrder> orderSubResult = wdtQimenDataDto.getOrder();
                if (ObjectUtils.isEmpty(orderSubResult)) {
                    log.warn("当前查询没有数据，奇门加载数据异常："+response.getMessage()+response.getSubMessage()+ URLUtil.decode(response.getRequestUrl()));
//                    throw new ServiceException("旺店通没有查询到订单数据");
                }
                orderResult.addAll(orderSubResult);
                pageNo++;
            } while (pageNo <= totalPage);


            // 基础数据 直播间分类数据+成本、规格、口味
            Map<String, FinanceBaseProduct> baseProductMap = baseProductDao.selectBaseProduct();

            Map<String, String> baseZbjType = baseZbjTypeDao.selectBaseZbjType();
            // 系列
            Map<String, String> prodSeriesMap = productDao.selectProdSeries();

            // 入库订单表
            log.info("开始插入订单数据，数量：{}", orderResult.size());
            Date finalStartTime = startTime;
            Date finalEndTime = endTime;
            orderResult.forEach(order -> {
                order.setDelFlag(0);
                order.setBatchNo(batchNo);
                order.setStartTime(finalStartTime);
                order.setEndTime(finalEndTime);
                order.setSyncType(syncType);
                order.setCreateTime(new Date());
                order.setUpdateTime(new Date());
                order.setCreatedDate( new Date(Long.valueOf(order.getCreated())));
            });
            orderDao.saveOrUpdateBatch(orderResult);


            List<WdtQimenOrderDetail> mergeList = new ArrayList<>();
            // 用于同一个订单号，来赋值直播间信息
            Map<String, WdtQimenOrderDetail> orderZbjMap = new HashMap<>();
            // 入库订单明细表
            for (WdtQimenOrder order : orderResult) {
                List<WdtQimenOrderDetail> orderDetails = order.getDetailList();
                for (WdtQimenOrderDetail orderDetail : orderDetails) {
                    orderDetail.setSeries(prodSeriesMap.get(orderDetail.getGoodsNo()));
                    orderDetail.setTradeStatus(order.getTradeStatus());
                    orderDetail.setShopNo(order.getShopNo());
                    orderDetail.setShopName(order.getShopName());
                    orderDetail.setShopRemark(order.getShopRemark());
                    orderDetail.setReceivable(order.getReceivable());
                    orderDetail.setReceiverArea(order.getReceiverArea());
                    orderDetail.setConsignTime(order.getConsignTime());
                    orderDetail.setTradeTime(order.getTradeTime());
                    orderDetail.setBatchNo(batchNo);
                    orderDetail.setTradeNo(order.getTradeNo());
                    if(ObjectUtils.isEmpty(orderDetail.getPayTime())){
                        orderDetail.setPayTime(order.getPayTime());
                    }
                    // 计算分销信息
                    orderDetail.setFenxiaoNick(order.getFenxiaoNick());
                    orderDetail.setFenxiaoId(orderDetail.getFenxiaoId(order, orderDetail.getTradeId()));
                    orderDetail.setFenxiaoName(orderDetail.getFenxiaoName(order, orderDetail.getTradeId()));
                    orderDetail.setStartTime(finalStartTime);
                    orderDetail.setEndTime(finalEndTime);
                    orderDetail.setSyncType(syncType);
                    // 如果存在相同的
                    WdtQimenOrderDetail sameDetail = orderZbjMap.get(orderDetail.getSrcTid());

                    orderDetail.setZbjName(orderDetail.getFenxiaoName(), orderDetail.getRemark(), sameDetail);
                    orderDetail.setZbjId(orderDetail.getRemark(), sameDetail);
                    orderDetail.setZbjZbId(orderDetail.getFenxiaoId(), orderDetail.getRemark(), sameDetail);
                    orderDetail.setZbjSaleType(orderDetail.getRemark(), sameDetail);
                    // 确定分销商类型
                    orderDetail.setZbjQdType(orderDetail.getFenxiaoId(), orderDetail.getZbjZbId(), baseZbjType);
                    // 同一个订单id的，设置直播间类型信息也相同
                    if (ObjectUtils.isNotEmpty(orderDetail.getZbjName())) {
                        orderZbjMap.put(orderDetail.getSrcTid(), orderDetail);
                    }
                    orderDetail.setFlavorErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getFlavor() : "");
                    orderDetail.setSpecNameErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getSpec() : "");
                    // 将退款成功的数据分摊后总价置为0 2025年02月06日16:25:30
                    if (ObjectUtils.isNotEmpty(orderDetail.getRefundStatus()) && orderDetail.getRefundStatus().equals(5)) {
                        orderDetail.setShareAmount(new BigDecimal(0));
                        orderDetail.setActualCost(new BigDecimal(0));
                        orderDetail.setStandardCost(new BigDecimal(0));
                        orderDetail.setSumActualCost(new BigDecimal(0));
                        orderDetail.setSumStandardCost(new BigDecimal(0));
                    } else {
                        orderDetail.setActualCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                baseProductMap.get(orderDetail.getGoodsNo()).getActualCost() :
                                new BigDecimal(0)
                        );
                        orderDetail.setStandardCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                baseProductMap.get(orderDetail.getGoodsNo()).getStandardCost() :
                                new BigDecimal(0)
                        );
                        orderDetail.setSumActualCost(orderDetail.getActualCost().multiply(orderDetail.getNum()));
                        orderDetail.setSumStandardCost(orderDetail.getStandardCost().multiply(orderDetail.getNum()));

                    }
                }
                mergeList.addAll(orderDetails);
            }
            log.info("开始插入订单详情数据，数量：{}", mergeList.size());

            // 批量插入
            orderDetailDao.saveOrUpdateBatch(mergeList);

            String remark = String.format("订单数量:%s,详情数量：%s", orderResult.size(), mergeList.size());
            log.info("插入订单和订单详情完成，批次{}开始时间{},结束时间{},订单数量:{},详情数量：{},下一次开始页数:{}", batchNo, DateUtil.formatDateTime(startTime),
                    DateUtil.formatDateTime(endTime), orderResult.size(), mergeList.size(), detailDto.getPageNo());
            detailDto.setDetailInfo(method, remark, syncType, orderResult.size(), mergeList.size(), totalCount, batchNo, endTime, startTime);
            orderLogInfoDao.insert(detailDto);
        } catch (Exception e) {
            detailDto.setCode(HttpStatus.HTTP_INTERNAL_ERROR);
            detailDto.setMessage(e.toString());
            log.error(e.getMessage(), e);
            // 记录错误日志表 collect_error_info
            errorInfoDao.insert(detailDto);
        }
        return detailDto;
    }


}
