package com.sfa.job.service.order.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HttpStatus;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONReader;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.DateUtils;
import com.sfa.job.domain.order.dao.*;
import com.sfa.job.domain.order.entity.*;
import com.sfa.job.domain.order.wq.FinanceOrderCostWq;
import com.sfa.job.pojo.request.WdtOrderQueryVO;
import com.sfa.job.pojo.request.WdtRefundQueryVO;
import com.sfa.job.pojo.response.*;
import com.sfa.job.service.order.IFinanceOrderSyncService;
import com.sfa.job.util.WangdiantongUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 旺店通订单Service业务层处理
 *
 * @author lvbencai
 * @date 2025年01月07日17:00:16
 */
@DS("bi")
@Slf4j
@Service
public class FinanceOrderSyncServiceImpl implements IFinanceOrderSyncService {
    @Autowired
    WangdiantongUtil wangdiantongUtil;

    @Autowired
    FinanceOrderDetailDao orderDetailDao;
    @Autowired
    FinanceOrderDao orderDao;
    @Autowired
    WdtRefundOrderDao wdtRefundOrderDao;
    @Autowired
    WdtRefundOrderDetailDao wdtRefundOrderDetailDao;

    @Autowired
    WdtSwapOrderDao wdtSwapOrderDao;
    @Autowired
    WdtSwapOrderDetailDao wdtSwapOrderDetailDao;
    @Autowired
    WdtRefundAmountDetailDao amountDetailDao;

    @Autowired
    FinianceBaseProductDao baseProductDao;
    @Autowired
    FinanceBaseZbjTypeDao baseZbjTypeDao;
    @Autowired
    CollectOrderLogDao orderLogInfoDao;
    @Autowired
    CollectErrorLogDao errorInfoDao;
    @Autowired
    IProductDao productDao;

    /**
     * 多线程调用此方法
     *
     * @param startTime
     * @param endTime
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public SyncOrderDetailDto syncWdtOrder(Date startTime, Date endTime, Integer syncType) {
        SyncOrderDetailDto detailDto = new SyncOrderDetailDto();
        String batchNo = syncType + DateUtils.dateTimeNow() + Thread.currentThread().getId();
        String method = "sales.TradeQuery.queryWithDetail";

        try {
            detailDto.setBatchNo(batchNo);
            detailDto.setStartTime(startTime);
            detailDto.setEndTime(endTime);
            detailDto.setPageNo(0);

            // 调用新方法设置时间范围
            setupTimeRange(detailDto, syncType,60, method);
            startTime = detailDto.getStartTime();
            endTime = detailDto.getEndTime();

            // 调用查询旺店通接口api 获取最新日期前的2小时
            Date currentLatest = DateUtils.addHours(new Date(), -2);

            if (currentLatest.compareTo(startTime) < 0) {
                detailDto.setMessage(String.format("开始时间%s不能大于当前时间%s", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(currentLatest)));
//                log.warn(String.format("开始时间%s不能大于当前时间%s", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(currentLatest)));
                return detailDto;
            }
            if (!validateStartTime(detailDto )) {
                return detailDto;
            }
//            log.info("查询订单参数：开始时间{},结束时间{},当前页:{}", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime), detailDto.getPageNo());

            WdtOrderQueryVO wdtOrderQueryVO = new WdtOrderQueryVO();
            wdtOrderQueryVO.setStart_time(com.alibaba.fastjson2.util.DateUtils.format(startTime));
            wdtOrderQueryVO.setEnd_time(com.alibaba.fastjson2.util.DateUtils.format(endTime));
            HashMap<String, Object> params = new HashMap<>();
            params.putAll(BeanUtil.beanToMap(wdtOrderQueryVO));

            // 旺店通入参 和其他入参分开传入   旺店通入参bean转map
            WangdiantongQueryDto wangdiantongQueryDto = wangdiantongUtil.queryWithDetail(params,detailDto.getPageNo(), method);
            JSONArray orderAllArray = wangdiantongQueryDto.getResult();
            if (ObjectUtils.isEmpty(orderAllArray)) {
                throw new ServiceException("旺店通没有查询到订单数据");
            }
            detailDto.setPageNo(wangdiantongQueryDto.getNextBeginPageNo());
            //转化成orderList
            List<FinanceOrder> orders = JSONArray.parseArray(orderAllArray.toString(), FinanceOrder.class, JSONReader.Feature.SupportSmartMatch);

            // 基础数据 直播间分类数据+成本、规格、口味
            Map<String, FinanceBaseProduct> baseProductMap = baseProductDao.selectBaseProduct();

            Map<String, String> baseZbjType = baseZbjTypeDao.selectBaseZbjType();
            // 系列
            Map<String, String> prodSeriesMap = productDao.selectProdSeries();

            // 入库订单表
            log.info("开始插入订单数据，数量：{}", orderAllArray.size());
            Date finalStartTime = startTime;
            Date finalEndTime = endTime;
            orders.forEach(order -> {
                order.setDelFlag(0);
                order.setBatchNo(batchNo);
                order.setStartTime(finalStartTime);
                order.setEndTime(finalEndTime);
                order.setSyncType(syncType);
                order.setCreateTime(new Date());
                order.setUpdateTime(new Date());
            });
            orderDao.saveOrUpdateBatch(orders);


            List<FinanceOrderDetail> mergeList = new ArrayList<>();
            // 用于同一个订单号，来赋值直播间信息
            Map<String, FinanceOrderDetail> orderZbjMap = new HashMap<>();
            // 入库订单明细表
            for (FinanceOrder order : orders) {
                List<FinanceOrderDetail> orderDetails = order.getDetailList();
                for (FinanceOrderDetail orderDetail : orderDetails) {
                    orderDetail.setSeries(prodSeriesMap.get(orderDetail.getGoodsNo()));
                    orderDetail.setTradeStatus(order.getTradeStatus());
                    orderDetail.setShopNo(order.getShopNo());
                    orderDetail.setShopName(order.getShopName());
                    orderDetail.setShopRemark(order.getShopRemark());
                    orderDetail.setReceivable(order.getReceivable());
                    orderDetail.setReceiverArea(order.getReceiverArea());
                    orderDetail.setConsignTime(order.getConsignTime());
                    orderDetail.setTradeTime(order.getTradeTime());
                    orderDetail.setBatchNo(batchNo);
                    orderDetail.setTradeNo(order.getTradeNo());
                    // 计算分销信息
                    orderDetail.setFenxiaoNick(orderDetail.getFenxiaoNick(order));
                    orderDetail.setFenxiaoId(orderDetail.getFenxiaoId(order, orderDetail.getTradeId()));
                    orderDetail.setFenxiaoName(orderDetail.getFenxiaoName(order, orderDetail.getTradeId()));
                    orderDetail.setStartTime(finalStartTime);
                    orderDetail.setEndTime(finalEndTime);
                    orderDetail.setSyncType(syncType);
                    // 如果存在相同的
                    FinanceOrderDetail sameDetail = orderZbjMap.get(orderDetail.getSrcTid());

                    orderDetail.setZbjName(orderDetail.getFenxiaoName(), orderDetail.getRemark(), sameDetail);
                    orderDetail.setZbjId(orderDetail.getRemark(), sameDetail);
                    orderDetail.setZbjZbId(orderDetail.getFenxiaoId(), orderDetail.getRemark(), sameDetail);
                    orderDetail.setZbjSaleType(orderDetail.getRemark(), sameDetail);
                    // 确定分销商类型
                    orderDetail.setZbjQdType(orderDetail.getFenxiaoId(), orderDetail.getZbjZbId(), baseZbjType);
                    // 同一个订单id的，设置直播间类型信息也相同
                    if (ObjectUtils.isNotEmpty(orderDetail.getZbjName())) {
                        orderZbjMap.put(orderDetail.getSrcTid(), orderDetail);
                    }
                    orderDetail.setFlavorErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getFlavor() : "");
                    orderDetail.setSpecNameErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getSpec() : "");
                    // 将退款成功的数据分摊后总价置为0 2025年02月06日16:25:30
                    if (ObjectUtils.isNotEmpty(orderDetail.getRefundStatus()) && orderDetail.getRefundStatus().equals(5)) {
                        orderDetail.setShareAmount(new BigDecimal(0));
                        orderDetail.setActualCost(new BigDecimal(0));
                        orderDetail.setStandardCost(new BigDecimal(0));
                        orderDetail.setSumActualCost(new BigDecimal(0));
                        orderDetail.setSumStandardCost(new BigDecimal(0));
                    } else {
                        orderDetail.setActualCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                baseProductMap.get(orderDetail.getGoodsNo()).getActualCost() :
                                new BigDecimal(0)
                        );
                        orderDetail.setStandardCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                baseProductMap.get(orderDetail.getGoodsNo()).getStandardCost() :
                                new BigDecimal(0)
                        );
                        orderDetail.setSumActualCost(orderDetail.getActualCost().multiply(orderDetail.getNum()));
                        orderDetail.setSumStandardCost(orderDetail.getStandardCost().multiply(orderDetail.getNum()));

                    }
                }
                mergeList.addAll(orderDetails);
            }
            log.info("开始插入订单详情数据，数量：{}", mergeList.size());

            // 批量插入
            orderDetailDao.saveOrUpdateBatch(mergeList);

            String remark = String.format("订单数量:%s,详情数量：%s", orders.size(), mergeList.size());
            log.info("插入订单和订单详情完成，批次{}开始时间{},结束时间{},订单数量:{},详情数量：{},下一次开始页数:{}", batchNo, DateUtil.formatDateTime(startTime),
                    DateUtil.formatDateTime(endTime), orders.size(), mergeList.size(), detailDto.getPageNo());

            detailDto.setDetailInfo(method, remark,syncType, orders.size(), mergeList.size(), wangdiantongQueryDto.getTotalCount(), batchNo, endTime, startTime);
            orderLogInfoDao.insert(detailDto);
        } catch (Exception e) {
            detailDto.setCode(HttpStatus.HTTP_INTERNAL_ERROR);
            detailDto.setMessage(e.toString());
            log.error(e.getMessage(), e);
            // 记录错误日志表 collect_error_info
            errorInfoDao.insert(detailDto);

        }
        return detailDto;
    }


    /**
     * 同步退换货数据
     *
     * @param startTime
     * @param endTime
     * @param syncType
     * @return
     */
    @Override
    public SyncOrderDetailDto syncWdtRefundOrder(Date startTime, Date endTime, Integer syncType) {
        SyncOrderDetailDto detailDto = new SyncOrderDetailDto();
        String batchNo = syncType + DateUtils.dateTimeNow() + Thread.currentThread().getId();
        String method = "aftersales.refund.Refund.search";

        try {
            detailDto.setBatchNo(batchNo);
            detailDto.setStartTime(startTime);
            detailDto.setEndTime(endTime);
            detailDto.setPageNo(0);

            // 调用新方法设置时间范围
            setupTimeRange(detailDto, syncType, 60*24*1, method);
            startTime = detailDto.getStartTime();
            endTime = detailDto.getEndTime();
            if (!validateStartTime(detailDto)) {
                return detailDto;
            }

            log.info("查询订单参数：开始时间{},结束时间{},当前页:{}", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime), detailDto.getPageNo());

            // 旺店通入参 和其他入参分开传入   旺店通入参bean转map
            WdtRefundQueryVO wdtOrderQueryVO = new WdtRefundQueryVO();
            wdtOrderQueryVO.setSettle_from(com.alibaba.fastjson2.util.DateUtils.format(startTime));
            wdtOrderQueryVO.setSettle_to(com.alibaba.fastjson2.util.DateUtils.format(endTime));
            HashMap<String, Object> params = new HashMap<>();
            Map<String, Object> stringObjectMap = BeanUtil.beanToMap(wdtOrderQueryVO);
            params.putAll(stringObjectMap);
            // 退换单查询
            WangdiantongQueryDto wangdiantongQueryDto = wangdiantongUtil.queryWithDetail(params, detailDto.getPageNo(), method);
            JSONArray orderAllArray = wangdiantongQueryDto.getResult();
            if (ObjectUtils.isEmpty(orderAllArray)) {
                throw new ServiceException("旺店通没有查询到订单数据");
            }
            detailDto.setPageNo(wangdiantongQueryDto.getNextBeginPageNo());

            // 保存数据
            List<WdtRefundOrder> refundOrders = JSONArray.parseArray(orderAllArray.toString(), WdtRefundOrder.class, JSONReader.Feature.SupportSmartMatch);
            wdtRefundOrderDao.saveOrUpdateBatch(refundOrders);

            List<WdtRefundOrderDetail> refundOrderDetails = new ArrayList<>();
            refundOrders.forEach(order -> {
                refundOrderDetails.addAll(order.getDetailList());
            });
            wdtRefundOrderDetailDao.saveOrUpdateBatch(refundOrderDetails);

            // 换货信息
            List<WdtSwapOrder> swapOrders = new ArrayList<>();
            List<WdtSwapOrderDetail> swapOrderDetail = new ArrayList<>();
            refundOrders.forEach(order -> {
                if (ObjectUtils.isNotEmpty(order.getSwapOrder())) {
                    swapOrders.add(order.getSwapOrder());
                    swapOrderDetail.addAll(order.getSwapOrder().getSwapOrderDetailList());
                }
            });
            wdtSwapOrderDao.saveOrUpdateBatch(swapOrders);
            wdtSwapOrderDetailDao.saveOrUpdateBatch(swapOrderDetail);
            // 金额明细
            List<WdtRefundAmountDetail> amountDetailList = new ArrayList<>();
            refundOrders.forEach(order -> {
                if (ObjectUtils.isNotEmpty(order.getAmountDetailList())) {
                    amountDetailList.addAll(order.getAmountDetailList());
                }
            });
            amountDetailDao.saveOrUpdateBatch(amountDetailList);
            log.info("插入订单和订单详情完成，批次{}开始时间{},结束时间{},退货订单数量:{},详情数量：{},下一次开始页数:{}", batchNo, DateUtil.formatDateTime(startTime),
                    DateUtil.formatDateTime(endTime), refundOrders.size(), refundOrderDetails.size(), detailDto.getPageNo());
            String remark = String.format("退货订单数量:%s,详情数量：%s,换货订单数量%s,换货详情数量%s,金额明细数量%s",
                    refundOrders.size(), refundOrderDetails.size(), swapOrders.size(), swapOrderDetail.size(), amountDetailList.size());
            detailDto.setDetailInfo(method, remark, syncType,refundOrders.size(), refundOrderDetails.size(),  wangdiantongQueryDto.getTotalCount(), batchNo, endTime, startTime);
            orderLogInfoDao.insert(detailDto);
        } catch (Exception e) {
            detailDto.setCode(HttpStatus.HTTP_INTERNAL_ERROR);
            detailDto.setMessage(e.toString());
            log.error(e.getMessage(), e);
            errorInfoDao.insert(detailDto);
        }
        return detailDto;
    }

    /**
     * 修改直播间类型
     *
     * @param startTime
     * @param endTime
     * @return
     */
    @Override
    public FinanceOrderDetailUpdateDto updateCost(Date startTime, Date endTime) {
        // 查询有问题的goods_no
        List<FinanceOrderDetailListDto> errorCostDetails = orderDetailDao.selectErrorCost(startTime, endTime);
        List<String > errorGoodsNos = new ArrayList<>();

        if(ObjectUtil.isNotEmpty(errorCostDetails)){
            List<String> goodNos = errorCostDetails.stream().map(FinanceOrderDetailListDto::getGoodsNo).collect(Collectors.toList());
            // 有问题的料号数据，查询产品基础信息表
            Map<String, FinanceBaseProduct> productMap = baseProductDao.selectBaseProductCost(goodNos);

            // 查询当前的成本数据
            for (int i = 0; i < goodNos.size(); i++) {
                String goodsNo = goodNos.get(i);
                FinanceBaseProduct financeBaseProduct = productMap.get(goodsNo);
                if(ObjectUtil.isNotEmpty(financeBaseProduct) && financeBaseProduct.getActualCost().compareTo(new BigDecimal("0"))>0){
                    // 循环修改数据
                    BigDecimal actualCost = financeBaseProduct.getActualCost();
                    BigDecimal standardCost = financeBaseProduct.getStandardCost();
                    FinanceOrderCostWq financeOrderCostWq = new FinanceOrderCostWq();
                    financeOrderCostWq.setGoodsNo(goodsNo);
                    financeOrderCostWq.setActualCost(actualCost);
                    financeOrderCostWq.setStandardCost(standardCost);
                    financeOrderCostWq.setStartTime(startTime);
                    financeOrderCostWq.setEndTime(endTime);
                    orderDetailDao.updateCost(financeOrderCostWq);
                }else{
                    log.error("料号{}没有查询到数据",goodsNo);
                    errorGoodsNos.add(goodsNo);
                }
            }
        }
        FinanceOrderDetailUpdateDto updateDto = new FinanceOrderDetailUpdateDto();
        updateDto.setErrorList(errorGoodsNos);
        updateDto.setTotalCount(errorCostDetails.size());
        updateDto.setErrorDetailList(errorCostDetails);
        return updateDto;
    }

    /**
     * 设置开始时间和结束时间
     * @param detailDto
     * @param syncType
     * @param interval
     * @param method
     */
    private void setupTimeRange(SyncOrderDetailDto detailDto, Integer syncType, Integer interval, String method) {
        Date latestTime = null;
        // startTime 为空，默认是定时任务调用
        if (ObjectUtils.isEmpty(detailDto.getStartTime())) {
            // 继续下一页的标识 pageNo =-1 标识终止分页 pageNo >0 标识继续分页
            boolean nextPageFlag = true;
            // 查询最新的采集时间
            CollectOrderLogDto collectOrderLog = orderLogInfoDao.selectOrderSyncLatest(syncType, method);
            if (ObjectUtils.isNotEmpty(collectOrderLog)) {
                latestTime = collectOrderLog.getLatestTime();
                Integer pageNoExist = collectOrderLog.getPageNo();

                detailDto.setPageNo(ObjectUtil.isNotEmpty(pageNoExist) && pageNoExist > 0 ? collectOrderLog.getPageNo() : 0);
                nextPageFlag = detailDto.getPageNo() > 0;

                if (nextPageFlag) {
                    detailDto.setStartTime(collectOrderLog.getStartTime());
                    detailDto.setEndTime(collectOrderLog.getEndTime());
                } else {
                    // 调用查询旺店通接口api 获取最新日期前的一个小时
                    detailDto.setStartTime(DateUtils.addMinutes(latestTime, -5));
                    detailDto.setEndTime(DateUtils.addMinutes(detailDto.getStartTime(), interval));
                }
            } else {
                // 默认上个月的第一天 00:00:00
                detailDto.setStartTime(DateUtil.beginOfDay(DateUtil.beginOfMonth(DateUtil.lastMonth())));
                detailDto.setEndTime(DateUtils.addMinutes(detailDto.getStartTime(), interval));
            }
        }
        latestTime = detailDto.getEndTime();
        detailDto.setLatestTime(latestTime);

    }

    private boolean validateStartTime(SyncOrderDetailDto detailDto) {
        Date currentLatest = DateUtils.addHours(new Date(), -2);
        if (currentLatest.compareTo(detailDto.getEndTime()) < 0) {
            detailDto.setTotalCount(0);
            String message = String.format("结束时间%s不能大于当前时间%s", DateUtil.formatDateTime(detailDto.getEndTime()), DateUtil.formatDateTime(currentLatest));
            detailDto.setMessage(message);
            log.warn(message);
            return false;
        }
        return true;
    }
}
