package com.sfa.job.service.qince.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.sfa.job.domain.qince.dao.IQinceUserStatisticDao;
import com.sfa.job.pojo.qince.response.QinceUserStatisticDTO;
import com.sfa.job.service.qince.IQinceUserStatisticService;
import com.sfa.job.util.QinCeUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 勤策考勤同步实现类（全新版：全程 Java8+ 日期 API，无 Date 类型，过滤employee_code>=11位的数据不入库）
 */
@Slf4j
@Service
@DS("promotion")
public class QinceUserStatisticServiceImpl implements IQinceUserStatisticService {

    @Autowired
    private QinCeUtils qinCeUtils;

    @Autowired
    private IQinceUserStatisticDao qinceUserStatisticDao;

    // 配置项
    private static final int API_MAX_SIZE = 10000;
    // 工号过滤阈值：>=该长度则过滤不入库
    private static final int EMPLOYEE_CODE_FILTER_MIN_LENGTH = 11;

    // Java8+ 全局日期格式化器（线程安全，无需 ThreadLocal 包装）
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final DateTimeFormatter DATETIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    // 系统默认时区（全局复用，避免多次创建）
    private static final ZoneId DEFAULT_ZONE = ZoneId.systemDefault();

    @Override
    public JSONObject getUserStatistic(String startDate, String endDate, Integer page, Integer size) throws Exception {
        int currentPage = 1;
        JSONArray allUserArray = new JSONArray();
        while (true) {
            JSONObject pageResult = qinCeUtils.getUserStatistic(startDate, endDate, currentPage, API_MAX_SIZE);

            JSONObject dataObj = null;
            // 先以Object类型获取，避免强制转换报错
            Object responseData = pageResult.get("response_data");

            if (responseData != null) {
                if (responseData instanceof JSONObject) {
                    dataObj = (JSONObject) responseData;
                } else if (responseData instanceof String) {
                    String responseDataStr = (String) responseData;
                    if (StringUtils.hasText(responseDataStr)) {
                        try {
                            // 先尝试解析为 JSONObject
                            dataObj = JSONObject.parseObject(responseDataStr);
                        } catch (Exception e) {
                            log.warn("response_data 字符串无法解析为 JSONObject，尝试解析为 JSONArray，字符串内容：{}", responseDataStr.substring(0, Math.min(responseDataStr.length(), 200)));
                            // 如果解析 JSONObject 失败，尝试解析为 JSONArray（若接口直接返回数组）
                            JSONArray tempArray = JSONArray.parseArray(responseDataStr);
                            // 封装为 JSONObject（保持后续逻辑一致，data 字段对应该数组）
                            dataObj = new JSONObject();
                            dataObj.put("data", tempArray);
                        }
                    }
                } else {
                    // 情况3：其他类型，直接封装为 JSONObject（避免后续空指针）
                    log.warn("response_data 为不支持的类型：{}", responseData.getClass().getName());
                    dataObj = new JSONObject();
                }
            }

            // 从 dataObj 中获取 data 数组
            JSONArray data = dataObj == null ? new JSONArray() : dataObj.getJSONArray("data");

            if (CollectionUtils.isEmpty(data)) {
                log.info("第{}页无数据，全量数据拉取完成，共{}个用户", currentPage, allUserArray.size());
                break;
            }
            allUserArray.addAll(data);
            log.debug("拉取第{}页，新增{}用户，累计{}用户", currentPage, data.size(), allUserArray.size());

            if (data.size() < API_MAX_SIZE) {
                log.info("第{}页数据不足{}条，判定为最后一页", currentPage, API_MAX_SIZE);
                break;
            }
            currentPage++;
        }

        // 构造统一返回结果（兼容上层调用，自定义 response_data 字段）
        JSONObject totalResult = new JSONObject();
        totalResult.put("response_data", allUserArray);
        totalResult.put("total", allUserArray.size());
        return totalResult;
    }

    /**
     * 同步【前一天】全量考勤数据 - 事务保证原子性（全程 LocalDate，无精度问题）
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int queryAndSaveYesterdayAttendance() {
        LocalDate yesterday = LocalDate.now().minusDays(1);
        log.info("当前调用-目标同步前一天自然日：{}", yesterday.format(DATE_FORMATTER));
        return syncAttendanceByTargetDate(yesterday);
    }

    /**
     * 同步【当天】全量考勤数据 - 事务保证原子性（全程 LocalDate，无精度问题）
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int queryAndSaveTodayAttendance() {
        LocalDate today = LocalDate.now();
        log.info("当前调用-目标同步当天自然日：{}", today.format(DATE_FORMATTER));
        return syncAttendanceByTargetDate(today);
    }

    /**
     * 核心同步方法：按指定日期同步，全程操作 DTO（全程 LocalDate/LocalDateTime，无 Date）
     */
    private int syncAttendanceByTargetDate(LocalDate targetLocalDate) {
        String targetDateStr = targetLocalDate.format(DATE_FORMATTER);
        log.info("===== 开始同步{}勤策考勤数据 =====", targetDateStr);

        try {
            JSONObject fullData = this.getUserStatistic(targetDateStr, targetDateStr, 1, API_MAX_SIZE);
            JSONArray allUserArray = fullData.getJSONArray("response_data");
            if (CollectionUtils.isEmpty(allUserArray)) {
                log.info("===== {}无任何用户考勤数据，同步结束 =====", targetDateStr);
                return 0;
            }

            List<QinceUserStatisticDTO> attendanceDtoList = convertToAttendanceDTO(allUserArray, targetLocalDate);
            if (CollectionUtils.isEmpty(attendanceDtoList)) {
                log.info("===== {}无有效考勤记录生成，同步结束 =====", targetDateStr);
                return 0;
            }
            log.info("{}考勤DTO转换完成，共生成{}条有效记录（已过滤工号>=11位数据）", targetDateStr, attendanceDtoList.size());

            // 步骤3：调用 DAO 层批量新增/更新（全程 DTO 交互，无 Entity 暴露）
            return batchSaveOrUpdate(attendanceDtoList);
        } catch (Exception e) {
            log.error("===== {}考勤数据同步失败 =====", targetDateStr, e);
            return 0;
        }
    }

    /**
     * 转换为 DTO：全程 Java8+ 日期类型，新增工号>=11位过滤逻辑，严格匹配 DTO 结构
     */
    private List<QinceUserStatisticDTO> convertToAttendanceDTO(JSONArray allUserArray, LocalDate targetLocalDate) {
        List<QinceUserStatisticDTO> resultList = new ArrayList<>();
        String targetDateStr = targetLocalDate.format(DATE_FORMATTER);
        LocalDateTime now = LocalDateTime.now();

        for (int i = 0; i < allUserArray.size(); i++) {
            try {
                JSONObject userJson = allUserArray.getJSONObject(i);
                // 核心校验1：userId 为空则跳过（唯一标识，无则无效）
                String userIdStr = getStringValue(userJson, "userId");
                if (!StringUtils.hasText(userIdStr)) {
                    log.warn("当前调用-第{}条数据被过滤：无userId", i + 1);
                    continue;
                }

                QinceUserStatisticDTO dto = new QinceUserStatisticDTO();
                // 基础字段赋值（严格匹配 DTO 字段名和类型）
                dto.setQcUserId(Long.valueOf(userIdStr));
                dto.setUserName(getStringValue(userJson, "userName"));
                dto.setDeptName(getStringValue(userJson, "deptName"));
                dto.setSex(StringUtils.hasText(getStringValue(userJson, "sex")) ? getStringValue(userJson, "sex") : null);
                dto.setGroup(getStringValue(userJson, "group"));

                // 工号赋值：直接赋值字符串，保留前置零
                String codeStr = getStringValue(userJson, "code");
                String employeeCode = StringUtils.hasText(codeStr) ? codeStr.trim() : null;
                dto.setEmployeeCode(employeeCode);

                // 工号不为空 且 长度>=11位 → 直接过滤，不加入结果集
                if (StringUtils.hasText(employeeCode) && employeeCode.length() >= EMPLOYEE_CODE_FILTER_MIN_LENGTH) {
                    log.warn("当前调用-第{}条数据被过滤：工号[{}]长度为{}位，>=11位不入库",
                            i + 1, employeeCode, employeeCode.length());
                    continue;
                }

                // 自动填充字段：isDelete 已在 DTO 中默认赋值为 1，创建/修改时间赋值 LocalDateTime
                dto.setCreateTime(now);
                dto.setModifyTime(now);

                // 初始化考勤核心字段默认值
                dto.setWorkTime(null);
                dto.setRemarks(null);

                // 解析 attendance 数组（接口保证单日单用户仅 1 条）
                JSONArray attendanceArray = userJson.getJSONArray("attendance");
                if (!CollectionUtils.isEmpty(attendanceArray)) {
                    JSONObject attJson = attendanceArray.getJSONObject(0);
                    // 解析 attDate 为 LocalDate（纯日期，用于对比和赋值）
                    LocalDate attLocalDate = parseStrToLocalDate(attJson, "attDate");
                    if (attLocalDate != null) {
                        // 核心校验3：纯日期对比（无时分秒干扰，精准匹配目标日期）
                        if (targetLocalDate.equals(attLocalDate)) {
                            // 填充考勤核心字段
                            dto.setWorkTime(parseStrToDouble(attJson, "workTime"));
                            dto.setRemarks(StringUtils.hasText(getStringValue(attJson, "remarks")) ? getStringValue(attJson, "remarks").trim() : null);

                            // 给 DTO 的 attDate 赋值（解决数据库 att_date 为 null 的核心）
                            dto.setAttDate(attLocalDate);

                            // 填充上班打卡信息（全程 LocalDateTime，匹配 DTO 字段）
                            if (attJson.containsKey("checkin") && !attJson.getJSONArray("checkin").isEmpty()) {
                                JSONObject checkin = attJson.getJSONArray("checkin").getJSONObject(0);
                                fillCheckInInfo(dto, checkin);
                            }

                            // 填充下班打卡信息（全程 LocalDateTime，匹配 DTO 字段）
                            if (attJson.containsKey("checkout") && !attJson.getJSONArray("checkout").isEmpty()) {
                                JSONObject checkout = attJson.getJSONArray("checkout").getJSONObject(0);
                                fillCheckOutInfo(dto, checkout);
                            }

                            // 符合所有条件的记录添加到结果集
                            resultList.add(dto);
                        } else {
                            log.warn("当前调用-第{}条数据被过滤：attDate自然日不匹配（目标：{}，实际：{}）",
                                    i + 1, targetDateStr, attLocalDate.format(DATE_FORMATTER));
                        }
                    } else {
                        log.warn("当前调用-第{}条数据被过滤：attDate解析失败或为空", i + 1);
                    }
                } else {
                    log.warn("当前调用-第{}条数据被过滤：无attendance考勤数组", i + 1);
                }
            } catch (Exception e) {
                log.error("当前调用-第{}条数据转换失败，跳过", i + 1, e);
                continue;
            }
        }

        log.info("当前调用-DTO转换完成，有效自然日考勤数据量：{}（已过滤工号>=11位数据）", resultList.size());
        return resultList;
    }

    /**
     * 填充上班打卡信息：全程 LocalDateTime，严格匹配 DTO 字段类型，无类型转换异常
     */
    private void fillCheckInInfo(QinceUserStatisticDTO dto, JSONObject checkin) {
        dto.setCheckInAttdLieLocate(getIntegerValue(checkin, "attdLieLocate"));
        dto.setCheckInAttdAddress(getStringValue(checkin, "attdAddress"));
        dto.setCheckInAttdLcError(getIntegerValue(checkin, "attdLcError"));
        dto.setCheckInAttdStatus(getIntegerValue(checkin, "attdStatus"));
        // 直接解析为 LocalDateTime，匹配 DTO 字段，无中间转换
        dto.setCheckInAttdTime(parseStrToLocalDateTime(checkin, "attdTime"));
    }

    /**
     * 填充下班打卡信息：全程 LocalDateTime，严格匹配 DTO 字段类型，无类型转换异常
     */
    private void fillCheckOutInfo(QinceUserStatisticDTO dto, JSONObject checkout) {
        dto.setCheckOutAttdLieLocate(getIntegerValue(checkout, "attdLieLocate"));
        dto.setCheckOutAttdAddress(getStringValue(checkout, "attdAddress"));
        dto.setCheckOutAttdLcError(getIntegerValue(checkout, "attdLcError"));
        dto.setCheckOutAttdStatus(getIntegerValue(checkout, "attdStatus"));
        // 直接解析为 LocalDateTime，匹配 DTO 字段，无中间转换
        dto.setCheckOutAttdTime(parseStrToLocalDateTime(checkout, "attdTime"));
    }

    /**
     * 批量新增/更新：全程 DTO 操作，Java8+ 日期类型，无 Date 依赖
     */
    private int batchSaveOrUpdate(List<QinceUserStatisticDTO> attendanceDtoList) {
        // 1. 调用 DAO 层，获取已存在的 DTO 映射（无 Entity 暴露）
        Map<String, QinceUserStatisticDTO> existDtoMap = buildExistAttendanceMap(attendanceDtoList);

        // 2. 拆分新增/更新列表
        List<QinceUserStatisticDTO> insertList = new ArrayList<>();
        List<QinceUserStatisticDTO> updateList = new ArrayList<>();
        LocalDateTime now = LocalDateTime.now(); // 当前时间，用于更新 modifyTime

        for (QinceUserStatisticDTO dto : attendanceDtoList) {
            LocalDate attLocalDate = dto.getAttDate();
            if (attLocalDate == null) {
                log.warn("当前记录attDate为空，跳过批量操作：qcUserId={}", dto.getQcUserId());
                continue;
            }
            String targetDateStr = attLocalDate.format(DATE_FORMATTER);
            String uniqueKey = dto.getQcUserId() + "_" + targetDateStr;

            if (existDtoMap.containsKey(uniqueKey)) {
                // 已存在：从 DTO 映射中获取数据，更新字段（全程操作 DTO）
                QinceUserStatisticDTO existDto = existDtoMap.get(uniqueKey);
                dto.setId(existDto.getId()); // 主键 ID 从 DTO 中获取
                dto.setCreateTime(existDto.getCreateTime()); // 保留原有创建时间
                dto.setModifyTime(now); // 更新修改时间
                updateList.add(dto);
            } else {
                // 不存在：直接新增（createTime/modifyTime 已赋值）
                insertList.add(dto);
            }
        }

        // 3. 调用 DAO 层执行批量操作（纯 DTO 入参，无需关心持久化细节）
        int insertCount = qinceUserStatisticDao.batchInsert(insertList);
        int updateCount = qinceUserStatisticDao.batchUpdate(updateList);
        int totalCount = insertCount + updateCount;

        log.info("批量操作完成：新增{}条，更新{}条，总计{}条（已过滤工号>=11位数据）", insertCount, updateCount, totalCount);
        return totalCount;
    }

    /**
     * 构建已存在记录的映射：调用 DAO 层方法，返回纯 DTO 映射
     */
    private Map<String, QinceUserStatisticDTO> buildExistAttendanceMap(List<QinceUserStatisticDTO> attendanceDtoList) {
        List<Long> userIdList = attendanceDtoList.stream()
                .map(QinceUserStatisticDTO::getQcUserId)
                .distinct()
                .collect(Collectors.toList());
        List<LocalDate> dateList = attendanceDtoList.stream()
                .map(QinceUserStatisticDTO::getAttDate)
                .distinct()
                .filter(java.util.Objects::nonNull)
                .collect(Collectors.toList());

        // 调用 DAO 层，获取纯 DTO 映射（DAO 层已同步修改为 LocalDate 入参）
        return qinceUserStatisticDao.buildExistAttendanceMap(userIdList, dateList);
    }

    // ===================== 基础类型转换工具方法（全程 Java8+ 日期 API，无 Date，空值兼容+异常捕获） =====================
    private String getStringValue(JSONObject json, String key) {
        try {
            String value = json.getString(key);
            return StringUtils.hasText(value) ? value.trim() : null;
        } catch (Exception e) {
            return null;
        }
    }

    private Integer getIntegerValue(JSONObject json, String key) {
        try {
            String value = json.getString(key);
            return StringUtils.hasText(value) ? Integer.parseInt(value.trim()) : null;
        } catch (Exception e) {
            log.warn("数字字符串解析失败，key={}, 异常：", key, e);
            return null;
        }
    }

    /**
     * 解析字符串为 LocalDate（纯日期，yyyy-MM-dd，适配 DTO 的 attDate 字段）
     */
    private LocalDate parseStrToLocalDate(JSONObject json, String key) {
        try {
            String value = getStringValue(json, key);
            if (value == null) {
                return null;
            }
            return LocalDate.parse(value, DATE_FORMATTER);
        } catch (DateTimeParseException e) {
            log.warn("日期字符串解析为 LocalDate 失败，key={}, 异常：", key, e);
            return null;
        } catch (Exception e) {
            log.warn("日期转换为 LocalDate 未知异常，key={}, 异常：", key, e);
            return null;
        }
    }

    /**
     * 解析字符串为 LocalDateTime（带时分秒，yyyy-MM-dd HH:mm:ss，适配 DTO 的打卡时间字段）
     */
    private LocalDateTime parseStrToLocalDateTime(JSONObject json, String key) {
        try {
            String value = getStringValue(json, key);
            if (value == null) {
                return null;
            }
            // 先尝试解析为完整日期时间
            return LocalDateTime.parse(value, DATETIME_FORMATTER);
        } catch (DateTimeParseException e) {
            try {
                // 若只有日期，补全时分秒为 00:00:00
                LocalDate localDate = LocalDate.parse(getStringValue(json, key), DATE_FORMATTER);
                return localDate.atStartOfDay();
            } catch (DateTimeParseException ex) {
                log.warn("日期字符串解析为 LocalDateTime 失败，key={}, 异常：", key, ex);
                return null;
            }
        } catch (Exception e) {
            log.warn("日期转换为 LocalDateTime 未知异常，key={}, 异常：", key, e);
            return null;
        }
    }

    /**
     * 解析字符串为 Double（适配 workTime，避免转换异常）
     */
    private Double parseStrToDouble(JSONObject json, String key) {
        try {
            String value = getStringValue(json, key);
            if (value != null) {
                return Double.valueOf(value);
            }
        } catch (Exception e) {
            log.warn("数字字符串解析为 Double 失败，key={}, 异常：", key, e);
        }
        return null;
    }
}