package com.sfa.job.domain.feishu.dao;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.sfa.common.core.utils.bean.BeanUtils;
import com.sfa.common.core.utils.sdk.FeiShuUtil;
import com.sfa.job.domain.feishu.entity.FeishuLeaveInfo;
import com.sfa.job.domain.feishu.mapper.FeishuLeaveInfoMapper;
import com.sfa.job.pojo.feishu.response.FeishuLeaveInfoDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Map;

/**
 * 飞书请假信息Dao实现类
 */
@Service
@Slf4j
public class FeishuLeaveInfoDaoImpl implements FeishuLeaveInfoDao {

    @Resource
    private FeishuLeaveInfoMapper feishuLeaveInfoMapper;
    @Resource
    private FeiShuUtil feiShuUtil;

    // 本地缓存（本次同步有效，避免重复调用飞书接口）
    private Map<String, String> employeeNoCache = new ConcurrentHashMap<>();

    private String extractZhCnValue(JSONArray jsonArray) {
        if (CollectionUtils.isEmpty(jsonArray)) {
            return "";
        }
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject obj = jsonArray.getJSONObject(i);
            // 补充空对象防护
            if (obj != null && "zh-CN".equals(StringUtils.trimToEmpty(obj.getString("lang")))) {
                return StringUtils.trimToEmpty(obj.getString("value"));
            }
        }
        return "";
    }

    private String extractFirstArrayValue(JSONArray jsonArray) {
        if (CollectionUtils.isEmpty(jsonArray)) {
            return "";
        }
        Object first = jsonArray.get(0);
        return first == null ? "" : StringUtils.trimToEmpty(first.toString());
    }

    private void parseCorrectProcessInfo(JSONArray jsonArray, FeishuLeaveInfoDTO dto) {
        if (CollectionUtils.isEmpty(jsonArray)) {
            dto.setProcessApplyTime("");
            dto.setProcessId("");
            dto.setProcessStatus("");
            return;
        }
        JSONObject process = jsonArray.getJSONObject(0);
        if (process != null) {
            dto.setProcessApplyTime(StringUtils.trimToEmpty(process.getString("process_apply_time")));
            dto.setProcessId(StringUtils.trimToEmpty(process.getString("process_id")));
            dto.setProcessStatus(StringUtils.trimToEmpty(process.getString("process_status")));
        }
    }

    /**
     * 带缓存获取员工编号（避免重复调用飞书接口）
     */
    private String getEmployeeNoWithCache(String employmentId) {
        if (StringUtils.isBlank(employmentId)) {
            return "";
        }
        // 先查缓存
        if (employeeNoCache.containsKey(employmentId)) {
            return employeeNoCache.get(employmentId);
        }
        // 缓存未命中，调用接口查询
        String employeeNo = feiShuUtil.getEmployeeNoByEmploymentId(employmentId);
        // 存入缓存
        employeeNoCache.put(employmentId, employeeNo);
        return employeeNo;
    }

    // ************************ 接口实现方法 ************************
    @Override
    public List<FeishuLeaveInfoDTO> parseFeishuLeaveRawJson(String rawJson) {
        List<FeishuLeaveInfoDTO> dtoList = new ArrayList<>();
        if (StringUtils.isBlank(rawJson)) {
            log.warn("飞书原始JSON为空，解析结果为空");
            return dtoList;
        }

        try {
            JSONObject root = JSONObject.parseObject(rawJson);
            JSONArray leaveList = root.getJSONArray("leave_request_list");
            if (CollectionUtils.isEmpty(leaveList)) {
                log.info("飞书响应无有效请假数据");
                return dtoList;
            }

            for (int i = 0; i < leaveList.size(); i++) {
                JSONObject leaveObj = leaveList.getJSONObject(i);
                FeishuLeaveInfoDTO dto = new FeishuLeaveInfoDTO();

                // 业务字段赋值
                dto.setEmploymentId(StringUtils.trimToEmpty(leaveObj.getString("employment_id")));
                dto.setEndTime(StringUtils.trimToEmpty(leaveObj.getString("end_time")));
                dto.setGrantSource(StringUtils.trimToEmpty(leaveObj.getString("grant_source")));
                dto.setLeaveDuration(StringUtils.trimToEmpty(leaveObj.getString("leave_duration")));
                dto.setLeaveDurationUnit(leaveObj.getInteger("leave_duration_unit"));
                dto.setLeaveRequestId(StringUtils.trimToEmpty(leaveObj.getString("leave_request_id")));
                dto.setLeaveRequestStatus(leaveObj.getInteger("leave_request_status"));
                dto.setLeaveTypeId(StringUtils.trimToEmpty(leaveObj.getString("leave_type_id")));
                dto.setNotes(StringUtils.trimToEmpty(leaveObj.getString("notes")));
                dto.setReturnTime(StringUtils.trimToEmpty(leaveObj.getString("return_time")));
                dto.setStartTime(StringUtils.trimToEmpty(leaveObj.getString("start_time")));
                dto.setSubmittedAt(StringUtils.trimToEmpty(leaveObj.getString("submitted_at")));
                dto.setSubmittedBy(StringUtils.trimToEmpty(leaveObj.getString("submitted_by")));
                dto.setTimeZone(StringUtils.trimToEmpty(leaveObj.getString("time_zone")));

                // 提取中文名称和数组第一条数据
                dto.setEmploymentName(extractZhCnValue(leaveObj.getJSONArray("employment_name")));
                dto.setLeaveTypeName(extractZhCnValue(leaveObj.getJSONArray("leave_type_name")));
                dto.setLeaveProcessId(extractFirstArrayValue(leaveObj.getJSONArray("leave_process_id")));
                dto.setLeaveCorrectProcessId(extractFirstArrayValue(leaveObj.getJSONArray("leave_correct_process_id")));

                // 解析嵌套更正流程
                parseCorrectProcessInfo(leaveObj.getJSONArray("leave_correct_process_info"), dto);

                dtoList.add(dto);
            }
            log.info("飞书请假数据解析完成，共{}条有效DTO", dtoList.size());
        } catch (Exception e) {
            log.error("飞书请假数据解析异常", e);
            throw new RuntimeException("飞书请假数据解析失败", e);
        }
        return dtoList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String syncFeishuLeaveData(String syncDate) {
        // 日期格式校验
        if (StringUtils.isBlank(syncDate) || !syncDate.matches("^\\d{4}-\\d{2}-\\d{2}$")) {
            log.error("同步日期格式错误，要求为 yyyy-MM-dd，传入值：{}", syncDate);
            return String.format("同步失败：日期格式错误，要求为 yyyy-MM-dd，传入值：%s", syncDate);
        }

        // 调用飞书工具类获取原始JSON
        String rawJson = feiShuUtil.getAllLeaveRecords(syncDate);
        // 解析为DTO列表
        List<FeishuLeaveInfoDTO> dtoList = parseFeishuLeaveRawJson(rawJson);
        if (CollectionUtils.isEmpty(dtoList)) {
            return String.format("同步[%s]无有效请假数据，新增0条，更新0条", syncDate);
        }

        // 4. 初始化统计参数
        Date now = new Date();
        int totalCount = dtoList.size();
        int addCount = 0;
        int updateCount = 0;
        int skipCount = 0;

        // 清空缓存（本次同步独立缓存）
        employeeNoCache.clear();

        // 遍历处理每条DTO
        for (FeishuLeaveInfoDTO dto : dtoList) {
            // 跳过无唯一ID的记录
            if (StringUtils.isBlank(dto.getLeaveRequestId())) {
                skipCount++;
                continue;
            }

            // 设置系统字段
            dto.setSyncCreateTime(now);
            dto.setSyncUpdateTime(now);
            dto.setIsDelete(FeishuLeaveInfoDTO.IS_DELETE_NO);

            // 核心逻辑：查询并赋值 employee_no（带缓存）
            String employeeNo = getEmployeeNoWithCache(dto.getEmploymentId());
            dto.setEmploymentNo(employeeNo);

            //  查询是否已存在
            FeishuLeaveInfoDTO existDto = selectByLeaveRequestId(dto.getLeaveRequestId());
            if (existDto == null) {
                // 新增：DTO → 实体
                FeishuLeaveInfo entity = BeanUtils.transitionDto(dto, FeishuLeaveInfo.class);
                int insertResult = feishuLeaveInfoMapper.insert(entity);
                if (insertResult > 0) {
                    addCount++;
                    log.info("新增请假数据成功，唯一ID：{}，员工编号：{}", dto.getLeaveRequestId(), employeeNo);
                }
            } else {
                // 更新：构造条件，批量更新
                FeishuLeaveInfo entity = BeanUtils.transitionDto(dto, FeishuLeaveInfo.class);
                LambdaUpdateWrapper<FeishuLeaveInfo> updateWrapper = new LambdaUpdateWrapper<>();
                updateWrapper.eq(FeishuLeaveInfo::getLeaveRequestId, dto.getLeaveRequestId())
                        .eq(FeishuLeaveInfo::getLeaveRequestStatus, FeishuLeaveInfo.IS_DELETE_NO);

                int updateResult = feishuLeaveInfoMapper.update(entity, updateWrapper);
                if (updateResult > 0) {
                    updateCount++;
                    log.info("更新请假数据成功，唯一ID：{}，员工编号：{}", dto.getLeaveRequestId(), employeeNo);
                } else {
                    log.warn("更新请假数据无变更，唯一ID：{}，员工编号：{}", dto.getLeaveRequestId(), employeeNo);
                }
            }
        }

        // 构造同步结果
        return String.format("同步[%s]完成\n总计处理：%d条\n新增：%d条\n更新：%d条\n跳过无效记录：%d条",
                syncDate, totalCount, addCount, updateCount, skipCount);
    }

    @Override
    public FeishuLeaveInfoDTO selectByLeaveRequestId(String leaveRequestId) {
        if (StringUtils.isBlank(leaveRequestId)) {
            return null;
        }

        // 查询实体
        LambdaQueryWrapper<FeishuLeaveInfo> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(FeishuLeaveInfo::getLeaveRequestId, leaveRequestId)
                .eq(FeishuLeaveInfo::getIsDelete, FeishuLeaveInfo.IS_DELETE_NO);
        FeishuLeaveInfo entity = feishuLeaveInfoMapper.selectOne(queryWrapper);

        // 转换为DTO返回
        return BeanUtils.transitionDto(entity, FeishuLeaveInfoDTO.class);
    }

}