package com.sfa.job.xxljob.erp;

import com.alibaba.fastjson2.JSONObject;
import com.sfa.job.pojo.zzcenter.ZzProductQualityDto;
import com.sfa.job.service.zzcenter.IZzProQualityService;
import com.sfa.job.util.T100Util;
import com.xxl.job.core.context.XxlJobHelper;
import com.xxl.job.core.handler.annotation.XxlJob;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 质检报告推送T100定时任务（每小时执行）
 * @Author: DouXinYu
 * @Date: 2026-02-05 17:21
 */
@Component
@Slf4j
public class ZzQualityTask {

    @Autowired
    private IZzProQualityService zzProQualityService;
    @Autowired
    private T100Util t100Util;

    // 单次接口调用重试次数（处理偶发故障）
    private static final int RETRY_TIMES = 2;
    // 重试间隔（毫秒），避免高频重试触发T100限流
    private static final long RETRY_INTERVAL = 500;

    /**
     * 质检报告推送数据到T100 每小时执行一次： * 0 0/1 * * * ?
     */
    @XxlJob("ZzQualityToT100Job")
    public void execute() {
        XxlJobHelper.log("======================== 未同步质检报告推送数据到T100  开始 ================================");
        // 统计本次任务执行结果，运维一眼可知
        AtomicInteger totalCount = new AtomicInteger(0);
        AtomicInteger successCount = new AtomicInteger(0);
        AtomicInteger failCount = new AtomicInteger(0);

        try {
            // 获取未同步的质检报告数据
            List<ZzProductQualityDto> zzProductQualityDtoList = zzProQualityService.queryNotSyncList(0);
            totalCount.set(zzProductQualityDtoList.size());
            XxlJobHelper.log("本次查询到未同步质检报告数据共【{}】条", totalCount.get());
            if (zzProductQualityDtoList.isEmpty()) {
                XxlJobHelper.log("无未同步数据，任务直接结束");
                return;
            }

            // 循环处理，复用JSONObject减少创建（小优化）
            JSONObject deptJson = new JSONObject();
            JSONObject dataMap = new JSONObject();
            for (ZzProductQualityDto dto : zzProductQualityDtoList) {
                try {
                    dataMap.clear();
                    deptJson.clear();

                    dataMap.put("imjyuc001", dto.getPrdCode());
                    dataMap.put("imjyuc002", dto.getManufactureBatchNo());
                    dataMap.put("imjyuc003", dto.getQualityUrl());
                    dataMap.put("imjyuc004", dto.getManufacturersName());
                    deptJson.putObject("payload").putObject("std_data").putObject("parameter").putArray("data").add(dataMap);

                    // 加重试机制，处理偶发故障
                    boolean pushSuccess = retryPush(deptJson);
                    if (pushSuccess) {
                        dto.setSyncStatus(1);
                        zzProQualityService.updateSyncStatus(dto);
                        successCount.incrementAndGet();
                        XxlJobHelper.log("推送T100成功，料号：{}，批号：{}", dto.getPrdCode(), dto.getManufactureBatchNo());
                    } else {
                        failCount.incrementAndGet();
                        log.error("推送T100失败，多次重试后仍未成功，数据：{}", dto);
                    }
                } catch (Exception e) {
                    failCount.incrementAndGet();
                    // 核心：打印完整堆栈，保留异常根因，方便排查
                    log.error("处理单条质检报告数据时发生异常，数据：{}", dto, e);
                }
            }
        } catch (Exception e) {
            // 任务整体异常，标记为失败，XXL-Job管理台会告警
            XxlJobHelper.handleFail("质检报告推送T100任务整体执行异常：" + e.getMessage());
            log.error("质检报告推送T100任务整体执行异常", e);
            return;
        }
        XxlJobHelper.log("======================== 未同步质检报告推送数据到T100  结束 ================================");
        XxlJobHelper.log("本次任务执行统计：总条数【{}】，成功【{}】，失败【{}】",
                totalCount.get(), successCount.get(), failCount.get());
    }

    /**
     * 带重试的T100推送方法，处理偶发故障
     */
    private boolean retryPush(JSONObject deptJson) {
        for (int i = 0; i < RETRY_TIMES; i++) {
            try {
                // 调用原有T100方法
                t100Util.createProQuality(deptJson);
                return true;
            } catch (Exception e) {
                // 重试时打印warn日志，区分永久失败和临时重试
                log.warn("推送T100第【{}】次重试失败，错误信息：{}", i + 1, e.getMessage());
                // 最后一次重试失败，直接返回false
                if (i == RETRY_TIMES - 1) {
                    return false;
                }
                try {
                    Thread.sleep(RETRY_INTERVAL);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    log.error("重试间隔时线程被中断", ie);
                    return false;
                }
            }
        }
        return false;
    }
}