package com.sfa.job.service.order.impl;

import cn.hutool.core.date.DateUtil;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.link.bi.domain.dao.IFinanceOrderDao;
import com.link.bi.domain.entity.*;
import com.link.bi.domain.mapper.FinanceOrderMapper;
import com.link.bi.pojo.response.FinanceSyncOrderDetailDto;
import com.link.bi.service.*;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.DateUtils;
import com.sfa.job.service.order.FinanceOrderSyncService;
import com.sfa.job.util.WangdiantongUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;

/**
 * 旺店通订单Service业务层处理
 *
 * @author lvbencai
 * @date 2025年01月07日17:00:16
 */
@DS("bi")
@Slf4j
@Service
public class FinanceOrderSyncServiceImpl extends ServiceImpl<FinanceOrderMapper, FinanceOrder> implements FinanceOrderSyncService {
    @Autowired
    IFinanceOrderDao orderDao;
    @Autowired
    WangdiantongUtil wangdiantongUtil;

    @Autowired
    FinanceOrderDetailService detailService;
    @Autowired
    FinanceBaseProductService baseProductService;
    @Autowired
    FinanceBaseZbjTypeService baseZbjTypeService;
    @Autowired
    CollectOrderLogInfoService collectOrderLogInfoService;
    @Autowired
    IProductService productService;
    /**
     * 多线程调用此方法
     * @param startTime
     * @param endTime
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public FinanceSyncOrderDetailDto syncWangdiantongOrder(Date startTime, Date endTime, Integer syncType) {
        FinanceSyncOrderDetailDto detailDto = new FinanceSyncOrderDetailDto();
        String batchNo = syncType + DateUtils.dateTimeNow() + Thread.currentThread().getId();
        try {
            if (ObjectUtils.isEmpty(startTime)) {
                // 查询最新的采集时间
                Date date = collectOrderLogInfoService.selectLatest(syncType);
                if (ObjectUtils.isNotEmpty(date)) {
                    startTime = date;
                } else {
                    // 默认上个月的第一天 00:00:00
                    startTime = DateUtil.beginOfDay(DateUtil.beginOfMonth(DateUtil.lastMonth()));
                }
                // 调用查询旺店通接口api 获取最新日期前的一个小时
                startTime = DateUtils.addMinutes(startTime, -3);
            }
            if (ObjectUtils.isEmpty(endTime)) {
                endTime = DateUtils.addMinutes(startTime, 60);
            }
            log.info("查询订单参数：开始时间{},结束时间{}", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime));

            List<FinanceOrder> orders = wangdiantongUtil.queryWithDetail(startTime, endTime);

            if (ObjectUtils.isEmpty(orders)) {
                throw new ServiceException("旺店通没有查询到订单数据");
            }
            // 基础数据 直播间分类数据+成本、规格、口味
            Map<String, FinanceBaseProduct> baseProductMap = baseProductService.selectBaseProduct();
            Map<String, String> baseZbjType = baseZbjTypeService.selectBaseZbjType();
            // 系列
            Map<String, String> prodSeries =  productService.selectProdSeries();
            // 入库订单表
            log.info("开始插入订单数据，数量：{}", orders.size());
            Date finalStartTime = startTime;
            Date finalEndTime = endTime;
            orders.forEach(order -> {
                order.setBatchNo(batchNo);
                order.setStartTime(finalStartTime);
                order.setEndTime(finalEndTime);
                order.setSyncType(syncType);
            });
            this.saveOrUpdateBatch(orders);


            List<FinanceOrderDetail> mergeList = new ArrayList<>();
            // 用于同一个订单号，来赋值直播间信息
            Map<String, FinanceOrderDetail> orderZbj = new HashMap<>();
            // 入库订单明细表
            for (FinanceOrder order : orders) {
                List<FinanceOrderDetail> orderDetails = order.getDetailList();
                orderDetails.forEach(orderDetail -> {
                    orderDetail.setSeries(prodSeries.get(orderDetail.getGoodsNo()));

                    orderDetail.setTradeStatus(order.getTradeStatus());
                    orderDetail.setShopNo(order.getShopNo());
                    orderDetail.setShopName(order.getShopName());
                    orderDetail.setShopRemark(order.getShopRemark());
                    orderDetail.setReceivable(order.getReceivable());
                    orderDetail.setReceiverArea(order.getReceiverArea());
                    orderDetail.setConsignTime(order.getConsignTime());
                    orderDetail.setTradeTime(order.getTradeTime());
                    orderDetail.setBatchNo(batchNo);
                    orderDetail.setTradeNo(order.getTradeNo());
                    // 计算分销信息
                    orderDetail.setFenxiaoNick(orderDetail.getFenxiaoNick(order));
                    orderDetail.setFenxiaoId(orderDetail.getFenxiaoId(order, orderDetail.getTradeId()));
                    orderDetail.setFenxiaoName(orderDetail.getFenxiaoName(order, orderDetail.getTradeId()));
                    orderDetail.setStartTime(finalStartTime);
                    orderDetail.setEndTime(finalEndTime);
                    orderDetail.setSyncType(syncType);
                    // 如果存在相同的
                    FinanceOrderDetail sameDetail = orderZbj.get(orderDetail.getSrcTid());

                    orderDetail.setZbjName(orderDetail.getZbjName(orderDetail.getRemark(), sameDetail));
                    orderDetail.setZbjId(orderDetail.getZbjId(orderDetail.getRemark(), sameDetail));
                    orderDetail.setZbjZbId(orderDetail.getZbjZbId(orderDetail.getFenxiaoId(), orderDetail.getRemark(), sameDetail));
                    orderDetail.setZbjSaleType(orderDetail.getZbjSaleType(orderDetail.getRemark(), sameDetail));
                    // 确定分销商类型
                    orderDetail.setZbjQdType(orderDetail.getZbjQdType(orderDetail.getFenxiaoId(), orderDetail.getZbjZbId(), baseZbjType));
                    if (ObjectUtils.isNotEmpty(orderDetail.getZbjName())) {
                        orderZbj.put(orderDetail.getSrcTid(), orderDetail);
                    }
                    orderDetail.setFlavorErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getFlavor() : "");
                    orderDetail.setSpecNameErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getSpec() : "");
                    orderDetail.setActualCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getActualCost() : new BigDecimal(0));
                    orderDetail.setStandardCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getStandardCost() : new BigDecimal(0));
                });
                mergeList.addAll(orderDetails);
            }
            log.info("开始插入订单详情数据，数量：{}", mergeList.size());

            // 批量插入
            detailService.saveOrUpdateBatch(mergeList);
            detailDto.setOrders(orders);
            detailDto.setOrderCount(orders.size());
            detailDto.setOrderDetailCount(mergeList.size());
            detailDto.setOrderDetails(mergeList);
            detailDto.setStartTime(startTime);
            detailDto.setEndTime(endTime);
            detailDto.setBatchNo(batchNo);

            log.info("完成插入订单和订单详情数据，开始时间{},结束时间{},订单数量:{},详情数量：{}", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime), orders.size(), mergeList.size());

            CollectOrderLogInfo collectOrderLogInfo = new CollectOrderLogInfo();
            collectOrderLogInfo.setSyncType(syncType);
            collectOrderLogInfo.setOrderCount(orders.size());
            collectOrderLogInfo.setOrderDetailCount(mergeList.size());
            collectOrderLogInfo.setBatchNo(batchNo);
            collectOrderLogInfo.setLatestTime(endTime);
            collectOrderLogInfoService.save(collectOrderLogInfo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            // 记录错误日志表 collect_error_info
            CollectErrorInfo errorInfo =new CollectErrorInfo();
            errorInfo.setBatchNo(batchNo);
            errorInfo.setType("");
            errorInfo.setErrorMsg(e.getMessage());
            errorInfo.setEndTime(endTime);
            errorInfo.setStartTime(startTime);
            errorInfo.setCollectTime(new Date());
            // ????

            throw e;
        }
        return detailDto;
    }

}
