package com.sfa.job.service.order.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONReader;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.DateUtils;
import com.sfa.job.domain.order.dao.*;
import com.sfa.job.domain.order.entity.*;
import com.sfa.job.domain.order.mapper.FinanceOrderMapper;
import com.sfa.job.pojo.request.WangdiantongQueryVO;
import com.sfa.job.pojo.response.FinanceSyncOrderDetailDto;
import com.sfa.job.pojo.response.WangdiantongQueryDto;
import com.sfa.job.service.order.FinanceOrderSyncService;
import com.sfa.job.util.WangdiantongUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 旺店通订单Service业务层处理
 *
 * @author lvbencai
 * @date 2025年01月07日17:00:16
 */
@DS("bi")
@Slf4j
@Service
public class FinanceOrderSyncServiceImpl extends ServiceImpl<FinanceOrderMapper, FinanceOrder> implements FinanceOrderSyncService {
    @Autowired
    WangdiantongUtil wangdiantongUtil;

    @Autowired
//    FinanceOrderDetailService detailService;
    FinanceOrderDetailDao detailDao;
    @Autowired
    FinianceBaseProductDao baseProductDao;
    @Autowired
    FinanceBaseZbjTypeDao baseZbjTypeDao;
    @Autowired
    CollectOrderLogInfoDao orderLogInfoDao;
    @Autowired
    CollectErrorInfoDao errorInfoDao;
    @Autowired
    IProductDao productDao;

    /**
     * 多线程调用此方法
     *
     * @param startTime
     * @param endTime
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public FinanceSyncOrderDetailDto syncWangdiantongOrder(Date startTime, Date endTime, Integer syncType) {
        FinanceSyncOrderDetailDto detailDto = new FinanceSyncOrderDetailDto();
        String batchNo = syncType + DateUtils.dateTimeNow() + Thread.currentThread().getId();
        AtomicInteger beginPageNo = new AtomicInteger(0);
        // 继续下一页的标识 pageNo =-1 标识终止分页 pageNo >0 标识继续分页
        Boolean nextPageFlag = true;
        try {
            // startTime 为空，默认是定时任务调用
            if (ObjectUtils.isEmpty(startTime)) {
                Date latestTime = null;
                // 查询最新的采集时间
                CollectOrderLogInfo collectOrderLogInfo = orderLogInfoDao.selectOrderSyncLatest(syncType);
                if (ObjectUtils.isNotEmpty(collectOrderLogInfo)) {
                    latestTime = collectOrderLogInfo.getLatestTime();
                    Integer pageNoExist = collectOrderLogInfo.getPageNo();
                    beginPageNo.set(ObjectUtil.isNotEmpty(pageNoExist) && pageNoExist > 0 ? collectOrderLogInfo.getPageNo() : 0);
                    nextPageFlag = beginPageNo.get() > 0;
                } else {
                    // 默认上个月的第一天 00:00:00
                    startTime = DateUtil.beginOfDay(DateUtil.beginOfMonth(DateUtil.lastMonth()));
                }

                if (nextPageFlag) {
                    startTime = DateUtils.addMinutes(latestTime, -60);
                } else {
                    // 调用查询旺店通接口api 获取最新日期前的一个小时
                    startTime = DateUtils.addMinutes(latestTime, -3);
                }

            }
            // 调用查询旺店通接口api 获取最新日期前的2小时
            Date currentLatest = DateUtils.addHours(new Date(), -2);
            if (currentLatest.compareTo(startTime) < 0) {
                detailDto.setMessage(String.format("开始时间%s不能大于当前时间%s",startTime,currentLatest));
                log.warn(detailDto.getMessage());
                return detailDto;
            }

            if (ObjectUtils.isEmpty(endTime)) {
                endTime = DateUtils.addMinutes(startTime, 60);
            }
            log.info("查询订单参数：开始时间{},结束时间{},当前页:{}", DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime), beginPageNo.get());

            WangdiantongQueryVO wangdiantongQueryVO = new WangdiantongQueryVO();
            wangdiantongQueryVO.setStart_time(com.alibaba.fastjson2.util.DateUtils.format(startTime));
            wangdiantongQueryVO.setEnd_time(com.alibaba.fastjson2.util.DateUtils.format(endTime));

            // 旺店通入参 和其他入参分开传入   旺店通入参bean转map
            WangdiantongQueryDto wangdiantongQueryDto = wangdiantongUtil.queryWithDetail(wangdiantongQueryVO, beginPageNo);
            JSONArray orderAllArray = wangdiantongQueryDto.getResult();
            if (ObjectUtils.isEmpty(orderAllArray)) {
                throw new ServiceException("旺店通没有查询到订单数据");
            }

            //转化成orderList
            List<FinanceOrder> orders = JSONArray.parseArray(orderAllArray.toString(), FinanceOrder.class, JSONReader.Feature.SupportSmartMatch);

            // 基础数据 直播间分类数据+成本、规格、口味
            Map<String, FinanceBaseProduct> baseProductMap = baseProductDao.selectBaseProduct();

            Map<String, String> baseZbjType = baseZbjTypeDao.selectBaseZbjType();
            // 系列
            List<PrdInfo> prdInfos = productDao.selectProdSeries();
            // 转换成map
            Map<String, String> prodSeriesMap = new HashMap<>();
            prdInfos.stream().forEach(prdInfo -> {
                prodSeriesMap.put(prdInfo.getPrdCode(), prdInfo.getPrdName());
            });
            // 入库订单表
            log.info("开始插入订单数据，数量：{}", orderAllArray.size());
            Date finalStartTime = startTime;
            Date finalEndTime = endTime;
            orders.forEach(order -> {
                order.setBatchNo(batchNo);
                order.setStartTime(finalStartTime);
                order.setEndTime(finalEndTime);
                order.setSyncType(syncType);
            });
            this.saveOrUpdateBatch(orders);


            List<FinanceOrderDetail> mergeList = new ArrayList<>();
            // 用于同一个订单号，来赋值直播间信息
            Map<String, FinanceOrderDetail> orderZbj = new HashMap<>();
            // 入库订单明细表
            for (FinanceOrder order : orders) {
                List<FinanceOrderDetail> orderDetails = order.getDetailList();
                for (FinanceOrderDetail orderDetail : orderDetails) {
                    orderDetail.setSeries(prodSeriesMap.get(orderDetail.getGoodsNo()));

                    orderDetail.setTradeStatus(order.getTradeStatus());
                    orderDetail.setShopNo(order.getShopNo());
                    orderDetail.setShopName(order.getShopName());
                    orderDetail.setShopRemark(order.getShopRemark());
                    orderDetail.setReceivable(order.getReceivable());
                    orderDetail.setReceiverArea(order.getReceiverArea());
                    orderDetail.setConsignTime(order.getConsignTime());
                    orderDetail.setTradeTime(order.getTradeTime());
                    orderDetail.setBatchNo(batchNo);
                    orderDetail.setTradeNo(order.getTradeNo());
                    // 计算分销信息
                    orderDetail.setFenxiaoNick(orderDetail.getFenxiaoNick(order));
                    orderDetail.setFenxiaoId(orderDetail.getFenxiaoId(order, orderDetail.getTradeId()));
                    orderDetail.setFenxiaoName(orderDetail.getFenxiaoName(order, orderDetail.getTradeId()));
                    orderDetail.setStartTime(finalStartTime);
                    orderDetail.setEndTime(finalEndTime);
                    orderDetail.setSyncType(syncType);
                    // 如果存在相同的
                    FinanceOrderDetail sameDetail = orderZbj.get(orderDetail.getSrcTid());

                    orderDetail.setZbjName(orderDetail.getZbjName(orderDetail.getRemark(), sameDetail));
                    orderDetail.setZbjId(orderDetail.getZbjId(orderDetail.getRemark(), sameDetail));
                    orderDetail.setZbjZbId(orderDetail.getZbjZbId(orderDetail.getFenxiaoId(), orderDetail.getRemark(), sameDetail));
                    orderDetail.setZbjSaleType(orderDetail.getZbjSaleType(orderDetail.getRemark(), sameDetail));
                    orderDetail.setFenxiaoId(orderDetail.getZbjZbId());
                    orderDetail.setFenxiaoName(orderDetail.getZbjName());

                    // 确定分销商类型
                    orderDetail.setZbjQdType(orderDetail.getZbjQdType(orderDetail.getFenxiaoId(), orderDetail.getZbjZbId(), baseZbjType));
                    if (ObjectUtils.isNotEmpty(orderDetail.getZbjName())) {
                        orderZbj.put(orderDetail.getSrcTid(), orderDetail);
                    }
                    orderDetail.setFlavorErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getFlavor() : "");
                    orderDetail.setSpecNameErp(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ? baseProductMap.get(orderDetail.getGoodsNo()).getSpec() : "");
                    orderDetail.setActualCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                    baseProductMap.get(orderDetail.getGoodsNo()).getActualCost() :
                                    new BigDecimal(0)
//                                            (ObjectUtils.isNotEmpty(order.getGoodsTypeCount()) && (order.getGoodsTypeCount().signum()!= 0) ? order.getGoodsCost().divide(order.getGoodsTypeCount()).setScale(4, BigDecimal.ROUND_HALF_UP) : new BigDecimal(0))
                    );
                    orderDetail.setStandardCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                    baseProductMap.get(orderDetail.getGoodsNo()).getStandardCost() :
                                    new BigDecimal(0)
//                                            (ObjectUtils.isNotEmpty(order.getGoodsTypeCount()) && (order.getGoodsTypeCount().signum()!= 0)  ? order.getGoodsCost().divide(order.getGoodsTypeCount()).setScale(4, BigDecimal.ROUND_HALF_UP) : new BigDecimal(0))
                    );
                    orderDetail.setSumActualCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                            baseProductMap.get(orderDetail.getGoodsNo()).getActualCost().multiply(orderDetail.getNum()) :
                            new BigDecimal(0));
                    orderDetail.setSumStandardCost(ObjectUtils.isNotEmpty(baseProductMap.get(orderDetail.getGoodsNo())) ?
                                    baseProductMap.get(orderDetail.getGoodsNo()).getStandardCost().multiply(orderDetail.getNum())  :
                                    new BigDecimal(0)
//                                            (ObjectUtils.isNotEmpty(order.getGoodsTypeCount()) && (order.getGoodsTypeCount().signum()!= 0)  ? order.getGoodsCost().divide(order.getGoodsTypeCount()).setScale(4, BigDecimal.ROUND_HALF_UP) : new BigDecimal(0))
                    );
                }
                mergeList.addAll(orderDetails);
            }
            log.info("开始插入订单详情数据，数量：{}", mergeList.size());

            // 批量插入
            detailDao.saveOrUpdateBatch(mergeList);


            log.info("插入订单和订单详情完成，批次{}开始时间{},结束时间{},订单数量:{},详情数量：{}", batchNo, DateUtil.formatDateTime(startTime), DateUtil.formatDateTime(endTime), orders.size(), mergeList.size());

            CollectOrderLogInfo collectOrderLogInfo = new CollectOrderLogInfo();
            collectOrderLogInfo.setSyncType(syncType);
            collectOrderLogInfo.setOrderCount(orders.size());
            collectOrderLogInfo.setOrderDetailCount(mergeList.size());
            collectOrderLogInfo.setBatchNo(batchNo);
            collectOrderLogInfo.setLatestTime(endTime);
            collectOrderLogInfo.setPageNo(beginPageNo.get());
            collectOrderLogInfo.setTotalCount(wangdiantongQueryDto.getTotalCount());
            orderLogInfoDao.insert(collectOrderLogInfo);

            // 接口测试使用
            detailDto.setOrderCount(orders.size());
            detailDto.setOrderDetailCount(mergeList.size());
            detailDto.setStartTime(finalStartTime);
            detailDto.setPageNo(beginPageNo.get());
            detailDto.setEndTime(finalEndTime);
            detailDto.setTotalCount(wangdiantongQueryDto.getTotalCount());
            detailDto.setBatchNo(batchNo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            // 记录错误日志表 collect_error_info
            CollectErrorInfo errorInfo = new CollectErrorInfo();
            errorInfo.setBatchNo(batchNo);
            errorInfo.setType("");
            errorInfo.setErrorMsg(e.getMessage());
            errorInfo.setEndTime(endTime);
            errorInfo.setStartTime(startTime);
            errorInfo.setCollectTime(new Date());
            errorInfoDao.insert(errorInfo);
            throw e;
        }
        return detailDto;
    }

}
