package com.sfa.job.xxljob.feishu;

import com.xxl.job.core.context.XxlJobHelper;
import com.xxl.job.core.handler.annotation.XxlJob;
import com.sfa.job.service.feishu.IFeishuLeaveInfoService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

/**
 * 飞书请假数据XXL-JOB定时任务类（适配实际Service：syncFeishuLeaveData(String syncDate)）
 * 任务：每天00:02同步前一天的请假数据
 */
@Slf4j
@Component
public class FeiShuLeaveXxlJob {

    @Autowired
    private IFeishuLeaveInfoService feishuLeaveInfoService;

    // 日期格式化器（与勤策保持一致，适配 syncDate 入参：yyyy-MM-dd）
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    /**
     * 飞书请假数据同步任务（XXL-JOB核心方法）
     * 调度配置：每天02:00执行（Cron表达式：0 0 2 * * ?）
     */
    @XxlJob("feiShuLeaveSyncJob")
    public void feiShuLeaveSyncJob() {
        // 任务日志记录（XXL-JOB控制台可见）
        XxlJobHelper.log("===== 开始执行飞书请假数据同步任务（同步前一天数据） =====");

        //  获取前一天日期（格式：yyyy-MM-dd），适配Service入参要求
        LocalDate yesterday = LocalDate.now().minusDays(1);
        String syncDate = yesterday.format(DATE_FORMATTER);
        XxlJobHelper.log("===== 本次同步目标日期：{} =====", syncDate);

        try {
            // 调用你实际的飞书请假同步方法（入参：前一天日期字符串）
            String syncResult = feishuLeaveInfoService.syncFeishuLeaveData(syncDate);

            // 任务结果日志（适配Service返回的字符串结果）
            XxlJobHelper.log("===== 飞书请假数据同步任务执行完成，同步结果：{} =====", syncResult);
            // 标记任务执行成功（XXL-JOB状态回调）
            XxlJobHelper.handleSuccess("飞书请假数据同步成功，同步结果：" + syncResult);
        } catch (Exception e) {
            // 异常处理与日志记录
            log.error("===== 飞书请假数据同步任务执行失败 =====", e);
            XxlJobHelper.log("===== 飞书请假数据同步任务执行失败，异常信息：{} =====", e.getMessage());
            // 标记任务执行失败（XXL-JOB状态回调）
            XxlJobHelper.handleFail("飞书请假数据同步失败，异常信息：" + e.getMessage());
        }
    }
}