package com.sfa.common.core.web.domain;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.sfa.common.core.constant.HttpStatus;
import com.sfa.common.core.exception.GlobalException;
import com.sfa.common.core.utils.bean.BeanUtils;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author : liqiulin
 * @date : 2024-04-19 13
 * @describe : 分页配置，默认每页10条记录
 */
public class PageInfo<T> implements Serializable {
    private static final long serialVersionUID = 1L;

    /**
     * 当前页数，第1页开始
     */
    private int pageNum;

    /**
     * 每页条数
     */
    private int pageSize;

    /**
     * 总条数
     */
    private int total;

    /**
     * 总页数
     */
    private int totalPage;

    private List<T> rows;

    private Map<String, Object> queryParams;

    /**
     * current：当前页数，必须大于等于 1，默认值为 1。
     * size：每页条数，默认值为 10。
     * total：总条数，默认值为 0。
     * records：当前页数据集合，默认值为空集合。
     * searchCount：是否进行 count 查询，默认值为 true，表示会统计总条数。
     * pages：总页数，通过计算得出。
     * optimizeCountSql：是否优化 count 查询，默认值为 true。
     * hitCount：是否对 count 进行 limit 优化，默认值为 false。
     * countId：count 查询的列名，默认为 null，表示所有列。
     * maxLimit：设置最大的 limit 查询限制，默认值为 -1，表示不限制。
     *
     * @param iPage 查询出的分页对象
     * @param t     需要转换的对象（用于返回前端，不能直接返回DO）
     * @param <P>   分页对象中的类型
     */
    public <P> void pageCovert(IPage<P> iPage, Class<T> t) {
        this.total = (int) iPage.getTotal();
        this.totalPage = (int) iPage.getPages();

        List<P> iPageRecords = iPage.getRecords();
        if (CollectionUtils.isEmpty(iPageRecords)) {
            this.rows = new ArrayList<>();
            return;
        }

        try {
            List<T> tList = new ArrayList<>(iPageRecords.size() * 2);
            for (P pr : iPageRecords) {
                T newT = t.newInstance();
                // 把原对象数据拷贝到新的对象
                BeanUtils.copyProperties(pr, newT);
                tList.add(newT);
            }
            this.rows = tList;
        } catch (Exception e) {
            throw new GlobalException(HttpStatus.ERROR);
        }
    }

    /**
     * 当前方法直接返回DO数据，在Service层无需再次处理DO数据时使用
     * @param iPage DB中的DO数据
     * PS：如需在Service层对DO数据二次处理时，请使用IPage<P> iPage, Class<T> t，返回DTO数据，将DO数据与业务隔离
     */
    public PageInfo(IPage<T> iPage) {
        this.pageNum = (int) iPage.getCurrent();
        this.pageSize = (int) iPage.getSize();
        this.total = (int) iPage.getTotal();
        this.totalPage = (int) iPage.getPages();
        this.rows = iPage.getRecords();
    }

    public int getSkipNum() {
        return this.pageSize * (this.pageNum - 1);
    }

    public int getPageNum() {
        return pageNum;
    }

    public void setPageNum(int pageNum) {
        this.pageNum = pageNum;
    }

    public int getPageSize() {
        return pageSize;
    }

    public void setPageSize(int pageSize) {
        this.pageSize = pageSize;
    }

    public int getTotalPage() {
        return totalPage;
    }

    public void setTotalPage(int totalPage) {
        this.totalPage = totalPage;
    }

    public List<T> getRows() {
        return rows;
    }

    public void setRows(List<T> rows) {
        this.rows = rows;
    }

    public Map<String, Object> getQueryParams() {
        return queryParams;
    }

    public void setQueryParams(Map<String, Object> queryParams) {
        this.queryParams = queryParams;
    }

    public int getTotal() {
        return total;
    }

    public void setTotal(int total) {
        this.total = total;
        if (this.pageSize == 0) {
            this.totalPage = 0;
        } else {
            int pages = this.total / this.pageSize;
            if (this.total % this.pageSize != 0L) {
                ++pages;
            }
            this.totalPage = pages;
        }
    }

    public PageInfo() {
    }
}
