package com.sfa.common.core.utils.sdk;

import cn.hutool.http.HttpUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.google.gson.JsonParser;
import com.lark.oapi.Client;
import com.lark.oapi.core.request.RequestOptions;
import com.lark.oapi.core.utils.Jsons;
import com.lark.oapi.okhttp.*;
import com.lark.oapi.service.approval.v4.model.*;
import com.lark.oapi.service.authen.v1.model.GetUserInfoResp;
import com.lark.oapi.service.authen.v1.model.GetUserInfoRespBody;
import com.lark.oapi.service.contact.v3.enums.ChildrenDepartmentDepartmentIdTypeEnum;
import com.lark.oapi.service.contact.v3.enums.FindByDepartmentUserDepartmentIdTypeEnum;
import com.lark.oapi.service.contact.v3.enums.FindByDepartmentUserUserIdTypeEnum;
import com.lark.oapi.service.contact.v3.model.ChildrenDepartmentReq;
import com.lark.oapi.service.contact.v3.model.ChildrenDepartmentResp;
import com.lark.oapi.service.contact.v3.model.FindByDepartmentUserReq;
import com.lark.oapi.service.contact.v3.model.FindByDepartmentUserResp;
import com.lark.oapi.service.corehr.v1.model.*;
import com.lark.oapi.service.corehr.v1.model.LeaveRequest;
import com.lark.oapi.service.corehr.v2.model.*;
import com.lark.oapi.service.corehr.v2.model.Department;
import com.sfa.common.core.enums.ECode;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.exception.auth.NotLoginException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.lang.Object;
import java.math.BigDecimal;
import java.util.*;

/**
 * @author : liqiulin
 * @date : 2024-12-06 16
 * @describe :
 */
@Component
public class FeiShuUtil {
    private static final Logger log = LoggerFactory.getLogger(FeiShuUtil.class);
    @Value("${feishu.luzx.app_id}")
    private String appId;
    @Value("${feishu.luzx.app_secret}")
    private String appSecret;
    @Value("${feishu.lbxy.app_id}")
    private String lbxyAppId;
    @Value("${feishu.lbxy.app_secret}")
    private String lbxyAppSecret;
    @Value("${feishu.bot_1.app_id}")
    private String botAppId1;
    @Value("${feishu.bot_1.app_secret}")
    private String botAppSecret1;

    /**
     * 飞书获取用户信息
     */
    @Value("${feishu.luzx_redirectUri}")
    private String redirectUri;
    @Value("${feishu.lbxy_redirectUri}")
    private String lbxyRedirectUri;

    /**
     * 飞书群机器人推送，此机器人为勤策消息通知机器人
     */
    public static String qinceWebhook;

    /**
     * 飞书群机器人推送，此机器人为syncprice消息通知机器人
     */
    public static String syncpriceWebhook;
    /*链路中心*/
    public static final String APP_LUZX = "luzx";
    /*卤币学院*/
    public static final String APP_LBXY = "lbxy";

    private static final String[] LEAVE_STATUS = new String[] {"1","9"};

    private static final String UTF_8 = "UTF-8";
    private static final String CONTENT_TYPE = "application/json; charset=utf-8";
    private static final String SHEET_URL = "https://open.feishu.cn/open-apis/sheets/v2/spreadsheets/";
    private static final String TENANT_ACCESS_TOKEN_URL = "https://open.feishu.cn/open-apis/auth/v3/tenant_access_token/internal";
    private static final String VALUES_APPEND = "/values_append";
    private static final String VALUES_IMAGE = "/values_image";
    private static final String VALUES_BATCH_GET = "/values_batch_get";
    private static final String VALUES = "/values";
    private static final String STYLE = "/style";


    /**
     * 根据用户的登录临时code获取useraccessToken
     * 应用：链路中心
     */
    public String createUserAccessToken(String code,String app) {
        try {
            HashMap<String, String> bodyMap = new HashMap<>();
            bodyMap.put("grant_type", "authorization_code");
            bodyMap.put("client_id", APP_LUZX.equals(app) ? appId : lbxyAppId);
            bodyMap.put("client_secret", APP_LUZX.equals(app) ? appSecret : lbxyAppSecret);
            bodyMap.put("redirect_uri", APP_LUZX.equals(app) ? redirectUri : lbxyRedirectUri);

            bodyMap.put("code", code);

            OkHttpClient client = new OkHttpClient();
            RequestBody body = RequestBody.create(MediaType.get("application/json"), JSONObject.toJSONString(bodyMap));
            Request build = new Request.Builder().url("https://open.feishu.cn/open-apis/authen/v2/oauth/token").addHeader("Content-Type", "application/json; charset=utf-8").post(body).build();
            Response execute = client.newCall(build).execute();
            JSONObject rj = JSONObject.parseObject(execute.body().string());
            if (!rj.containsKey("access_token")) {
                log.error("飞书获取access_token失败！,入参：{}，返回：{}", JSONObject.toJSONString(bodyMap), rj.toString());
                throw new NotLoginException(ECode.FEISHU_ACCESS_TOKEN_ERROR);
            }
            return rj.getString("access_token");
        } catch (Exception e) {
            throw new NotLoginException(ECode.FEISHU_ACCESS_TOKEN_ERROR);
        }
    }

    /**
     * 根据用户的userAccessToken获取用户信息
     */
    public GetUserInfoRespBody getUserInfo(String userAccessToken,String app) {
        try {
            Client client = APP_LUZX.equals(app) ? getClientLUZX() : getClientLBXY();
            GetUserInfoResp resp = client.authen().userInfo().get(RequestOptions.newBuilder()
                    .userAccessToken(userAccessToken)
                    .build());
            if (!resp.success()) {
                log.error("根据userAccessToken获取用户信息失败：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            return resp.getData();
        } catch (Exception e) {
            log.error("获取用户信息失败，停止执行！");
            return null;
        }
    }

    /**
     * 批量获取飞书人事部门信息
     * https://open.feishu.cn/document/corehr-v1/organization-management/department/batch_get?appId=cli_a8d8e0fc58d9100e
     */
    public Department getDepartmentCorehr(String deptId) {
        // 判断部门ID，如果是"od-"开头，部门ID类型为"open_department_id"，否则为"people_corehr_department_id"
        String departmentIdType = deptId.startsWith("od-") ? "open_department_id" : "people_corehr_department_id";
        String[] deptIds = new String[]{deptId};
        try {
            Client client = getClientLUZX();
            // 创建请求对象
            BatchGetDepartmentReq req = BatchGetDepartmentReq.newBuilder()
                    .userIdType("open_id")
                    .departmentIdType(departmentIdType)
                    .batchGetDepartmentReqBody(BatchGetDepartmentReqBody.newBuilder()
                            .departmentIdList(deptIds)
                            .fields(new String[]{
                                    "department_name",
                                    "parent_department_id",
                                    "code",
                                    "manager",
                                    "is_root",
                                    "effective_date",
                                    "expiration_date",
                                    "active",
                                    "custom_fields"
                            })
                            .departmentNameList(new String[]{})
                            .build())
                    .build();
            BatchGetDepartmentResp resp = client.corehr().v2().department().batchGet(req);
            // 处理服务端错误
            if (!resp.success()) {
                log.error("获取飞书部门信息失败：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            Department[] items = resp.getData().getItems();
            return items[0];
        } catch (Exception e) {
            log.error("获取飞书人事部门失效，停止执行！");
            throw new RuntimeException(e);
        }
    }

    public DepartmentParentInfo[] getParentsDepartmentReq(String deptId){
        // 延迟3s

        String[] deptIds = new String[]{deptId};
        String departmentIdType = deptId.startsWith("od-") ? "open_department_id" : "people_corehr_department_id";
        try {
            Client client = getClientLUZX();
            ParentsDepartmentReq req = ParentsDepartmentReq.newBuilder()
                    .departmentIdType(departmentIdType)
                    .parentsDepartmentReqBody(ParentsDepartmentReqBody.newBuilder()
                            .departmentIdList(deptIds)
                            .build())
                    .build();
            ParentsDepartmentResp resp = client.corehr().v2().department().parents(req);
            if(!resp.success()) {
                log.error("获取飞书人事部门上级树结构失效：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            return resp.getData().getItems()[0].getParentDepartmentList();
        } catch (Exception e) {
            log.error("获取飞书人事部门上级树结构失效，停止执行！");
            throw new RuntimeException(e);
        }

    }

    /**
     * 根据部门父编码获取子部门
     * 公司编码："0"
     */
    public JSONArray getDeptByParentId(String parentId) {
        try {
            Client client = getClientLUZX();
            ChildrenDepartmentReq req = ChildrenDepartmentReq.newBuilder()
                    .departmentId(parentId)
                    .departmentIdType(ChildrenDepartmentDepartmentIdTypeEnum.DEPARTMENT_ID)
                    .pageSize(50)
                    .build();
            ChildrenDepartmentResp resp = client.contact().department().children(req);
            if (!resp.success()) {
                log.error("根据部门父编码获取子部门失败：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            JSONArray items = JSONObject.parse(Jsons.DEFAULT.toJson(resp.getData())).getJSONArray("items");
            return items;
        } catch (Exception e) {
            log.error("获取子部门列表失败，停止执行！");
        }
        return null;
    }

    /**
     * 获取部门用户5
     */
    public JSONArray getUsersByDeptId(String deptCode) {
        try {
            Client client = getClientLUZX();
            FindByDepartmentUserReq req = FindByDepartmentUserReq.newBuilder()
                    .userIdType(FindByDepartmentUserUserIdTypeEnum.OPEN_ID)
                    .departmentIdType(FindByDepartmentUserDepartmentIdTypeEnum.DEPARTMENT_ID)
                    .departmentId(deptCode)
                    .pageSize(50)
                    .build();
            FindByDepartmentUserResp resp = client.contact().user().findByDepartment(req);
            if (!resp.success()) {
                log.error("获取部门用户：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            JSONArray items = JSONObject.parse(Jsons.DEFAULT.toJson(resp.getData())).getJSONArray("items");
            return items;
        } catch (Exception e) {
            log.error("获取部门用户失败，停止执行！部门id：{}", deptCode);
        }
        return null;
    }

    /**
     * 获取电子表格数据
     */
    public JSONArray pullSheet(String sheetToken, String autoToken, String ranges) {
        try {
            Map<String, Object> params = new HashMap<>();
            params.put("ranges", ranges);
            String result = HttpUtil.createGet(SHEET_URL + sheetToken + VALUES_BATCH_GET).form(params).auth("Bearer " + autoToken).contentType(CONTENT_TYPE).execute().body();
            JSONObject resultJson = JSONObject.parseObject(result);
            int code = resultJson.getInteger("code");
            if (code != 0) {
                throw new RuntimeException("获取飞书表格范围数据错误：" + resultJson.getString("msg"));
            }
            return resultJson.getJSONObject("data").getJSONArray("valueRanges");
        } catch (Exception e) {
            log.error("拉取表格数据失败，停止执行！", e);
        }
        return new JSONArray();
    }

    /**
     * 写入电子表格数据
     */
    public Integer pushSetToSheet(String range, List<Object> datas, String sheetToken, String autoToken) {
        // 组装参数
        Map<String, Map<String, Object>> bodyMap = new HashMap<>();
        Map<String, Object> valueRange = new HashMap<>();
        valueRange.put("range", range);
        valueRange.put("values", datas);
        bodyMap.put("valueRange", valueRange);
        String bodyJson = JSONObject.toJSONString(bodyMap);

        // body上传
        String result = HttpUtil.createPost(SHEET_URL + sheetToken + VALUES_APPEND)
                .auth("Bearer " + autoToken).contentType(CONTENT_TYPE)
                .body(bodyJson)
                .execute().body();

        // 上传结果判断
        JSONObject resultJson = JSONObject.parseObject(result);
        return resultJson.getInteger("code");
    }

    /**
     * 创建审批
     */
    public String createApproval(Map<String, Object> approval) {
        try {
            Client client = getClientLUZX();
            CreateInstanceReq req = CreateInstanceReq.newBuilder()
                    .instanceCreate(InstanceCreate.newBuilder()
                            .approvalCode(approval.get("approval_code").toString())
                            .form(JSONObject.toJSONString(approval.get("form")))
                            .userId(approval.get("user_id").toString())
                            .allowResubmit(Boolean.getBoolean(approval.get("allow_resubmit").toString()))
                            .allowSubmitAgain(Boolean.getBoolean(approval.get("allow_submit_again").toString()))
                            .build()
                    ).build();
            CreateInstanceResp resp = client.approval().v4().instance().create(req);
            if (!resp.success()) {
                throw new RuntimeException(String.format("%s,code:%s,msg:%s,reqId:%s, resp:%s", "创建飞书审批失败",
                        resp.getCode(), resp.getMsg(), resp.getRequestId(), Jsons.createGSON(true, false).toJson(JsonParser.parseString(new String(resp.getRawResponse().getBody(), UTF_8)))));
            }
            return JSONObject.parseObject(JSONObject.toJSONString(resp.getData())).getString("instance_code");
        } catch (Exception e) {
            throw new ServiceException(String.format(ECode.FS_CREATE_APPROVAL_ERROR.getMsg(), e.getMessage()), ECode.FS_CREATE_APPROVAL_ERROR.getCode());
        }
    }

    public String getApprovalInfo(String approvalCode) {
        try {
            Client client = getClientLUZX();
            GetInstanceReq req = GetInstanceReq.newBuilder().instanceId(approvalCode).build();
            GetInstanceResp resp = client.approval().v4().instance().get(req);
            // 处理服务端错误
            if (!resp.success()) {
                log.error("创建飞书审批失败：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            return JSONObject.toJSONString(resp.getData());
        } catch (Exception e) {
            log.error(e.getMessage());
            return null;
        }
    }

    /**
     * 创建飞书客户端
     * 应用：王小卤链路中心
     */
    public Client getClientLUZX() {
        return Client.newBuilder(appId, appSecret).build();
    }
    /**
     * 创建飞书客户端
     * 应用：卤币学院
     */
    public Client getClientLBXY() {
        return Client.newBuilder(lbxyAppId, lbxyAppSecret).build();
    }

    /**
     * 创建飞书客户端
     * 应用：飞书表格更新机器人_T1
     */
    public String getBotToken1() {
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.put("app_id", botAppId1);
        bodyMap.put("app_secret", botAppSecret1);
        String bodyJson = JSONObject.toJSONString(bodyMap);

        String result = HttpUtil.createPost(TENANT_ACCESS_TOKEN_URL)
                .contentType(CONTENT_TYPE)
                .body(bodyJson)
                .execute().body();

        JSONObject resultJson = JSONObject.parseObject(result);
        int code = resultJson.getInteger("code");
        if (code != 0) {
            throw new RuntimeException("获取tenant_access_token错误");
        }
        return resultJson.getString("tenant_access_token");
    }

    public void appendStyle(String sheetToken, Map<String, Object> body, String autoToken) {
        // body上传
        String result = HttpUtil.createRequest(Method.PUT, SHEET_URL + sheetToken + STYLE)
                .auth("Bearer " + autoToken).contentType(CONTENT_TYPE)
                .body(JSONObject.toJSONString(body))
                .execute().body();
    }

    public JSONObject pullSheetOneRange(String sheetToken, String botToken1, String ranges) {
        return pullSheet(sheetToken, botToken1, ranges).getJSONObject(0);
    }

    public Employee getEmployeeReq(String employmentId) {
        String[] employmentIds = {employmentId};
        return getPeopleReq(employmentIds,null);
    }
    public Employee getPersonReq(String personId) {
        String[] personIds = {personId};
        return getPeopleReq(null,personIds);
    }
    /**
     * 批量查询员工信息
     * https://open.feishu.cn/document/corehr-v1/employee/batch_get?appId=cli_a8d8e0fc58d9100e
     */
    private Employee getPeopleReq(String[] employmentIds,String[] personIds) {

        try {
            Client client = getClientLUZX();
            // 创建请求对象
            BatchGetEmployeeReq req = BatchGetEmployeeReq.newBuilder()
                    .userIdType("people_corehr_id")
                    .batchGetEmployeeReqBody(BatchGetEmployeeReqBody.newBuilder()
                            .fields(new String[]{
                                    "employee_number",
                                    "person_info.legal_name",
                                    "department_id",
                                    "person_info.bank_account_list"
                            })
                            .employmentIds(employmentIds)
                            .personIds(personIds)
                            .build())
                    .build();

            BatchGetEmployeeResp resp = client.corehr().v2().employee().batchGet(req);
            // 处理服务端错误
            if (!resp.success()) {
                log.error("批量查询员工信息：code:{}，msg:{}，body{}",resp.getCode(),resp.getMsg(),new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            // 只查询一个员工
            Employee item = resp.getData().getItems()[0];
            return item;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }


    }

    public Offboarding searchOffboardings(String employmentId) {
        try {
            Client client = getClientLUZX();
            SearchOffboardingReq req = SearchOffboardingReq.newBuilder()
                    .pageSize(1)
                    .userIdType("people_corehr_id")
                    .searchOffboardingReqBody(SearchOffboardingReqBody.newBuilder()
                            .employmentIds(new String[]{employmentId})
                            .build())
                    .build();

            // 发起请求
            SearchOffboardingResp resp = client.corehr().v1().offboarding().search(req);
            // 处理服务端错误
            if (!resp.success()) {
                log.error("查询员工离职信息异常：code:{}，msg:{}，body{}", resp.getCode(), resp.getMsg(), new String(resp.getRawResponse().getBody(), UTF_8));
                return null;
            }
            Offboarding item = resp.getData().getItems()[0];
            return item;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

    }

    /**
     * 从审批信息字符串中提取核心业务值（产品名称、编码、箱规、供货价等）
     * @param approvalInfoStr 完整的审批信息JSON字符串（包含form字段的那个字符串）
     * @return 提取的核心值Map：key=字段名，value=字段值（已做类型转换）
     */
    public Map<String, Object> getApprovalFormValues(String approvalInfoStr) {
        // 定义返回的核心值Map，key用固定标识方便调用，value存对应值
        Map<String, Object> formValueMap = new HashMap<>();
        // 初始化默认值，避免空指针
        // 产品名称（字符串）
        formValueMap.put("productName", null);
        // 产品编码（字符串）
        formValueMap.put("productCode", null);
        // 产品规格（字符串）
        formValueMap.put("productSpec", null);
        // 2B供货价（BigDecimal）
        formValueMap.put("supplyPrice", null);
        // 标准成本（BigDecimal）
        formValueMap.put("standardCost", null);

        try {
            //  校验入参
            if (StringUtils.isBlank(approvalInfoStr)) {
                log.error("审批信息字符串为空，无法提取字段值");
                return formValueMap;
            }

            // 解析审批信息JSON字符串
            JSONObject approvalJson = JSONObject.parseObject(approvalInfoStr);

            //  提取form字段（字符串类型的JSON数组）并解析
            String formStr = approvalJson.getString("form");
            if (StringUtils.isBlank(formStr)) {
                log.error("审批信息中form字段为空，无法提取字段值");
                return formValueMap;
            }
            JSONArray formFields = JSONArray.parseArray(formStr);

            // 4遍历form数组，提取核心值（按字段名精准匹配）
            for (int i = 0; i < formFields.size(); i++) {
                JSONObject field = formFields.getJSONObject(i);
                String fieldName = field.getString("name");
                Object fieldValue = field.get("value");

                // 跳过空字段名/空值
                if (StringUtils.isBlank(fieldName) || fieldValue == null) {
                    continue;
                }

                // 按业务需求提取指定字段，做类型转换和容错
                switch (fieldName) {
                    case "产品名称":
                        formValueMap.put("productName", fieldValue.toString().trim());
                        break;
                    case "产品编码":
                        formValueMap.put("productCode", fieldValue.toString().trim());
                        break;
                    case "产品箱规":
                        try {
                            Integer spec = Integer.parseInt(fieldValue.toString().trim());
                            formValueMap.put("productSpec", spec);
                        } catch (NumberFormatException e) {
                            log.error("产品箱规格式错误，值为：{}", fieldValue);
                            formValueMap.put("productSpec", null);
                        }
                        break;
                    case "2B供货价/出厂价(与T100一致的最小售卖单元)":
                        try {
                            BigDecimal price = new BigDecimal(fieldValue.toString().trim());
                            formValueMap.put("supplyPrice", price);
                        } catch (NumberFormatException e) {
                            log.error("2B供货价格式错误，值为：{}", fieldValue);
                            formValueMap.put("supplyPrice", null);
                        }
                        break;
                    case "标准成本(与T100一致的最小售卖单元)":
                        try {
                            BigDecimal cost = new BigDecimal(fieldValue.toString().trim());
                            formValueMap.put("standardCost", cost);
                        } catch (NumberFormatException e) {
                            log.error("标准成本格式错误，值为：{}", fieldValue);
                            formValueMap.put("standardCost", null);
                        }
                        break;
                    // 如需提取其他字段，直接在case里添加即可
                    default:
                        break;
                }
            }

            log.info("审批信息字段提取完成，结果：{}", formValueMap);

        } catch (Exception e) {
            log.error("解析审批信息、提取字段值异常", e);
        }

        return formValueMap;
    }

    @Value("${feishu.group_robot.syncprice_webhook}")
    public void setSyncpriceWebhook(String syncpriceWebhook) {
        FeiShuUtil.syncpriceWebhook = syncpriceWebhook;
    }

    @Value("${feishu.group_robot.qince_webhook}")
    public void setQinceWebhook(String qinceWebhook) {
        FeiShuUtil.qinceWebhook = qinceWebhook;
    }

    /**
     * 根据 employmentId 查询员工编号（employee_no）
     * @param employmentId 员工雇佣ID
     * @return 员工编号，查询失败返回空字符串
     */
    public String getEmployeeNoByEmploymentId(String employmentId) {
        // 空值校验
        if (StringUtils.isBlank(employmentId)) {
            log.warn("employment_id 为空，无法查询员工编号");
            return "";
        }

        try {
            Employee employee = getEmployeeReq(employmentId);
            // 提取 employee_no（对应 Employee 中的 employeeNumber 字段）
            if (employee != null && StringUtils.isNotBlank(employee.getEmployeeNumber())) {
                return StringUtils.trimToEmpty(employee.getEmployeeNumber());
            } else {
                log.warn("根据 employment_id:{} 未查询到有效员工编号", employmentId);
                return "";
            }
        } catch (Exception e) {
            log.error("根据 employment_id:{} 查询员工编号异常", employmentId, e);
            return "";
        }
    }

    /**
     * 查询指定日期的所有飞书请假记录（自动分页，获取全部数据）
     * @param date 格式：yyyy-MM-dd
     * @return 封装了所有请假记录的JSON字符串（包含leave_request_list和total字段）
     */
    public String getAllLeaveRecords(String date) {
        List<LeaveRequest> allRecords = new ArrayList<>();
        String pageToken = "";
        do {
            try {
                Client client = getClientLUZX();
                LeaveRequestHistoryLeaveReq req = LeaveRequestHistoryLeaveReq.newBuilder()
                        .pageSize("100")
                        .pageToken(pageToken)
                        .leaveRequestStatus(LEAVE_STATUS)
                        .leaveUpdateTimeMin(date + " 00:00:00")
                        .leaveUpdateTimeMax(date + " 23:59:59")
                        .build();

                LeaveRequestHistoryLeaveResp resp = client.corehr().v1().leave().leaveRequestHistory(req);
                if (!resp.success()) {
                    log.error("分页获取请假记录失败：code={}, msg={}", resp.getCode(), resp.getMsg());
                    break;
                }
                LeaveRequestHistoryLeaveRespBody data = resp.getData();
                if (data != null && data.getLeaveRequestList() != null && data.getLeaveRequestList().length > 0) {
                    allRecords.addAll(Arrays.asList(data.getLeaveRequestList()));
                }
                pageToken = data.getPageToken();
            } catch (Exception e) {
                log.error("分页获取请假记录异常", e);
                break;
            }
        } while (StringUtils.isNotBlank(pageToken));

        JSONObject result = new JSONObject();
        // 直接存入扁平列表
        result.put("leave_request_list", allRecords);
        // 总数为实际记录数
        result.put("total", allRecords.size());
        return result.toJSONString();
    }
}
