package com.link.report.service.query.impl;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.link.report.config.ExtraQuickBiConfig;
import com.link.report.domain.dao.ExtraReportQuickbiEmbedDao;
import com.link.report.domain.entity.ExtraReportQuickbiEmbed;
import com.link.report.domain.wq.ExtraReportQuickbiEmbedWq;
import com.link.report.pojo.request.ExtraReportQuickbiEmbedBo;
import com.link.report.pojo.response.ExtraReportQuickbiEmbedVo;
import com.link.report.service.query.ExtraReportQuickbiEmbedService;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.redis.service.RedisService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * QuickBI报表嵌入信息服务实现类
 */
@Slf4j
@Service
public class ExtraReportQuickbiEmbedServiceImpl implements ExtraReportQuickbiEmbedService {


    @Resource
    private final ExtraReportQuickbiEmbedDao extraReportQuickbiEmbedDao;
    @Resource
    private ExtraQuickBiConfig extraQuickBiConfig;

    @Resource
    private RedisService cacheService;

    @Autowired
    public ExtraReportQuickbiEmbedServiceImpl(ExtraReportQuickbiEmbedDao extraReportQuickbiEmbedDao) {
        this.extraReportQuickbiEmbedDao = extraReportQuickbiEmbedDao;
    }

    /**
     * 新增QuickBI报表嵌入信息
     */
    @Override
    public ExtraReportQuickbiEmbedVo insert(ExtraReportQuickbiEmbedBo bo) {
        // 领域对象转换与业务逻辑处理
        ExtraReportQuickbiEmbed entity = convertBoToEntity(bo);
        entity.create(); // 调用领域对象行为

        // 通过DAO进行数据持久化
        extraReportQuickbiEmbedDao.insert(entity);

        return ExtraReportQuickbiEmbedVo.convertEntityToVo(entity);
    }

    /**
     * 更新QuickBI报表嵌入信息
     */
    @Override
    public ExtraReportQuickbiEmbedVo update(ExtraReportQuickbiEmbedBo bo) {
        ExtraReportQuickbiEmbed entity = extraReportQuickbiEmbedDao.selectById(bo.getId());
        if (entity == null) {
            throw new RuntimeException("记录不存在");
        }

        // 领域对象更新与业务逻辑处理
        entity.updateFromBo(bo); // 调用领域对象行为

        extraReportQuickbiEmbedDao.update(entity);
        return ExtraReportQuickbiEmbedVo.convertEntityToVo(entity);
    }

    /**
     * 根据ID查询QuickBI报表嵌入信息
     */
    @Override
    public ExtraReportQuickbiEmbedVo getById(Long id) {
        ExtraReportQuickbiEmbed entity = extraReportQuickbiEmbedDao.selectById(id);
        return entity != null ? ExtraReportQuickbiEmbedVo.convertEntityToVo(entity) : null;
    }

    /**
     * 根据pageId查询QuickBI报表嵌入信息
     */
    @Override
    public ExtraReportQuickbiEmbedVo getByPageId(String pageId) {
        ExtraReportQuickbiEmbedWq extraReportQuickbiEmbedWq = new ExtraReportQuickbiEmbedWq();
        extraReportQuickbiEmbedWq.setPageId(pageId);
        ExtraReportQuickbiEmbed entity = extraReportQuickbiEmbedDao.selectOne(extraReportQuickbiEmbedWq);
        return entity != null ? ExtraReportQuickbiEmbedVo.convertEntityToVo(entity) : null;
    }

    /**
     * 查询所有QuickBI报表嵌入信息列表
     */
    @Override
    public List<ExtraReportQuickbiEmbedVo> listAll() {
        List<ExtraReportQuickbiEmbed> entities = extraReportQuickbiEmbedDao.selectAll();
        return entities.stream()
                .map(ExtraReportQuickbiEmbedVo::convertEntityToVo)
                .collect(Collectors.toList());
    }

    /**
     * 删除QuickBI报表嵌入信息（软删除）
     */
    @Override
    public boolean deleteById(Long id) {
        ExtraReportQuickbiEmbed entity = extraReportQuickbiEmbedDao.selectById(id);
        if (entity == null) {
            return false;
        }

        entity.markAsDeleted(); // 调用领域对象行为
        extraReportQuickbiEmbedDao.update(entity);
        return true;
    }

    @Override
    public ExtraReportQuickbiEmbedVo queryAccessTicket(String pageId) {
        // 查询workId
        ExtraReportQuickbiEmbedWq wq = new ExtraReportQuickbiEmbedWq();
        wq.setPageId(pageId);
        ExtraReportQuickbiEmbed entity = extraReportQuickbiEmbedDao.selectOne(wq);
        if (entity == null) {
            log.error("记录不存在, pageId:{}", pageId);
            throw new ServiceException("记录不存在");
        }
        String workId = entity.getPageId();

        // 保存accessTicket
        String accessTicketNew = checkAccessTicketExist(workId);

        entity.setAccessTicket(accessTicketNew);
        extraReportQuickbiEmbedDao.update(entity);
        ExtraReportQuickbiEmbedVo vo = ExtraReportQuickbiEmbedVo.convertEntityToVo(entity);
        return vo;
    }

    /**
     * 检查accessTicket
     * 缓存中不存在则从QuickBI获取accessTicket
     *
     * @param workId
     * @return
     */
    private String checkAccessTicketExist(String workId) {
        // 从配置中获取appId和appSecret
        String accessId = extraQuickBiConfig.getAccessId();
        String accessKey = extraQuickBiConfig.getAccessKey();
        String aliyunUid = extraQuickBiConfig.getAliyunUid();
        // 从缓存中获取
        String accessTicketCache = cacheService.getCacheObject(getRedisKey(workId));
        if (accessTicketCache != null) {
            return accessTicketCache;
        }
        String url = String.format("https://bi.aliyun.com/openapi/ac3rd/ticket/create?worksId=%s&aliyunUid=%s&accessKey=%s&accessId=%s&validityTime=%s",
                workId, aliyunUid, accessKey, accessId, "240");
        String body = HttpUtil.createRequest(Method.GET, url)
                .execute()
                .body();
        log.info("获取accessTicket结果：{}", body);

        JSONObject jsonObject = JSONObject.parseObject(body);
        String code = jsonObject.getString("code");
        if (ObjectUtil.isEmpty(code) || !code.equals("0")) {
            throw new ServiceException("获取accessTicket失败");
        }
        String accessTicketNew = jsonObject.getJSONObject("data").getString("accessTicket");

        // 缓存更新
        cacheService.setCacheObject(getRedisKey(workId), accessTicketNew, 240L, TimeUnit.MINUTES);

        return accessTicketNew;
    }

    private String getRedisKey(String workId) {
        return "extra:report:quickbi:accessTicket:" + workId;
    }

    // 对象转换方法
    private ExtraReportQuickbiEmbed convertBoToEntity(ExtraReportQuickbiEmbedBo bo) {
        ExtraReportQuickbiEmbed entity = new ExtraReportQuickbiEmbed();
        // 设置属性值...
        entity.setName(bo.getName());
        entity.setPreviewUrl(bo.getPreviewUrl());
        entity.setPageId(bo.getPageId());
        entity.setAccessTicket(bo.getAccessTicket());
        entity.setReportType(bo.getReportType());
        return entity;
    }


    @Override
    public boolean saveBatch(Collection<ExtraReportQuickbiEmbed> entityList, int batchSize) {
        return false;
    }

    @Override
    public boolean saveOrUpdateBatch(Collection<ExtraReportQuickbiEmbed> entityList, int batchSize) {
        return false;
    }

    @Override
    public boolean updateBatchById(Collection<ExtraReportQuickbiEmbed> entityList, int batchSize) {
        return false;
    }

    @Override
    public boolean saveOrUpdate(ExtraReportQuickbiEmbed entity) {
        return false;
    }

    @Override
    public ExtraReportQuickbiEmbed getOne(Wrapper<ExtraReportQuickbiEmbed> queryWrapper, boolean throwEx) {
        return null;
    }

    @Override
    public Map<String, Object> getMap(Wrapper<ExtraReportQuickbiEmbed> queryWrapper) {
        return null;
    }

    @Override
    public <V> V getObj(Wrapper<ExtraReportQuickbiEmbed> queryWrapper, Function<? super Object, V> mapper) {
        return null;
    }

    @Override
    public BaseMapper<ExtraReportQuickbiEmbed> getBaseMapper() {

        return null;
    }
}
