package com.link.report.service.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.sfa.common.core.utils.ServletUtils;
import com.sfa.common.security.service.TokenService;
import com.sfa.common.security.utils.SecurityUtils;
import com.sfa.system.api.model.LoginUser;
import org.apache.commons.lang3.StringUtils;
import org.jeecg.modules.jmreport.api.JmReportTokenServiceI;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * @description: 自定义报表鉴权(如果不进行自定义, 则所有请求不做权限控制)
 * @author: wzlUp
 * @date: 2023/06/25 16:37
 * @motto: Done is better than perfect.
 */
@Component
public class JimuReportTokenServiceImpl implements JmReportTokenServiceI {

    @Autowired
    private TokenService tokenService;

    /**
     * 通过请求获取Token
     */
    @Override
    public String getToken(HttpServletRequest request) {
        String token = "";
        // 定时任务会为空
        if (request == null) {
            return "";
        }
        if (request != null) {
            token = request.getParameter("token");
            if (ObjectUtil.isEmpty(token)) {
                token = request.getHeader("X-Access-Token");
            }
        }
        String jmToken = request.getHeader("token");
        if (ObjectUtil.isEmpty(token) && ObjectUtil.isNotEmpty(jmToken)) {
            token = jmToken;
        }
        LoginUser loginUser = tokenService.getLoginUser(token);
        if (loginUser != null) {
            return token;
        }
        return "";
    }

    /**
     * 获取登录人用户名
     */
    @Override
    public String getUsername(String s) {
        LoginUser loginUser = tokenService.getLoginUser(s);
        return loginUser.getUsername();
    }

    @Override
    public String[] getRoles(String s) {
        LoginUser loginUser = tokenService.getLoginUser(s);
        if (ObjectUtil.isEmpty(loginUser)) {
            return new String[0];
        }
        Set<String> roles = loginUser.getRoles();
        // it可以设计任何报表
        if (!ObjectUtil.isAllNotEmpty(loginUser.getSysUser(), loginUser.getSysUser().getDept()
                , loginUser.getSysUser().getDept().getDeptName()) &&
                // 信息中心可以编辑所有报表
                loginUser.getSysUser().getDept().getDeptName().contains("信息技术中心")) {
            roles.add("admin");
        }
        return roles.toArray(new String[0]);
    }

    /**
     * Token校验
     */
    @Override
    public Boolean verifyToken(String s) {
        HttpServletRequest request = ServletUtils.getRequest();
        String requestURI = request.getRequestURI();
        // 排除分享页面的一些不需要校验的请求url
        if (requestURI.contains("/jmreport/dictCodeSearch") ||
                requestURI.contains("/jmreport/query/multiple/initValue")) {
            return true;
        }
        if (s != null && s.length() > 0) {
            LoginUser loginUser = tokenService.getLoginUser(s);
            return loginUser != null;
        }
        return false;
    }


    /**
     * 自定义请求头
     */
    @Override
    public HttpHeaders customApiHeader() {
        HttpHeaders header = new HttpHeaders();
        header.add("X-Access-Token", SecurityUtils.getToken());
        return header;
    }

    /**
     * 获取多租户id
     *
     * @return tenantId
     */
    @Override
    public String getTenantId() {
        HttpServletRequest request = ServletUtils.getRequest();
        // 定时任务会为空
        if (null == request) {
            return "NO";
        }
        String token = request.getParameter("token");
        String header = request.getHeader("X-Access-Token");
        LoginUser loginUser = null;
        if (StringUtils.isNoneBlank(token)) {
            loginUser = tokenService.getLoginUser(token);
        } else if (StringUtils.isNotBlank(header)) {
            loginUser = tokenService.getLoginUser(header);
        } else {
            //都不具备则不能访问
            return "NO";
        }
        Long deptId = loginUser.getSysUser().getDept().getDeptId();
        return String.valueOf(deptId);
    }

    @Override
    public Map<String, Object> getUserInfo(String token) {
        // 将所有信息存放至map 解析sql会根据map的键值解析,可自定义其他值
        Map<String, Object> map = new HashMap<>(20);
        LoginUser loginUser = tokenService.getLoginUser(token);
        map.put("sysUserCode", loginUser.getUsername());
        //设置当前日期（年月日）
        map.put("sysData", DateUtil.date());
        //设置昨天日期（年月日）
        map.put("sysYesterDay", DateUtil.yesterday());
        //设置当前登录用户昵称
        map.put("sysUserName", loginUser.getSysUser().getNickName());
        //设置当前登录用户部门ID
        map.put("deptId", loginUser.getSysUser().getDeptId());
        //设置当前登录用户描述
//        map.put("describe", loginUser.getSysUser().getDept().getDescribes());
        map.put("describe", loginUser.getSysUser().getDept().getRemark());
        return map;
    }

    public String[] getPermissions(String token) {
        LoginUser loginUser = tokenService.getLoginUser(token);
        if (loginUser == null) {
            return new String[0];
        }
        Set<String> permissions = loginUser.getPermissions();
        if (ObjectUtil.isNotEmpty(permissions)) {
            permissions.add("drag:datasource:testConnection");
            permissions.add("onl:drag:clear:recoverry");
            permissions.add("drag:analysis:sql");
            permissions.add("drag:design:getTotalData");
            return permissions.toArray(new String[0]);
        }

        return new String[0];
    }

}
