package com.link.report.interceptor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;

@Slf4j
@Component
public class LoginInterceptor implements HandlerInterceptor {
    // 域名信息
    @Value("${jeecg.jmreport.domainUrl}")
    private String domainUrl;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        // 1. 检查用户是否已登录从header中获取token X-token
        String token = request.getHeader("token");
        if (token == null) {
            String xtoken = request.getHeader("X-token");
            if (xtoken != null) {
                token = xtoken;
            }
        }
        String requestURI = request.getRequestURI();
        log.info("requestURI = " + requestURI);
        // 2. 如果未登录，则重定向到登录页
        if (token == null) {
            Cookie[] cookies = request.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    if (cookie.getName().equals("Admin-Token")) {
                        token = cookie.getValue();
                        // TODO 加上鉴权和权限认证

                        // 已登录，继续执行
                        return true;
                    }
                }
            }
            // 未登录，重定向到登录页面
            log.info("未登录，重定向到登录页面");
            if(domainUrl  == null ){
                domainUrl = getCurrentUrl(request);
            }
//            String encode = URLEncodeUtil.encode(requestURI);
            String encodedUrl = URLEncoder.encode(requestURI, "UTF-8");

            response.sendRedirect(domainUrl + "/#/login?redirect=" + encodedUrl);  // 重定向到登录页面
            return false;
        }

        return true;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {
        // 可选的后续处理
        log.info("postHandle 方法被调用");
        log.info("modelAndView = " + modelAndView);
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        // 可选的最终处理
        log.info("afterCompletion 方法被调用");
    }

    // 工具方法：获取当前请求的完整URL（包含参数）
    private String getCurrentUrl(HttpServletRequest request) {
        StringBuffer url = request.getRequestURL();  // 如：http://example.com/user/profile
        String queryString = request.getQueryString();  // 如：id=1&name=test
        if (queryString != null) {
            url.append("?").append(queryString);  // 拼接参数：http://example.com/user/profile?id=1&name=test
        }
        return url.toString();
    }
}
