package com.link.hub.service.weChatMiniProgram.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.link.hub.config.TencentMapConfig;
import com.link.hub.service.weChatMiniProgram.TencentMapService;
import com.sfa.common.core.exception.ServiceException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

/**
 * 腾讯地图业务层
 */
@Slf4j
@Service
public class TencentMapServiceImpl implements TencentMapService {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 根据经纬度查询地理位置
     * @param latitude 纬度（如：39.984154）
     * @param longitude 经度（如：116.307490）
     * @param getPoi 是否返回周边POI（1=返回，0=不返回）
     * @return 腾讯地图地理信息响应
     * @throws Exception 调用异常/解析异常
     */
    public JSONObject getLocationByLngLat(Double latitude, Double longitude, Integer getPoi)  {
        // 1. 参数校验
        if (latitude == null || longitude == null) {
            throw new IllegalArgumentException("纬度/经度不能为空");
        }
        if (getPoi == null) {
            getPoi = 0; // 默认不返回POI
        }

        // 2. 拼接请求URL（带参数）
        String location = latitude + "," + longitude;
        String requestUrl = UriComponentsBuilder.fromHttpUrl(TencentMapConfig.TENCENT_MAP_GEOCODER_URL)
                .queryParam("location", location)
                .queryParam("key", TencentMapConfig.TENCENT_MAP_KEY)
                .queryParam("get_poi", getPoi)
                .toUriString();

        // 3. 发送GET请求调用腾讯地图API
        ResponseEntity<String> responseEntity;
        try {
            responseEntity = restTemplate.getForEntity(requestUrl, String.class);
        } catch (RestClientException e) {
            log.error(String.format("调用腾讯地图API失败：%s", e.getMessage()), e);
            throw new ServiceException("调用腾讯地图API失败：" + e.getMessage());
        }

        // 4. 解析JSON响应
        String responseBody = responseEntity.getBody();
        if (responseBody == null) {
            log.error("腾讯地图API返回空结果");
            throw new ServiceException("腾讯地图API返回空结果");
        }
        JSONObject geoResponse = JSON.parseObject(responseBody);

        // 5. 校验API返回状态
        if (geoResponse.getInteger("status") != 0) {
            log.error("腾讯地图API调用失败:{}（状态码：{}）", geoResponse.getString("message"), geoResponse.getInteger("status"));
            throw new ServiceException("腾讯地图API调用失败：" + geoResponse.getString("message") + "（状态码：" + geoResponse.getInteger("status") + "）");
        }

        return geoResponse;
    }
}
