package com.link.hub.service.officeAccount.impl;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.link.hub.config.WeChatOfficeAccountConfig;
import com.link.hub.domain.weChatMiniProgram.entity.WechatMiniProgramUser;
import com.link.hub.domain.weChatOfficialAccount.entity.WechatOfficialAccountUser;
import com.link.hub.pojo.mp.dto.WeChatMpUserDto;
import com.link.hub.pojo.oa.dto.WeChatOaUserQueryDto;
import com.link.hub.pojo.oa.dto.WechatOaUserDto;
import com.link.hub.pojo.oa.dto.WechatOaUserResDto;
import com.link.hub.service.officeAccount.WechatOfficeAccountService;
import com.link.hub.service.officeAccount.WechatOfficialAccountUserService;
import com.link.hub.service.weChatMiniProgram.WechatMiniProgramUserService;
import com.link.hub.util.WeChatPlatFormUtils;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.redis.service.RedisService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;


/**
 * 小程序服务实现
 */
@Slf4j
@Service
public class WechatOfficeAccountServiceServiceImpl implements WechatOfficeAccountService {
    @Resource
    private WeChatOfficeAccountConfig officeAccountConfig;
    @Resource
    private RedisService redisService;
    @Resource
    private WechatOfficialAccountUserService oaUserService;
    @Resource
    private WechatMiniProgramUserService miniProgramUserService;
    @Resource
    private WeChatOfficeAccountConfig accountConfig;


    /**
     * 检查用户是否关注公众号
     * @param openid
     * @return
     */
    @Override
    public Boolean checkIsSubscribe(String openid) {

        // 使用 openid openid查询小程序用户
        WeChatMpUserDto dto = new WeChatMpUserDto();
        dto.setOpenid(openid);
        WechatMiniProgramUser user = miniProgramUserService.queryMiniProgramUserOne(dto);

        log.info("unionid: {}", user);
        if (user == null || user.getUnionid() == null) {
            log.error("用户未关联小程序，openid: {}", openid);
            throw new ServiceException("小程序用户错误");
        }
        //  使用unionid查询微信公众号用户
        WeChatOaUserQueryDto oaUserDto = new WeChatOaUserQueryDto();
        oaUserDto.setUnionid(user.getUnionid());
        WechatOfficialAccountUser wechatOfficialAccountUser = oaUserService.selectOne(oaUserDto);
        if (wechatOfficialAccountUser == null) {
            log.error("用户还未关注公众号，openid: {},unionid: {}", openid,user.getUnionid());
            return false;
        }
        if(wechatOfficialAccountUser.getSubscribeStatus() == 0){
            log.error("用户未关注公众号,状态是未关注，openid: {},unionid: {}", openid,user.getUnionid());
            return false;
        }

        return wechatOfficialAccountUser.getSubscribeStatus() == 1;
    }



    @Override
    public List<String> getUserList(String nextOpenId) {
        try {
            String accessToken = WeChatPlatFormUtils.getStableAccessToken(accountConfig.getAppId(), accountConfig.getAppSecret());
            String url = "https://api.weixin.qq.com/cgi-bin/user/get?access_token=" + accessToken;
            if (nextOpenId != null && !nextOpenId.isEmpty()) {
                url += "&next_openid=" + nextOpenId;
            }

            String result = HttpUtil.get(url);

            log.info("WeChat Official Account user list  ");

            JSONObject jsonObject = JSON.parseObject(result);
            String total = jsonObject.get("total").toString();
            Integer count = jsonObject.getInteger("count");
            String nextOpenIdNew = jsonObject.getString("next_openid");
            if (count == 0) {
                log.info("没有更多用户了，total: {}，count: {}", total, count);
                return new ArrayList<>();
            }

            JSONObject dataJson = jsonObject.getJSONObject("data");
            List<String> openids = dataJson.getList("openid", String.class);

            log.info("total: {},返回{}条", total, openids.size());
            return openids;
        } catch (Exception e) {
            log.error("Failed to get official account user list", e);
            throw new RuntimeException("获取公众号用户列表失败", e);
        }
    }

    @Override
    public Integer saveUserList() {
        boolean flag = true;
        String nextOpenId = null;
        // 从数据库查询最大id的openid
        WechatOfficialAccountUser maxIdUser = oaUserService.selectMaxIdUser();

        // 从数据库查询最大id的openid
        if (maxIdUser != null) {
            nextOpenId = maxIdUser.getOpenid();
        }
        Integer count = 0;

        while (flag) {
            List<String> openIdList = getUserList(nextOpenId);
            if (openIdList == null || openIdList.isEmpty()) {
                flag = false;
                break;
            }
            count += openIdList.size();
            // 保存入库 分批次按照1000条处理一次
            int batchSize = 1000;
            int offset = 0;

            for (int i = offset; i < openIdList.size(); i += batchSize) {
                List<String> opeIds = openIdList.subList(offset, Math.min(offset + batchSize, openIdList.size()));
                List<WechatOaUserDto> officialAccountUsers = new ArrayList<>();

                for (int j = 0; j < opeIds.size(); j++) {
                    String userOpenid = opeIds.get(j);
                    WechatOaUserDto wechatOfficialAccountUser = WechatOaUserDto.create(userOpenid);
                    officialAccountUsers.add(wechatOfficialAccountUser);
                }
                oaUserService.saveOrUpdateBatch(officialAccountUsers);
                offset += batchSize;
            }


            flag = true;
            nextOpenId = openIdList.get(openIdList.size() - 1);
        }
        return count;


    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<WechatOaUserDto> saveUserDetail(Integer pageNum, Integer pageSize) {
        // 分页批量获取用户基本信息
        List<WechatOaUserDto> userDtos = oaUserService.selectPageList(pageNum, pageSize);
        if (userDtos == null || userDtos.isEmpty()) {
            return userDtos;
        }
        // 请求微信 批量返回用户基本信息
        List<WechatOaUserResDto> userInfoBatch = WeChatPlatFormUtils.getUserInfoBatch(userDtos, officeAccountConfig.getAppId(), officeAccountConfig.getAppSecret());


        oaUserService.updateUserInfoBatch(userInfoBatch);
        return userDtos;
    }

    @Override
    public void saveAllUserDetail() {
        // 分页批量获取用户基本信息
        Integer pageNum = 1;
        Integer pageSize = 100;
        while (true) {
            List<WechatOaUserDto> userDtos = saveUserDetail(pageNum, pageSize);
            if (userDtos == null || userDtos.isEmpty()) {
                break;
            }
            pageNum++;

            // 请求微信 批量返回用户基本信息
            List<WechatOaUserResDto> userInfoBatch = WeChatPlatFormUtils.getUserInfoBatch(userDtos, officeAccountConfig.getAppId(), officeAccountConfig.getAppSecret());
            oaUserService.updateUserInfoBatch(userInfoBatch);
        }
    }
}
