package com.link.hub.domain.weChatOfficialAccount.dao.impl;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.link.hub.domain.weChatOfficialAccount.dao.WechatOfficialAccountUserDao;
import com.link.hub.domain.weChatOfficialAccount.entity.WechatOfficialAccountUser;
import com.link.hub.domain.weChatOfficialAccount.mapper.WechatOfficialAccountUserMapper;
import com.link.hub.domain.weChatOfficialAccount.mq.WeChatOaUserQueryWq;
import com.link.hub.pojo.oa.dto.WechatOaUserDto;
import com.link.hub.pojo.oa.dto.WechatOaUserResDto;
import com.sfa.common.core.web.domain.PageInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * 微信公众号用户 DAO 实现类
 */
@Repository
public class WechatOfficialAccountUserDaoImpl implements WechatOfficialAccountUserDao {

    private final WechatOfficialAccountUserMapper userMapper;

    @Autowired
    public WechatOfficialAccountUserDaoImpl(WechatOfficialAccountUserMapper userMapper) {
        this.userMapper = userMapper;
    }

    @Override
    public WechatOfficialAccountUser getUserByOpenId(String openId) {
        return userMapper.selectByOpenId(openId);
    }

    @Override
    public boolean saveOrUpdateUser(WechatOfficialAccountUser user) {
        if (user.getId() == null) {
            // 新增用户
            return userMapper.insert(user) > 0;
        } else {
            // 更新用户
            return userMapper.updateById(user) > 0;
        }
    }

    @Override
    public void saveBatch(List<WechatOaUserDto> officialAccountUsers) {
        // 批量保存 1000条保存一次
        int batchSize = 1000;
        for (int i = 0; i < officialAccountUsers.size(); i += batchSize) {
            List<WechatOaUserDto> batch = officialAccountUsers.subList(i, Math.min(i + batchSize, officialAccountUsers.size()));
            for (WechatOaUserDto userDto : batch) {
                WechatOfficialAccountUser user = WechatOfficialAccountUser.convert(userDto);
                userMapper.insert(user);
            }
        }

    }

    /**
     * 批量更新
     * 通过openid更新
     *
     * @param officialAccountUsers
     */
    @Override
    public void updateBatch(List<WechatOaUserDto> officialAccountUsers) {
        // 批量保存 1000条保存一次
        int batchSize = 1000;
        for (int i = 0; i < officialAccountUsers.size(); i += batchSize) {
            List<WechatOaUserDto> batch = officialAccountUsers.subList(i, Math.min(i + batchSize, officialAccountUsers.size()));
            for (WechatOaUserDto userDto : batch) {

                WechatOfficialAccountUser user = WechatOfficialAccountUser.convert(userDto);
                userMapper.update(user, new LambdaQueryWrapper<WechatOfficialAccountUser>()
                        .eq(WechatOfficialAccountUser::getOpenid, user.getOpenid()));
            }
        }

    }

    @Override
    public List<WechatOfficialAccountUser> selectByOpenIds(Set<String> allOpenIds) {

        return userMapper.selectList(new LambdaQueryWrapper<WechatOfficialAccountUser>()
                .in(WechatOfficialAccountUser::getOpenid, allOpenIds));
    }

    @Override
    public List<WechatOfficialAccountUser> selectList() {
        // 分页查询
        LambdaQueryWrapper<WechatOfficialAccountUser> wrapper = new LambdaQueryWrapper<WechatOfficialAccountUser>()
                .last("limit 10")
                .orderByDesc(WechatOfficialAccountUser::getId);
        List<WechatOfficialAccountUser> officialAccountUsers = userMapper.selectList(wrapper);
        return officialAccountUsers;
    }

    @Override
    public PageInfo selectPageList(Integer pageNum, Integer pageSize) {
        // 分页查询
        LambdaQueryWrapper<WechatOfficialAccountUser> wrapper = new LambdaQueryWrapper<WechatOfficialAccountUser>();
        wrapper.isNotNull(WechatOfficialAccountUser::getUnionid);
        wrapper.orderByDesc(WechatOfficialAccountUser::getId);
        Page<WechatOfficialAccountUser> pageRes = new Page<>(pageNum, pageSize);
        IPage<WechatOfficialAccountUser> page = userMapper.selectPage(pageRes, wrapper);
        PageInfo pageInfo = new PageInfo<>(page);
        return pageInfo;
    }

    @Override
    public void updateUserInfoBatch(List<WechatOaUserResDto> userInfoBatchs) {

        for (int i = 0; i < userInfoBatchs.size(); i++) {
            WechatOaUserResDto userInfo = userInfoBatchs.get(i);
            WechatOfficialAccountUser user = WechatOfficialAccountUser.convertByUserInfo(userInfo);
            userMapper.update(user, new LambdaQueryWrapper<WechatOfficialAccountUser>()
                    .eq(WechatOfficialAccountUser::getOpenid, user.getOpenid()));
        }
    }

    @Override
    public WechatOfficialAccountUser selectOne(WeChatOaUserQueryWq wq) {
        LambdaQueryWrapper<WechatOfficialAccountUser> wrapper = buildWrapper(wq);
        WechatOfficialAccountUser wechatOfficialAccountUser = userMapper.selectOne(wrapper);
        return wechatOfficialAccountUser;
    }

    private LambdaQueryWrapper<WechatOfficialAccountUser> buildWrapper(WeChatOaUserQueryWq wq) {
        LambdaQueryWrapper<WechatOfficialAccountUser> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(ObjectUtil.isNotEmpty(wq.getUnionid()),WechatOfficialAccountUser::getUnionid, wq.getUnionid());

        queryWrapper.eq(ObjectUtil.isNotEmpty(wq.getOpenid()),WechatOfficialAccountUser::getOpenid, wq.getOpenid());
        return queryWrapper;
    }
}
