package com.link.hub.domain.weChatMiniProgram.dao.impl;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.link.hub.domain.weChatMiniProgram.dao.LotteryParticipateDao;
import com.link.hub.domain.weChatMiniProgram.entity.LotteryParticipate;
import com.link.hub.domain.weChatMiniProgram.mapper.LotteryParticipateMapper;
import com.link.hub.domain.weChatMiniProgram.wq.LotteryParticipateWq;
import com.link.hub.pojo.mp.dto.LotteryDto;
import com.link.hub.pojo.mp.dto.LotteryParticipateDto;
import com.link.hub.pojo.mp.vo.LotteryVo;
import com.sfa.common.core.exception.ServiceException;
import org.springframework.stereotype.Repository;

import javax.annotation.Resource;
import java.util.List;


/**
 * LotteryParticipateDaoImpl
 * 抽奖参与记录数据访问层实现类
 * 提供抽奖参与记录的数据库操作实现，包括查询列表和我的参与记录查询
 *
 * @author 吕本才
 * @version 1.0
 * @since 2025年12月10日
 */
@Repository
public class LotteryParticipateDaoImpl implements LotteryParticipateDao {
    @Resource
    private LotteryParticipateMapper lotteryParticipateMapper;

    @Override
    public List<LotteryParticipate> queryList(LotteryParticipateWq wq) {
        LambdaQueryWrapper<LotteryParticipate> wrapper = new LambdaQueryWrapper<>();

        if (wq.getActivityId() != null) {
            wrapper.eq(LotteryParticipate::getActivityId, wq.getActivityId());
        }
        if (wq.getUserId() != null) {
            wrapper.eq(LotteryParticipate::getUserId, wq.getUserId());
        }
        if (wq.getParticipateTimeStart() != null && wq.getParticipateTimeEnd() != null) {
            wrapper.between(LotteryParticipate::getParticipateTime, wq.getParticipateTimeStart(), wq.getParticipateTimeEnd());
        }
        if (wq.getSource() != null) {
            wrapper.eq(LotteryParticipate::getSource, wq.getSource());
        }
        wrapper.eq(LotteryParticipate::getDeleteFlag, wq.getDeleteFlag());

        return lotteryParticipateMapper.selectList(wrapper);
    }



    @Override
    public boolean checkUserParticipated(Long activityId, Long userId) {
        LambdaQueryWrapper<LotteryParticipate> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LotteryParticipate::getActivityId, activityId)
                .eq(LotteryParticipate::getUserId, userId)
                .eq(LotteryParticipate::getDeleteFlag, 0);
        return lotteryParticipateMapper.selectCount(wrapper) > 0;
    }

    @Override
    public int getUserParticipateCount(Long userId) {
        return 0;
    }

    /**
     * 检查用户是否已参与活动（利用唯一索引 uk_activity_user）
     */
    @Override
    public boolean hasParticipated(Long activityId, Long userId) {
        QueryWrapper<LotteryParticipate> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(LotteryParticipate::getActivityId, activityId)
                .eq(LotteryParticipate::getUserId, userId)
                // 仅查询未删除记录
                .eq(LotteryParticipate::getDeleteFlag, 0);
        return lotteryParticipateMapper.selectCount(queryWrapper) > 0;
    }

    /**
     * 保存参与记录（插入前自动校验唯一性）
     */
    @Override
    public LotteryParticipate saveParticipate(LotteryParticipateDto participateDto) {

        // MyBatis Plus 的 insert 方法会自动处理唯一索引冲突
        LotteryParticipate participatePO = LotteryParticipate.create(participateDto);
        int flag = lotteryParticipateMapper.insert(participatePO);
        if (flag == 0) {
            throw new ServiceException("保存参与记录失败");
        }
        return participatePO;
    }

    /**
     * 查询具体参与记录
     */
    @Override
    public LotteryParticipate getParticipateRecord(Long activityId, Long userId) {
        QueryWrapper<LotteryParticipate> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(LotteryParticipate::getActivityId, activityId)
                .eq(LotteryParticipate::getUserId, userId)
                .eq(LotteryParticipate::getDeleteFlag, 0);
        return lotteryParticipateMapper.selectOne(queryWrapper);
    }

    /**
     * 统计活动参与人数
     */
    @Override
    public long countParticipateByActivity(Long activityId) {
        QueryWrapper<LotteryParticipate> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(LotteryParticipate::getActivityId, activityId)
                .eq(LotteryParticipate::getDeleteFlag, 0);
        return lotteryParticipateMapper.selectCount(queryWrapper);
    }


    /**
     * 查询我的抽奖信息
     * @return
     */
    @Override
    public LotteryDto queryLotteryInfo(LotteryDto dto) {
        // 查询我的抽奖信息
        LambdaQueryWrapper<LotteryParticipate> wrapper =new LambdaQueryWrapper<LotteryParticipate>();
        wrapper.eq(ObjectUtil.isNotEmpty(dto.getActivityId()),LotteryParticipate::getActivityId, dto.getActivityId());
        wrapper.eq(LotteryParticipate::getUserId, dto.getUserId());

        LotteryParticipate lotteryParticipate = lotteryParticipateMapper.selectOne(wrapper);
        if (lotteryParticipate == null) {
            return null;
        }
        LotteryDto myLotteryDto = LotteryDto.convert(lotteryParticipate);
        return myLotteryDto;
    }



    @Override
    public List<LotteryVo> announcementList() {
        // 获取抽奖公示信息 包含了活动信息 获奖详情信息

        return null;
    }

}
