package com.link.bi.service.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.github.pagehelper.PageInfo;
import com.link.bi.domain.dao.IFinanceOrderDetailDao;
import com.link.bi.domain.entity.FinanceOrderDetail;
import com.link.bi.domain.wq.FinanceCostWq;
import com.link.bi.pojo.request.FinanceCostVo;
import com.link.bi.pojo.response.FinanceBiListDto;
import com.link.bi.pojo.response.FinanceExportBiListDto;
import com.link.bi.pojo.response.FinanceExportOrderDetailListDto;
import com.link.bi.pojo.response.FinanceOrderDetailListDto;
import com.link.bi.service.FinanceOrderDetailService;
import com.sfa.common.core.utils.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaTypeFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.MimeType;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * 旺店通订单明细Service业务层处理
 *
 * @author lvbencai
 * @date 2025年01月07日17:00:16
 */
@Slf4j
@Service
public class FinanceOrderDetailServiceImpl implements FinanceOrderDetailService {

    @Autowired
    IFinanceOrderDetailDao detailDao;


    //    @Autowired
//    private Executor defaultExecutor;
    @Override
    public PageInfo<FinanceBiListDto> selectBiFinanceCostList(FinanceCostVo financeCostVo) {
        PageInfo<FinanceBiListDto> pageInfo = new PageInfo<>();
        // 多线程，设置不同线程的参数
        List<Date[]> dates = DateUtils.splitTimeInterval(financeCostVo.getStartDate(), financeCostVo.getEndDate());
        if (dates.size() == 0) {
            return pageInfo;
        }
        if (dates.size() == 1) {
            FinanceCostWq qw = covertFinanceCostWq(financeCostVo);
            PageInfo<FinanceOrderDetail> orderDetailPageInfo = detailDao.selectBiFinanceCostList(qw);
            BeanUtils.copyProperties(orderDetailPageInfo, pageInfo);
            pageInfo.setList(orderDetailPageInfo.getList().stream().map(item -> {
                FinanceBiListDto dto = new FinanceBiListDto();
                BeanUtils.copyProperties(item, dto);
                dto.setActualCostSum(item.getSumActualCostSum());
                dto.setStandardCostSum(item.getSumStandardCostSum());
                dto.setShareAmountSum(item.getShareAmountSum().setScale(2, BigDecimal.ROUND_HALF_UP));
                dto.setActualCostGrossProfitSum(dto.getShareAmountSum().subtract(dto.getActualCostSum()).setScale(2, BigDecimal.ROUND_HALF_UP));
                dto.setStandardCostGrossProfitSum(dto.getShareAmountSum().subtract(dto.getStandardCostSum()).setScale(2, BigDecimal.ROUND_HALF_UP));
                dto.setUid(IdUtil.simpleUUID());
                return dto;
            }).collect(Collectors.toList()));
            return pageInfo;
        }
        if (dates.size() > 1) {
            // 多线程查询
            // 创建一个线程池
            ExecutorService executorService = Executors.newFixedThreadPool(20);
            // 创建一个任务列表
            List<Callable<PageInfo<FinanceOrderDetail>>> tasks = new ArrayList<>();

            final CountDownLatch endLock = new CountDownLatch(dates.size()); // 假设我们有多个查询条件，每个条件对应一个任务
            BlockingQueue<Future<PageInfo<FinanceOrderDetail>>> queue = new LinkedBlockingQueue<>();


            PageInfo<FinanceOrderDetail> orderDetailPageInfo = new PageInfo<>();
            long total = 0;
            int pages = 1;

            List<FinanceOrderDetail> aggregatedResult = new ArrayList<>();
            try {
                for (int i = 0; i < dates.size(); i++) {
                    FinanceCostWq wq = covertFinanceCostThreadWq(financeCostVo, dates, i);
                    Future<PageInfo<FinanceOrderDetail>> future = executorService.submit(new Callable<PageInfo<FinanceOrderDetail>>() {
                        @Override
                        public PageInfo<FinanceOrderDetail> call() throws Exception {
                            PageInfo<FinanceOrderDetail> detailPageInfo = detailDao.selectBiFinanceCostList(wq);
                            endLock.countDown();
                            log.info("{} has finished the job!");
                            return detailPageInfo;
                        }
                    });
                    queue.add(future);
                }

                endLock.await();
                log.info("endLock.await()结束-" + queue.size());

                // 汇聚结果
                for (Future<PageInfo<FinanceOrderDetail>> future : queue) {
                    //  获取结果
                    PageInfo<FinanceOrderDetail> financeOrderDetailPageInfo = future.get();
                    orderDetailPageInfo = financeOrderDetailPageInfo;
                    pages = Math.max(pages, financeOrderDetailPageInfo.getPages());
                    total = Math.max(total, financeOrderDetailPageInfo.getTotal());
                    aggregatedResult.addAll(financeOrderDetailPageInfo.getList());
                }

                BeanUtils.copyProperties(orderDetailPageInfo, pageInfo);
                pageInfo.setPages(pages);
                pageInfo.setTotal(total);
                // 分组求和aggregatedResult
                Set<String> preSet = new HashSet<>();
                Map<String, FinanceBiListDto> existMap = new HashMap<>();
                for (int i = 0; i < aggregatedResult.size(); i++) {
                    FinanceOrderDetail item = aggregatedResult.get(i);
                    // 判断几个字段 组合
                    String key = getKey(item, financeCostVo);
                    if (preSet.contains(key)) {
                        // 直接继续求和
                        FinanceBiListDto dto = existMap.get(key);
                        dto.setShareAmountSum(dto.getShareAmountSum().add(item.getShareAmountSum()));
                        dto.setActualCostSum(dto.getActualCostSum().add(item.getSumActualCostSum()));
                        dto.setStandardCostSum(dto.getStandardCostSum().add(item.getSumStandardCostSum()));
                    } else {
                        FinanceBiListDto dto = new FinanceBiListDto();
                        BeanUtils.copyProperties(item, dto);
                        dto.setActualCostSum(item.getSumActualCostSum());
                        dto.setStandardCostSum(item.getSumStandardCostSum());
                        dto.setUid(IdUtil.simpleUUID());
                        existMap.put(key, dto);
                        preSet.add(key);
                    }
                }
                List<FinanceBiListDto> valueList = new ArrayList<>(existMap.values());
                pageInfo.setList(valueList);
                // 计算实际成本毛利 标准成本毛利
                pageInfo.setList(valueList.stream().map(item -> {
                    FinanceBiListDto dto = new FinanceBiListDto();
                    BeanUtils.copyProperties(item, dto);
                    dto.setShareAmountSum(item.getShareAmountSum().setScale(2, BigDecimal.ROUND_HALF_UP));
                    dto.setActualCostGrossProfitSum(dto.getShareAmountSum().subtract(dto.getActualCostSum()).setScale(2, BigDecimal.ROUND_HALF_UP));
                    dto.setStandardCostGrossProfitSum(dto.getShareAmountSum().subtract(dto.getStandardCostSum()).setScale(2, BigDecimal.ROUND_HALF_UP));
                    dto.setUid(IdUtil.simpleUUID());
                    return dto;
                }).collect(Collectors.toList()));
                log.info("完成数据整合");
            } catch (InterruptedException | ExecutionException e) {
                e.printStackTrace();
                log.error(e.getMessage(), e);
            } catch (Exception e) {
                e.printStackTrace();
                log.error(e.getMessage(), e);
            } finally {
//            endLock.countDown();
                // 关闭线程池
                executorService.shutdown();
            }
        }


        return pageInfo;
    }

    private String getKey(FinanceOrderDetail item, FinanceCostVo financeCostVo) {
        //根据入参条件决定key
        String key = "";
        if (ObjectUtil.equals(Boolean.TRUE, financeCostVo.getZbjQdTypeAll()) || ObjectUtil.isNotEmpty(financeCostVo.getZbjQdType())) {
            key = key + "#" + item.getZbjQdType();
        }
        if (ObjectUtil.equals(Boolean.TRUE, financeCostVo.getFlavorAll()) || ObjectUtil.isNotEmpty(financeCostVo.getFlavor())) {
            key = key + "#" + item.getFlavorErp();
        }
        if (ObjectUtil.equals(Boolean.TRUE, financeCostVo.getSpecNameAll()) || ObjectUtil.isNotEmpty(financeCostVo.getSpecName())) {
            key = key + "#" + item.getSpecNameErp();
        }
        if (ObjectUtil.equals(Boolean.TRUE, financeCostVo.getSeriesAll()) || ObjectUtil.isNotEmpty(financeCostVo.getSeries())) {
            key = key + "#" + item.getSeries();
        }
        if (ObjectUtil.isEmpty(key)) {
            key = item.getZbjQdType();
        }
        return key;
    }

    private FinanceCostWq covertFinanceCostWq(FinanceCostVo financeCostVo) {
        FinanceCostWq qw = new FinanceCostWq();
        BeanUtils.copyProperties(financeCostVo, qw);
        // 初始化开始日期和结束日期
        if (ObjectUtil.isAllNotEmpty(qw.getStartDate(), qw.getEndDate())) {
            qw.setStartDate(DateUtils.dateStart(DateUtils.dateStart(qw.getStartDate())));
            qw.setEndDate(DateUtils.dateStart(DateUtils.dateStart(DateUtils.addDays(qw.getEndDate(), 1))));
        } else {
            //默认 t-1 天的数据查询
            qw.setEndDate(DateUtils.dateStart(new Date()));
            qw.setStartDate(DateUtils.dateStart(DateUtils.addDays(new Date(), -1)));
        }
        return qw;
    }

    /**
     * 转换成
     *
     * @param financeCostVo
     * @param intervals
     * @param i
     * @return
     */
    private FinanceCostWq covertFinanceCostThreadWq(FinanceCostVo financeCostVo, List<Date[]> intervals, int i) {
        FinanceCostWq qw = new FinanceCostWq();
        BeanUtils.copyProperties(financeCostVo, qw);
        // 初始化开始日期和结束日期
        if (ObjectUtil.isNotEmpty(intervals)) {
            qw.setStartDate(DateUtils.dateStart(DateUtils.dateStart(intervals.get(i)[0])));
            qw.setEndDate(DateUtils.dateStart(DateUtils.dateStart(DateUtils.addDays(intervals.get(i)[1], 1))));
        } else {
            //默认 t-1 天的数据查询
            qw.setEndDate(DateUtils.dateStart(new Date()));
            qw.setStartDate(DateUtils.dateStart(DateUtils.addDays(new Date(), -1)));
        }
        return qw;
    }

    @Override
    public PageInfo<FinanceOrderDetailListDto> selectBiFinanceCostDetailPage(FinanceCostVo financeCostVo) {
        FinanceCostWq qw = covertFinanceCostWq(financeCostVo);
        PageInfo<FinanceOrderDetail> financeOrderDetailPageInfo = detailDao.selectBiFinanceCostDetailList(qw);

        PageInfo<FinanceOrderDetailListDto> pageInfo = new PageInfo<>();
        BeanUtils.copyProperties(financeOrderDetailPageInfo, pageInfo);
        pageInfo.setList(financeOrderDetailPageInfo.getList().stream().map(item -> {
            FinanceOrderDetailListDto dto = new FinanceOrderDetailListDto();
            BeanUtils.copyProperties(item, dto);
            // 转换数据
            // 交易时间 毫秒转时间
            dto.setTradeTime(DateUtil.format(new Date(Long.parseLong(dto.getTradeTime())), "yyyy-MM-dd HH:mm:ss"));
            // 订单状态 转换 96 成本确认  110 已完成
            if (ObjectUtil.isNotEmpty(dto.getTradeStatus())) {
                if (dto.getTradeStatus().equals("96")) {
                    dto.setTradeStatus("成本确认");
                } else if (dto.getTradeStatus().equals("110")) {
                    dto.setTradeStatus("已完成");
                }
            } else {
                dto.setTradeStatus("-");
            }

            // 货品类型 确认是否是从产品基础信息表中获取  包材-泡沫箱 包材-气泡袋 包材-纸箱 产品 成品-主营 成品-组合包 物料-常规 物料-联名等
            return dto;
        }).collect(Collectors.toList()));

        return pageInfo;
    }

    /**
     * 导出数据
     */
    @Override
    public void exportBiFinanceCostList(FinanceCostVo financeCostVo, HttpServletResponse response) {
        // 区分类型，不同的数据导出 策略模式
        if (financeCostVo.getType().equals("1")) {
            financeCostVo.setTypeName("达人分类");
            this.exportBiFinanceCostData(financeCostVo, response);
        } else if (financeCostVo.getType().equals("2")) {
            financeCostVo.setTypeName("分类");
            this.exportBiFinanceCostData(financeCostVo, response);
        } else if (financeCostVo.getType().equals("3")) {
            financeCostVo.setTypeName("订单明细");
            this.exportBiFinanceCostOrderDetailData(financeCostVo, response);
        }
    }

    private void exportBiFinanceCostOrderDetailData(FinanceCostVo financeCostVo, HttpServletResponse response) {
        // 导出查询的数据
        financeCostVo.setPageNum(1);
        financeCostVo.setPageSize(1000000);
        PageInfo<FinanceOrderDetailListDto> financeBiListDtoPageInfo = this.selectBiFinanceCostDetailPage(financeCostVo);
        // easyExcel导出数据
        String fileNameOri = String.format("王小卤成本核算%s-%s%s信息-%s.xlsx", DateUtil.format(financeCostVo.getStartDate(), "yyyyMMdd"), DateUtil.format(financeCostVo.getEndDate(), "yyyyMMdd"), financeCostVo.getTypeName(), DateUtil.format(new Date(), "MMdd"));
        // 文件名中文名需要转义
        String fileName = null;
        try {
            fileName = URLEncoder.encode(fileNameOri, "UTF-8");

            String contentType = MediaTypeFactory.getMediaType(fileName).map(MimeType::toString).orElse("application/vnd.ms-excel");
            response.setContentType(contentType);
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
            // 这里需要设置不关闭流
            response.setHeader("Connection", "close");
            // 表头策略
            WriteCellStyle headWriteCellStyle = new WriteCellStyle();
            //设置背景颜色
            headWriteCellStyle.setFillForegroundColor(IndexedColors.WHITE.getIndex());
            headWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
            WriteFont headWriteFont = new WriteFont();
            headWriteFont.setFontHeightInPoints((short) 11);
            headWriteCellStyle.setWriteFont(headWriteFont);
            // 设置表头边框样式
            headWriteCellStyle.setBorderLeft(BorderStyle.THIN);
            headWriteCellStyle.setBorderTop(BorderStyle.THIN);
            headWriteCellStyle.setBorderRight(BorderStyle.THIN);
            headWriteCellStyle.setBorderBottom(BorderStyle.THIN);
            // 设置表头边框颜色
            headWriteCellStyle.setLeftBorderColor(IndexedColors.BLACK.getIndex());
            headWriteCellStyle.setTopBorderColor(IndexedColors.BLACK.getIndex());
            headWriteCellStyle.setRightBorderColor(IndexedColors.BLACK.getIndex());
            headWriteCellStyle.setBottomBorderColor(IndexedColors.BLACK.getIndex());

            //内容策略
            WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
            WriteFont contentWriteFont = new WriteFont();
            // 设置内容边框样式
            contentWriteCellStyle.setBorderLeft(BorderStyle.THIN);
            contentWriteCellStyle.setBorderTop(BorderStyle.THIN);
            contentWriteCellStyle.setBorderRight(BorderStyle.THIN);
            contentWriteCellStyle.setBorderBottom(BorderStyle.THIN);
            // 设置内容边框颜色
            contentWriteCellStyle.setLeftBorderColor(IndexedColors.BLACK.getIndex());
            contentWriteCellStyle.setTopBorderColor(IndexedColors.BLACK.getIndex());
            contentWriteCellStyle.setRightBorderColor(IndexedColors.BLACK.getIndex());
            contentWriteCellStyle.setBottomBorderColor(IndexedColors.BLACK.getIndex());

            // 字体大小
            contentWriteFont.setFontHeightInPoints((short) 10);
            contentWriteCellStyle.setWriteFont(contentWriteFont);

            //设置 水平居中
            contentWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
            HorizontalCellStyleStrategy horizontalCellStyleStrategy = new HorizontalCellStyleStrategy(headWriteCellStyle, contentWriteCellStyle);

            Collection<?> dataList = financeBiListDtoPageInfo.getList();
            EasyExcel.write(response.getOutputStream(), FinanceExportOrderDetailListDto.class).autoCloseStream(Boolean.TRUE).excelType(ExcelTypeEnum.XLSX).registerWriteHandler(horizontalCellStyleStrategy).registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())//自动列宽策略
                    .sheet(financeCostVo.getTypeName())
                    //获取数据填充
                    .doWrite(dataList);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 下载达人分类下的数据
     *
     * @param financeCostVo
     * @param response
     */
    private void exportBiFinanceCostData(FinanceCostVo financeCostVo, HttpServletResponse response) {
        // 导出查询的数据
        financeCostVo.setPageNum(1);
        financeCostVo.setPageSize(1000000);
        PageInfo<FinanceBiListDto> financeBiListDtoPageInfo = this.selectBiFinanceCostList(financeCostVo);

        // easyExcel导出数据
        String fileNameOri = String.format("王小卤成本核算%s-%s%s-%s.xlsx", DateUtil.format(financeCostVo.getStartDate(), "yyyyMMdd"), DateUtil.format(financeCostVo.getEndDate(), "yyyyMMdd"), financeCostVo.getTypeName(), DateUtil.format(new Date(), "MMdd"));
        // 文件名中文名需要转义
        String fileName = null;
        try {
            fileName = URLEncoder.encode(fileNameOri, "UTF-8");

            String contentType = MediaTypeFactory.getMediaType(fileName).map(MimeType::toString).orElse("application/vnd.ms-excel");
            response.setContentType(contentType);
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
            // 这里需要设置不关闭流
            response.setHeader("Connection", "close");
            // 表头策略
            WriteCellStyle headWriteCellStyle = new WriteCellStyle();
            //设置背景颜色
            headWriteCellStyle.setFillForegroundColor(IndexedColors.WHITE.getIndex());
            headWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
            WriteFont headWriteFont = new WriteFont();
            headWriteFont.setFontHeightInPoints((short) 11);
            headWriteCellStyle.setWriteFont(headWriteFont);
            // 设置表头边框样式
            headWriteCellStyle.setBorderLeft(BorderStyle.THIN);
            headWriteCellStyle.setBorderTop(BorderStyle.THIN);
            headWriteCellStyle.setBorderRight(BorderStyle.THIN);
            headWriteCellStyle.setBorderBottom(BorderStyle.THIN);
            // 设置表头边框颜色
            headWriteCellStyle.setLeftBorderColor(IndexedColors.BLACK.getIndex());
            headWriteCellStyle.setTopBorderColor(IndexedColors.BLACK.getIndex());
            headWriteCellStyle.setRightBorderColor(IndexedColors.BLACK.getIndex());
            headWriteCellStyle.setBottomBorderColor(IndexedColors.BLACK.getIndex());

            //内容策略
            WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
            WriteFont contentWriteFont = new WriteFont();
            // 设置内容边框样式
            contentWriteCellStyle.setBorderLeft(BorderStyle.THIN);
            contentWriteCellStyle.setBorderTop(BorderStyle.THIN);
            contentWriteCellStyle.setBorderRight(BorderStyle.THIN);
            contentWriteCellStyle.setBorderBottom(BorderStyle.THIN);
            // 设置内容边框颜色
            contentWriteCellStyle.setLeftBorderColor(IndexedColors.BLACK.getIndex());
            contentWriteCellStyle.setTopBorderColor(IndexedColors.BLACK.getIndex());
            contentWriteCellStyle.setRightBorderColor(IndexedColors.BLACK.getIndex());
            contentWriteCellStyle.setBottomBorderColor(IndexedColors.BLACK.getIndex());

            // 字体大小
            contentWriteFont.setFontHeightInPoints((short) 10);
            contentWriteCellStyle.setWriteFont(contentWriteFont);
            //设置 水平居中
            contentWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
            HorizontalCellStyleStrategy horizontalCellStyleStrategy = new HorizontalCellStyleStrategy(headWriteCellStyle, contentWriteCellStyle);

            Collection<?> dataList = financeBiListDtoPageInfo.getList();
            EasyExcel.write(response.getOutputStream(), FinanceExportBiListDto.class).autoCloseStream(Boolean.TRUE).excelType(ExcelTypeEnum.XLSX).registerWriteHandler(horizontalCellStyleStrategy).registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())//自动列宽策略
                    .sheet(financeCostVo.getTypeName())
                    //获取数据填充
                    .doWrite(dataList);
        } catch (UnsupportedEncodingException e) {
            log.error("编码异常" + e.getMessage(), e);
            throw new RuntimeException(e);
        } catch (IOException e) {
            log.error("IO异常" + e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }
}
