package com.link.bi.config.listener;


import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.link.bi.domain.entity.EcGoodsSupplyDemand;
import com.link.bi.pojo.response.PrdInfoDto;
import com.link.bi.service.IEcGoodsSupplyDemandService;
import com.sfa.common.core.exception.ServiceException;
import com.sfa.common.core.utils.DateUtils;
import com.sfa.common.core.utils.StringUtils;
import com.sfa.common.core.utils.poi.ExcelUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.sql.Date;
import java.text.ParseException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

// 定义Excel数据处理的监听器
@Data
@Slf4j
public class EcGoodsSupplyDemandListener extends AnalysisEventListener<Map<Integer, String>> {
    private static final int ROW_BATCH_SIZE = 10;
    private static final int DEFAULT_INDEX = 14;
    private static int total = 0;
    // 存储前五行的数据
    private List<String[]> titleHeaders = new ArrayList<>();

    // 存储解析后的完整数据
    private List<Map<Integer, String>> dataList = new ArrayList<>(ROW_BATCH_SIZE);
    // 存储每批次的完整数据
    private List<EcGoodsSupplyDemand> demandList = new ArrayList<>(ROW_BATCH_SIZE);

    // 存储错误记录信息
    private List<String> errorInfos = ListUtils.newArrayList();
    private List<String> warnInfos = ListUtils.newArrayList();

    private IEcGoodsSupplyDemandService service;
    private List<PrdInfoDto> prdInfoDtos;
    private Integer year;

    public EcGoodsSupplyDemandListener(IEcGoodsSupplyDemandService service, List<PrdInfoDto> prdInfoDtos, Integer year) {
        this.service = service;
        this.year = year;
        this.prdInfoDtos = prdInfoDtos;
    }

    /**
     * 前4行需要放到表头
     *
     * @param headMap
     * @param context
     */
    @Override
    public void invokeHeadMap(Map<Integer, String> headMap, AnalysisContext context) {
        //  可以在此处验证表头
//        if(ObjectUtil.isEmpty(headMap.get(0))){
//            throw new ServiceException("第一行表头日期错误");
//        }
        // 将前五行的数据设置到当前行数据对象中
        String[] rowData = new String[headMap.size()];
        for (Map.Entry<Integer, String> entry : headMap.entrySet()) {
            rowData[entry.getKey()] = entry.getValue();
        }
        titleHeaders.add(rowData);
    }

    /**
     * 读取Excel数据时的回调方法
     * 每行都会都一次 这里可以处理每行数据，包括使用标题头的数据进行计算
     * 例如，您可以在这里根据标题头的数据来计算或修改data对象的属性
     *
     * @param rowData
     * @param context
     */
    @Override
    public void invoke(Map<Integer, String> rowData, AnalysisContext context) {

        // 可以在这里对数据进行一些预处理或验证
        ReadRowHolder readRowHolder = context.readRowHolder();
        Integer rowNumber = readRowHolder.getRowIndex() + 1;

        // 校验 商品编码-短码是否有值 --关联查询出系列
        String prdCode = rowData.get(1);
        if (ObjectUtil.isNotEmpty(prdCode) && !NumberUtil.isNumber(prdCode)) {
            errorInfos.add("第 " + rowNumber + "行商品编码格式不正确");
            return;
        }

        // 校验 格式是否是数字
        Integer startYear = year;
        if (ObjectUtil.isEmpty(year)) {
            startYear = DateUtil.thisYear();
        }
        // 根据入参计算开始的列？？？ 根据titleHeaders的年月计算开始保存修改的列
        Integer indexStart_ByTitleHeaders = getIndexByTitleHeaders(titleHeaders, startYear,DEFAULT_INDEX);

        Integer endYear = startYear + 1;
        Integer indexEnd_ByTitleHeaders = getIndexByTitleHeaders(titleHeaders, endYear, indexStart_ByTitleHeaders);
        for (int i = indexStart_ByTitleHeaders; i < indexEnd_ByTitleHeaders; i++) {
            if (ObjectUtil.isNotEmpty(rowData.get(i)) && !NumberUtil.isNumber(rowData.get(i))) {
                errorInfos.add("第 " + rowNumber + "行第 " + i + " 列数据格式不正确");
            }

        }
        // 校验渠道列：平台  1 天猫; 2 抖音; 3 拼多多; 4 京东; 5 私域; 6 分销; 7 线下; 8 样品; 9 团购; 10 其他;
//        String channel = "1天猫; 2抖音; 3拼多多;4京东;5私域;6分销;7线下;8样品;9团购;10其他;";
//        if(ObjectUtil.isNotEmpty(rowData.get(11)) && channel.contains(rowData.get(11))){
//            errorInfos.add("第 "+rowNumber  + "行第 "+11+" 列数据格式不正确");
//            throw new ServiceException("第 "+rowNumber  + "行第 "+11+" 列数据格式不正确");
//        }
        // 将当前行数据添加到列表中
        dataList.add(rowData);

    }
    /**
     * 解析完成后回调方法
     * @param analysisContext
     */
    @Override
    public void doAfterAllAnalysed(AnalysisContext analysisContext) {
        log.info("数据解析完成,dataList.size:{}", dataList.size());
        if(warnInfos.size()>0){
            log.warn("警告记录:{}", JSONUtil.toJsonStr(warnInfos));
        }
        // 存储错误记录
        if(errorInfos.size()>0){
            log.error("解析错误记录:{}", JSONUtil.toJsonStr(errorInfos));
            throw new ServiceException("解析错误:"+JSONUtil.toJsonStr(errorInfos));
        }
        // 分批次入库数据
        if (!dataList.isEmpty()) {
            // 先删除
            service.deleteByYear(year);
            // 批量插入数据
            List<Map<Integer, String>> convertList = new ArrayList();
            for (int i = 0; i < dataList.size(); i++) {
                convertList.add(dataList.get(i));
                if (convertList.size() >= ROW_BATCH_SIZE) {
                    saveDemandList(convertList);
                    convertList.clear();
                }
            }
            if (convertList.size() > 0) {
                saveDemandList(convertList);
                convertList.clear();
            }

        }

    }


    /**
     * 保存数据
     * @param dataList
     */
    private void saveDemandList(List<Map<Integer, String>> dataList) {
        // 转换成实体类
        List<EcGoodsSupplyDemand> demandArrayList = new ArrayList<>();
        try {
            demandArrayList = convert(dataList, titleHeaders);
            demandArrayList.addAll(demandArrayList);
            demandList.addAll(demandArrayList);
        } catch (ParseException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
//        service.saveBatch(demandArrayList, demandArrayList.size());
        service.batchSaveEcGoodsSupplyDemands(demandArrayList);
        total += demandArrayList.size();
        log.info("已处理{}条数据", total);
    }

    private List<EcGoodsSupplyDemand> convert(List<Map<Integer, String>> dataList, List<String[]> titleHeaders) throws ParseException {
        List<EcGoodsSupplyDemand> demands = new ArrayList<>();

        // 转换成实体类
        for (int i = 0; i < dataList.size(); i++) {
            Map<Integer, String> rowDataMap = dataList.get(i);
            // 为啥rowDataMap.size()  titleHeaders  是270列？ 但是242列后面都是空的
            // 校验 格式是否是数字
            Integer startYear = year;
            if (ObjectUtil.isEmpty(year)) {
                startYear = DateUtil.thisYear();
            }
            // 根据入参计算开始的列？？？ 根据titleHeaders的年月计算开始保存修改的列
            Integer indexStart_ByTitleHeaders = getIndexByTitleHeaders(titleHeaders, startYear,DEFAULT_INDEX);

            Integer endYear = startYear + 1;
            Integer indexEnd_ByTitleHeaders = getIndexByTitleHeaders(titleHeaders, endYear, indexStart_ByTitleHeaders);

            // 前14列公共使用  从15列开始
            for (int j = indexStart_ByTitleHeaders; j < indexEnd_ByTitleHeaders; j++) {
                // 表头是空的，不再取数据了 记录一下日志  不记录到表格了吧
                if (ObjectUtil.isEmpty(titleHeaders.get(0)[j])) {
                    log.error(j + "列表头为空，不再导入当前列数据");
                    warnInfos.add(ExcelUtil.convertToExcelColumn(j) + "列表头为空");
                    continue;
                }
                EcGoodsSupplyDemand demand = new EcGoodsSupplyDemand();

                Long value = Optional.ofNullable(rowDataMap.get(j))
                        .filter(ObjectUtil::isNotEmpty)
                        .map(str -> str.replace(",", "").replace("(", "-").replace(")", ""))
                        .map(Long::parseLong)
                        .orElse(null);
                // demand取出integerStringMap的值，设置各个字段的值
                demand.setSupplyDemandAmount(value);

                // 类型 设置 1年 2月 3周 4日
                demand.setReportDateType(getSupplyDemandType(titleHeaders, j));
                // 周类型的 设置周数
                if (ObjectUtil.isNotEmpty(demand.getReportDateType())) {
                    if (demand.getReportDateType().equals(3L)) {
                        DateTimeFormatter inputFormatter = DateTimeFormatter.ofPattern("yyyy/M/d");
                        demand.setReportDateBegin(Date.valueOf(LocalDate.parse(titleHeaders.get(1)[j], inputFormatter)));
                        demand.setReportDateEnd(Date.valueOf(LocalDate.parse(titleHeaders.get(1)[j], inputFormatter)));
                        demand.setWeek(DateUtils.calculateWeekNumber(LocalDate.parse(titleHeaders.get(1)[j], inputFormatter)));
                        demand.setYear(getYearFromHead(titleHeaders, j, demand.getReportDateType()));
                        demand.setMonth(getMonthFromHead(titleHeaders, j, demand.getReportDateType()));
                    }
                    if (demand.getReportDateType().equals(2L)) {
                        demand.setMonth(getMonthFromHead(titleHeaders, j, demand.getReportDateType()));
                        demand.setYear(getYearFromHead(titleHeaders, j, demand.getReportDateType()));
                    }
                    if (demand.getReportDateType().equals(1L)) {
                        demand.setYear(getYearFromHead(titleHeaders, j, demand.getReportDateType()));
                    }
                }
                // 关联产品信息表
                demand.setPlatform(rowDataMap.get(11));
                demand.setPlatformId(StringUtils.getNumberFromStr(rowDataMap.get(11)));
                demand.setPrdCode(rowDataMap.get(1));
                demand.setPrdName(rowDataMap.get(12));
                // 暂时不存
                demand.setPrdStatus(1L);

                // 同步商品表
                demand.setSeries(getSeriesByPrdCode(demand.getPrdCode()));
                demand.setSeriesId(getSeriesIdByPrdCode(demand.getPrdCode()));
                // 69
                demand.setPrdBarcode(rowDataMap.get(0));
                demands.add(demand);
            }
        }
        return demands;
    }

    /**
     * 根据年份获取目标年份的数据下标
     * @param titleHeaders
     * @param targetYear
     * @param startIndex
     * @return
     */
    private Integer getIndexByTitleHeaders(List<String[]> titleHeaders, Integer targetYear, Integer startIndex) {
        // 根据入参的年份 只考虑当年的数据更新和插入
        int result = startIndex;
        for (int j = startIndex; j < titleHeaders.get(0).length; j++) {
            if (ObjectUtil.isEmpty(titleHeaders.get(0)[j]) || titleHeaders.get(0)[j].contains(String.valueOf(targetYear))) {
                result = j;
                break;
            }
        }
        return result;
    }

    /**
     * 从产品信息中获取大类信息
     * @param prdCode
     * @return
     */
    private String getSeriesByPrdCode(String prdCode) {
        for (int i = 0; i < prdInfoDtos.size(); i++) {
            if (prdInfoDtos.get(i).getPrdCode().equals(prdCode)) {
                return prdInfoDtos.get(i).getSeries();
            }
        }
        return null;
    }
    /**
     * 从产品信息中获取大类Id信息
     * @param prdCode
     * @return
     */
    private Integer getSeriesIdByPrdCode(String prdCode) {
        for (int i = 0; i < prdInfoDtos.size(); i++) {
            if (prdInfoDtos.get(i).getPrdCode().equals(prdCode)) {
                return prdInfoDtos.get(i).getSeriesId();
            }
        }
        return null;
    }


    /**
     * 从表头信息中获取月份信息
     * @param titleHeaders
     * @param j
     * @param reportDateType
     * @return
     */
    private Long getMonthFromHead(List<String[]> titleHeaders, int j, Long reportDateType) {
        // 获取月份    统计日期类型：1年 2月 3周 4日
        if (reportDateType.equals(2L)) {
            String dateStr = titleHeaders.get(1)[j];
            return Long.valueOf(dateStr.replace("月", ""));
        }
        String dateStr = titleHeaders.get(0)[j];
        if (reportDateType.equals(3L)) {
            return Long.valueOf(dateStr.substring(4, 6));
        }
        return null;
    }
    /**
     * 从表头信息中获取年份信息
     * @param titleHeaders
     * @param j
     * @param reportDateType
     * @return
     */
    private Long getYearFromHead(List<String[]> titleHeaders, int j, Long reportDateType) {
        // 获取年份    统计日期类型：1年 2月 3周 4日
        String dateStr = titleHeaders.get(0)[j];
        if (reportDateType.equals(2L) || reportDateType.equals(1L)) {
            return Long.valueOf(dateStr);
        }
        if (reportDateType.equals(3L)) {
            return Long.valueOf(dateStr.substring(0, 4));
        }
        return null;
    }

    /**
     * 根据表头获取统计日期类型
     * 统计日期类型：1年 2月 3周 4日
     * @param titleHeaders
     * @param j
     * @return
     */
    private Long getSupplyDemandType(List<String[]> titleHeaders, int j) {
        // 设置当前的类型  周  月类型的数据暂时不用
        String s1 = titleHeaders.get(0)[j];
        String startStr = titleHeaders.get(1)[j];
        if (ObjectUtil.isNotEmpty(s1)) {
            if (s1.length() == 4) {
                if (ObjectUtil.isNotEmpty(startStr) && startStr.contains("月")) {
                    // 如果是月份
                    return 2L;
                } else {
                    // 如果是年份
                    return 1L;
                }
            } else if (s1.length() == 6 && ObjectUtil.isNotEmpty(startStr)) {
                return 3L;
            }
        }
        log.error("当前titleHeaders列:{},没有匹配的类型", j);
        return null;
    }
}
