package com.wangxiaolu.export.service.impl;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.wangxiaolu.export.mapper.FeishuSheetRecordMapper;
import com.wangxiaolu.export.mapper.entity.FeishuSheetRecordDO;
import com.wangxiaolu.export.mapper.entity.TemporaryActivityClockDO;
import com.wangxiaolu.export.mapper.entity.TemporaryActivityPhotoDO;
import com.wangxiaolu.export.mapper.entity.TemporaryActivityReportedDO;
import com.wangxiaolu.export.service.ActivityToFeishuSheetService;
import com.wangxiaolu.export.util.FeishuSheetUtils;
import com.wangxiaolu.promotion.common.redis.RedisKeys;
import com.wangxiaolu.promotion.common.redis.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.swing.*;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : liqiulin
 * @date : 2024-07-29 15
 * @describe :
 */
@Slf4j
@Service
public class ActivityToFeishuSheetServiceImpl implements ActivityToFeishuSheetService {

    private static FeishuSheetRecordDO sheetInfoD = null;

    @Value("${feishu.activity_robot_1.app_id}")
    private String appId1;
    @Value("${feishu.activity_robot_1.app_secret}")
    private String appSecret1;
    @Value("${feishu.activity_robot_2.app_id}")
    private String appId2;
    @Value("${feishu.activity_robot_2.app_secret}")
    private String appSecret2;

    @Autowired
    RedisCache redisCache;
    @Autowired
    FeishuSheetUtils feishuSheetUtils;
    @Autowired
    FeishuSheetRecordMapper feishuSheetRecordMapper;

    @Override
    public void activityDataToFeishuSheet(List<TemporaryActivityReportedDO> reportedDos, Map<Long, List<TemporaryActivityPhotoDO>> activityPhotos, Map<Long, TemporaryActivityClockDO> clockMap, Map<String, List<TemporaryActivityPhotoDO>> clockPhotoMap) {
        sheetInfoD = null;
        // 查询要更新的电子表格、工作表，每天凌晨开始更新，所以获取前一天日期
        String month = DateUtil.format(DateUtil.yesterday(), "yyyy-MM");
        FeishuSheetRecordDO sheetInfo = feishuSheetRecordMapper.selectOneByCreateMonth(month);
        if (Objects.isNull(sheetInfo)) {
            log.error("电子文档token信息获取失败，取消上传");
            return;
        }
        sheetInfoD = sheetInfo;

        log.info("上传至电子文档中的活动记录共「{}」条", reportedDos.size());
        int index = 1;

        for (TemporaryActivityReportedDO rdo : reportedDos) {
            log.info("------- 上传标号{} start -------", index);
            try {
                // [temporaryId]-[createDate]： 做唯一key
                String clockKey = rdo.getTemporaryId() + "-" + rdo.getCreateDate();
                List<TemporaryActivityPhotoDO> temporaryActivityPhotoDOS = activityPhotos.get(rdo.getId());
                TemporaryActivityClockDO clockDo = clockMap.get(rdo.getId());
                List<TemporaryActivityPhotoDO> activityClock = clockPhotoMap.containsKey(clockKey) ? clockPhotoMap.get(clockKey) : new ArrayList<>();
                activityDataToSheetRow(rdo, temporaryActivityPhotoDOS, activityClock,clockDo);
            } catch (Exception e) {
                log.error("上传标号{}报错，跳过此次上传，上传数据：{}", index,JSONObject.toJSONString(rdo));
            }

            log.info("------- 上传标号{} end -------", index);
            ++index;
        }
    }

    /**
     * 组装单个活动记录，对应表格中的一行数据
     *
     * @param rdo                       活动数据
     * @param temporaryActivityPhotoDOS 活动中涉及到的照片（推广、POS）
     * @param activityClock（打卡照片）
     */
    private void activityDataToSheetRow(TemporaryActivityReportedDO rdo, List<TemporaryActivityPhotoDO> temporaryActivityPhotoDOS, List<TemporaryActivityPhotoDO> activityClock,TemporaryActivityClockDO clockDo) throws Exception {
        List<Object> row = new ArrayList<>();
        // 证明标识（请忽略），一定要在开头第一个单元格有数据，才能保证之后的数据即使缺失也不会串行
        // A-C：证明标识（请忽略） 问题标识（请忽略）	促销员姓名
        row.add("0");
        row.add(rdo.getId());
        row.add(rdo.getTemporaryId() + "-" + rdo.getTemporaryName());

        // 月份
        String createDate = rdo.getCreateDate();
        String[] dateArr = createDate.split("-");

        // 战区
        row.add(rdo.getOrgName());

        // 城市经理
        row.add(rdo.getApproveName());

        // 执行城市
        row.add(rdo.getCity());

        // 执行日期
        row.add(dateArr[1] + "月" + dateArr[2] + "日");

        // 场次
        row.add("1");

        // 经销商
        row.add(rdo.getDealerName());

        // 系统名称
        row.add(rdo.getLineName());

        // 门店名称
        row.add(rdo.getStoreName());

        // 活动形式
        row.add(rdo.getActivityPattern());

        // 是否提交审核
        row.add("APPROVED".equals(rdo.getApproveStatus()) ? "已审核" : "未提交");

        // 推广活动销额(元) N
        row.add("0");

        // 上传文本数据
        addactivityDataRow(row);

        String rowNumStr = redisCache.get(getRowNumRedisKey());
        if (StringUtils.isBlank(rowNumStr)) {
            log.error("行标识[{}]为未发生改变，当前行不做图片处理，上传数据：{}", rowNumStr, row.toString());
            return;
        }


        String rangFormat = sheetInfoD.getSheetId() + "!%s" + rowNumStr + ":%s" + rowNumStr;

        // S-V: 上班打卡时间/地点	午休下班打卡时间/地点	午休上班打卡时间/地点	下班打卡时间/地点
        if (Objects.nonNull(clockDo)){
            List<Object> rowClockData = new ArrayList<>();
            if (Objects.nonNull(clockDo.getClockInTime()) && Objects.nonNull(clockDo.getClockOutTime())){
                long tl = DateUtil.between(clockDo.getClockInTime(), clockDo.getClockOutTime(), DateUnit.MINUTE);
                long hour = tl / 60;
                long minute = tl % 60;
                rowClockData.add(String.valueOf(hour) + minute);
            }else {
                rowClockData.add("0");
            }
            rowClockData.add("【"+DateUtil.formatDateTime(clockDo.getClockInTime())+"】" + clockDo.getClockInAddress());
            rowClockData.add("【"+DateUtil.formatDateTime(clockDo.getNoonClockOutTime())+"】" + clockDo.getNoonClockOutAddress());
            rowClockData.add("【"+DateUtil.formatDateTime(clockDo.getNoonClockInTime())+"】" + clockDo.getNoonClockInAddress());
            rowClockData.add("【"+DateUtil.formatDateTime(clockDo.getClockOutTime())+"】" + clockDo.getClockOutAddress());
            String range =String.format(rangFormat, "S", "W");
            updateClockData(range,rowClockData);
        }

        /**
         * 获取推广、POS图片
         */
        Map<Integer, List<TemporaryActivityPhotoDO>> photoTypeMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(temporaryActivityPhotoDOS)) {
            photoTypeMap = temporaryActivityPhotoDOS.stream().collect(Collectors.groupingBy(TemporaryActivityPhotoDO::getType));
        }


        /**
         * 推广照>> 真实pos证明反馈
         */
        List<TemporaryActivityPhotoDO> photosType7 = Objects.isNull(photoTypeMap.get(7)) ? new ArrayList<>() : photoTypeMap.get(7);
        int photosType7Size = photosType7.size();

        // 真实pos证明反馈1 >>>
        if (!(photosType7Size < 1 || Objects.isNull(photosType7.get(0)))) {
            valuesImageToSheet(String.format(rangFormat, "O", "O"), photosType7.get(0).getPhotoUrl());
        }

        // 真实pos证明反馈2 >>>
        if (!(photosType7Size < 2 || Objects.isNull(photosType7.get(1)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "P", "P"), photosType7.get(1).getPhotoUrl());
        }

        // 真实pos证明反馈3 >>>
        if (!(photosType7Size < 3 || Objects.isNull(photosType7.get(2)))) {
            valuesImageToSheet(String.format(rangFormat, "Q", "Q"), photosType7.get(2).getPhotoUrl());
        }

        // 真实pos证明反馈4 >>>
        if (!(photosType7Size < 4 || Objects.isNull(photosType7.get(3)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "R", "R"), photosType7.get(3).getPhotoUrl());
        }
        log.info("上传pos照片完成，行号：{}", rowNumStr);

        /**
         * 打卡照片
         */

        Map<Integer, String> clockMap = activityClock.stream().collect(Collectors.toMap(TemporaryActivityPhotoDO::getType, TemporaryActivityPhotoDO::getPhotoUrl));
        // 上班打卡照片 >>>
        if (clockMap.containsKey(4)) {
            valuesImageToSheet(String.format(rangFormat, "X", "X"), clockMap.get(4));
        }

        // 午休下班打卡照片 >>>
        if (clockMap.containsKey(5)) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "Y", "Y"), clockMap.get(5));
        }

        // 午休上班打卡照片 >>>
        if (clockMap.containsKey(6)) {
            valuesImageToSheet(String.format(rangFormat, "Z", "Z"), clockMap.get(6));
        }

        // 下班打卡照片 >>>
        if (clockMap.containsKey(7)) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AA", "AA"), clockMap.get(7));
        }
        log.info("上传打卡照片完成，行号：{}", rowNumStr);

        /**
         * 推广试吃照片
         */
        List<TemporaryActivityPhotoDO> photosType1 = Objects.isNull(photoTypeMap.get(1)) ? new ArrayList<>() : photoTypeMap.get(1);
        int photosType1Size = photosType1.size();
        // 推广试吃照片1 >>>
        if (!(photosType1Size < 1 || Objects.isNull(photosType1.get(0)))) {
            valuesImageToSheet(String.format(rangFormat, "AB", "AB"), photosType1.get(0).getPhotoUrl());
        }

        // 推广试吃照片2 >>>
        if (!(photosType1Size < 2 || Objects.isNull(photosType1.get(1)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AC", "AC"), photosType1.get(1).getPhotoUrl());
        }

        // 推广试吃照片3 >>>
        if (!(photosType1Size < 3 || Objects.isNull(photosType1.get(2)))) {
            valuesImageToSheet(String.format(rangFormat, "AD", "AD"), photosType1.get(2).getPhotoUrl());
        }

        // 推广试吃照片4 >>>
        if (!(photosType1Size < 4 || Objects.isNull(photosType1.get(3)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AE", "AE"), photosType1.get(3).getPhotoUrl());
        }
        log.info("上传推广-试吃照片完成，行号：{}", rowNumStr);

        /**
         * 推广互动照片
         */
        List<TemporaryActivityPhotoDO> photosType2 = Objects.isNull(photoTypeMap.get(2)) ? new ArrayList<>() : photoTypeMap.get(2);
        int photosType2Size = photosType2.size();
        // 推广互动照片1 >>>
        if (!(photosType2Size < 1 || Objects.isNull(photosType2.get(0)))) {
            valuesImageToSheet(String.format(rangFormat, "AF", "AF"), photosType2.get(0).getPhotoUrl());
        }

        // 推广互动照片2 >>>
        if (!(photosType2Size < 2 || Objects.isNull(photosType2.get(1)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AG", "AG"), photosType2.get(1).getPhotoUrl());
        }

        // 推广互动照片3 >>>
        if (!(photosType2Size < 3 || Objects.isNull(photosType2.get(2)))) {
            valuesImageToSheet(String.format(rangFormat, "AH", "AH"), photosType2.get(2).getPhotoUrl());
        }

        // 推广互动照片4 >>>
        if (!(photosType2Size < 4 || Objects.isNull(photosType2.get(3)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AI", "AI"), photosType2.get(3).getPhotoUrl());
        }
        log.info("上传推广-互动照片完成，行号：{}", rowNumStr);

        /**
         * 推广成交照片
         */
        List<TemporaryActivityPhotoDO> photosType3 = Objects.isNull(photoTypeMap.get(3)) ? new ArrayList<>() : photoTypeMap.get(3);
        int photosType3Size = photosType3.size();
        // 推广成交照片1 >>>
        if (!(photosType3Size < 1 || Objects.isNull(photosType3.get(0)))) {
            valuesImageToSheet(String.format(rangFormat, "AJ", "AJ"), photosType3.get(0).getPhotoUrl());
        }

        // 推广成交照片2 >>>
        if (!(photosType3Size < 2 || Objects.isNull(photosType3.get(1)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AK", "AK"), photosType3.get(1).getPhotoUrl());
        }

        // 推广成交照片3 >>>
        if (!(photosType3Size < 3 || Objects.isNull(photosType3.get(2)))) {
            valuesImageToSheet(String.format(rangFormat, "AL", "AL"), photosType3.get(2).getPhotoUrl());
        }

        // 推广成交照片4 >>>
        if (!(photosType3Size < 4 || Objects.isNull(photosType3.get(3)))) {
            valuesImageToSheetByRobot2(String.format(rangFormat, "AM", "AM"), photosType3.get(3).getPhotoUrl());
        }
        log.info("上传推广-成交照片完成，行号：{}", rowNumStr);
    }

    private void addactivityDataRow(List<Object> rowData) {
        String redisKey = getRowNumRedisKey();
        String rowNumStr = redisCache.get(redisKey);
        Integer rowNumNext = StringUtils.isBlank(rowNumStr) ? 2 : Integer.parseInt(rowNumStr) + 1;
        String range = sheetInfoD.getSheetId() + "!A" + rowNumNext + ":N" + rowNumNext;

        Integer rowNumEnd = feishuSheetUtils.appendValueToSheet(range, rowData, sheetInfoD.getSheetToken(), getFeishuTenantToken1());
        if (rowNumEnd < 1) {
            redisCache.removeKey(redisKey);
        } else {
            redisCache.addToMinute(redisKey, rowNumEnd + "", 60);
        }
        log.info("修改文档行号：{}", rowNumEnd);
    }

    private void updateClockData(String range, List<Object> rowData) {
        feishuSheetUtils.updateData(range, rowData, sheetInfoD.getSheetToken(), getFeishuTenantToken2());
    }

    private void valuesImageToSheet(String range, String photoUrl) {
        feishuSheetUtils.valuesImageToSheet(range, photoUrl, sheetInfoD.getSheetToken(), getFeishuTenantToken1());
    }

//    @Async
    private void valuesImageToSheetByRobot2(String range, String photoUrl) {
        feishuSheetUtils.valuesImageToSheet(range, photoUrl, sheetInfoD.getSheetToken(), getFeishuTenantToken2());
    }

    private String getFeishuTenantToken1() {
        String tokenKey = RedisKeys.ExportKeys.FEISHU_TENANT_TOKEN_ACTIVITY_ROBOT_1.getKey();
        String token = redisCache.get(tokenKey);
        if (StringUtils.isNotBlank(token)) {
            return token;
        }

        JSONObject tokenObj = feishuSheetUtils.getFeishuTenantToken(appId1, appSecret1);
        token = tokenObj.getString("tenant_access_token");
        Integer expire = tokenObj.getInteger("expire");
        if (expire <= 110) {
            throw new RuntimeException("获取1号机器人tenant_access_token错误，有效期过短" + tokenObj);
        }
        redisCache.addToSeconds(tokenKey, token, expire - 100);
        return token;
    }

    private String getFeishuTenantToken2() {
        String tokenKey = RedisKeys.ExportKeys.FEISHU_TENANT_TOKEN_ACTIVITY_ROBOT_2.getKey();
        String token = redisCache.get(tokenKey);
        if (StringUtils.isNotBlank(token)) {
            return token;
        }

        JSONObject tokenObj = feishuSheetUtils.getFeishuTenantToken(appId2, appSecret2);
        token = tokenObj.getString("tenant_access_token");
        Integer expire = tokenObj.getInteger("expire");
        if (expire <= 110) {
            throw new RuntimeException("获取2号机器人tenant_access_token错误，有效期过短" + tokenObj);
        }
        redisCache.addToSeconds(tokenKey, token, expire - 100);
        return token;
    }

    private String getRowNumRedisKey() {
        String key = RedisKeys.ExportKeys.ACTIVITY_REPORTED_PUSH_FEISHU_SHEET.getKey();
        return key + DateUtil.today();
    }

}
