package com.wangxiaolu.promotion.domain.activityplanv2.dao.impl;

import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.wangxiaolu.promotion.common.enums.StatusType;
import com.wangxiaolu.promotion.common.util.BeanUtils;
import com.wangxiaolu.promotion.domain.activityplanv2.dao.ActivityPlanInfoDao;
import com.wangxiaolu.promotion.domain.activityplanv2.mapper.ActivityPlanInfoMapper;
import com.wangxiaolu.promotion.domain.activityplanv2.mapper.entity.ActivityPlanInfoDo;
import com.wangxiaolu.promotion.domain.manage.wrapperQo.ActivityPlanInfoWrapper;
import com.wangxiaolu.promotion.enums.plan.PlanStatus;
import com.wangxiaolu.promotion.exception.DataException;
import com.wangxiaolu.promotion.exception.FlowException;
import com.wangxiaolu.promotion.pojo.PageInfo;
import com.wangxiaolu.promotion.pojo.activity.planv2.dto.ActivityPlanInfoDto;
import com.wangxiaolu.promotion.result.basedata.RCode;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.exceptions.PersistenceException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.*;

/**
 * @author : liqiulin
 * @date : 2025-02-06 11
 * @describe :
 */
@Service
public class ActivityPlanInfoDaoImpl implements ActivityPlanInfoDao {

    @Autowired
    private ActivityPlanInfoMapper activityPlanInfoMapper;

    @Override
    public void saveList(JSONArray table, Long recordId) {
        try {
            activityPlanInfoMapper.saveList(table, recordId);
        } catch (PersistenceException e) {
            throw new DataException(RCode.ACTIVITY_PLAN_REPETITION_ERROR);
        }
    }



    @Override
    public void updateList(JSONArray table, Long recordId) {
        try {
            // 将table中的数据转换为ActivityPlanInfoDo
            for (int i = 0; i < table.size(); i++) {
                ActivityPlanInfoDo pdo = table.getObject(i, ActivityPlanInfoDo.class);
                activityPlanInfoMapper.updateOne(pdo, recordId);
            }
        } catch (PersistenceException e) {
            throw new DataException(RCode.ACTIVITY_PLAN_REPETITION_ERROR);
        }
    }

    @Override
    public ActivityPlanInfoDto selectById(Long id) {
        ActivityPlanInfoDo activityPlanInfoDo = activityPlanInfoMapper.selectById(id);
        return BeanUtils.transitionDto(activityPlanInfoDo, ActivityPlanInfoDto.class);
    }

    @Override
    public void updateById(ActivityPlanInfoDto planDto) {
        ActivityPlanInfoDo planDo = BeanUtils.transitionDto(planDto, ActivityPlanInfoDo.class);
        activityPlanInfoMapper.updateById(planDo);
    }

    @Override
    public void save(ActivityPlanInfoDto planDto) {
        try {
            ActivityPlanInfoDo planDo = new ActivityPlanInfoDo();
            BeanUtils.copyProperties(planDto, planDo);
            activityPlanInfoMapper.insert(planDo);
        } catch (PersistenceException e) {
            throw new DataException(RCode.ACTIVITY_PLAN_REPETITION_ERROR);
        }
    }

    @Override
    public List<ActivityPlanInfoDto> findClockList(ActivityPlanInfoWrapper wrapper) {
        List<ActivityPlanInfoDo> activityPlanInfoDos = activityPlanInfoMapper.findClockList(wrapper);
        return BeanUtils.transitionDtos(activityPlanInfoDos, ActivityPlanInfoDto.class);
    }

    @Override
    public void updatePlanStatus(Long planId, PlanStatus planStatus) {
        activityPlanInfoMapper.updatePlanStatus(planId,planStatus.getCode());
    }

    @Override
    public void updateByIds(List<Long> planIds, ActivityPlanInfoDto planDto) {
        errorMoreDate(planIds);
        activityPlanInfoMapper.updateByIds(planIds, planDto);
    }

    @Override
    public void page(PageInfo pageInfo, ActivityPlanInfoWrapper wrapper) {
        LambdaQueryWrapper<ActivityPlanInfoDo> qw = buildWrapper(wrapper);
        Page<ActivityPlanInfoDo> page = new Page<>(pageInfo.getPageNum(), pageInfo.getPageSize());
        Page<ActivityPlanInfoDo> doPage =  activityPlanInfoMapper.selectPage(page, qw);
        pageInfo.pageCovert(doPage);
        pageInfo.setRecords(doPage.getRecords());
    }

    @Override
    public ActivityPlanInfoDto selectPlan(String storeCode, Date date) {
        ActivityPlanInfoDo activityPlanInfoDo = activityPlanInfoMapper.selectPlan(storeCode, date);
        return BeanUtils.transitionDto(activityPlanInfoDo, ActivityPlanInfoDto.class);
    }

    @Override
    public void deleteByPlanIds(List<Long> planIds, String employeeNo,String operNo) {
        // 判断ID中是否包含之前数据、非个人数据
        Integer count = activityPlanInfoMapper.selectNotDelCount(planIds,employeeNo);
        if (count > 0 ) {
            throw new FlowException(RCode.ACTIVITY_PLAN_NOT_SELF_DELETE);
        }
        errorMoreDate(planIds);
        activityPlanInfoMapper.updateIsDelete(planIds,operNo);
    }

    /**
     * 根据地理位置信息查询本店本月门店列表
     * @param planDay 查询条件参数
     * @return 门店列表
     */
    @Override
    public List<Map<String, String>> findThisMonthStoreListByLocation(String planDay,String storeNameKeyword) {
        List<Map<String, String>> activityPlanInfoDoList = activityPlanInfoMapper.selectStoreInfoByCondition(planDay,storeNameKeyword);

        return activityPlanInfoDoList;
    }

    @Override
    public Integer selectCount(ActivityPlanInfoWrapper wrapper) {
        LambdaQueryWrapper<ActivityPlanInfoDo> wq = new LambdaQueryWrapper<>();
        wq.eq(ActivityPlanInfoDo::getStoreCode,wrapper.getStoreCode());
        wq.eq(ActivityPlanInfoDo::getEmployeeId,wrapper.getEmployeeId());
        wq.ne(ActivityPlanInfoDo::getId,wrapper.getId());
        wq.eq(ActivityPlanInfoDo::getDate,wrapper.getActivityDate());
        return activityPlanInfoMapper.selectCount(wq);
    }

    private LambdaQueryWrapper<ActivityPlanInfoDo> buildWrapper(ActivityPlanInfoWrapper wrapper) {
        LambdaQueryWrapper<ActivityPlanInfoDo> qw = new LambdaQueryWrapper<>();
        if (wrapper.rangeDate()){
            qw.between(ActivityPlanInfoDo::getDate, wrapper.getActivityStartDate(), wrapper.getActivityEndDate());
        }
        if (Objects.nonNull(wrapper.getActivityDate())){
            qw.eq(ActivityPlanInfoDo::getDate, wrapper.getActivityDate());
        }
        if (Objects.nonNull(wrapper.getEmployeeId())) {
            qw.eq(ActivityPlanInfoDo::getEmployeeId, wrapper.getEmployeeId());
        }
        if (StringUtils.isNotBlank(wrapper.getProvince())) {
            qw.eq(ActivityPlanInfoDo::getProvince, wrapper.getProvince());
        }
        if (StringUtils.isNotBlank(wrapper.getCity())) {
            qw.eq(ActivityPlanInfoDo::getCity, wrapper.getCity());
        }
        if (StringUtils.isNotBlank(wrapper.getArea())) {
            qw.eq(ActivityPlanInfoDo::getArea, wrapper.getArea());
        }
        if (StringUtils.isNotBlank(wrapper.getDealerId())) {
            qw.eq(ActivityPlanInfoDo::getDealerId, wrapper.getDealerId());
        }
        if (Objects.nonNull(wrapper.getPlanStatus())) {
            qw.eq(ActivityPlanInfoDo::getPlanStatus, wrapper.getPlanStatus().getCode());
        }
        if (StringUtils.isNotBlank(wrapper.getOrgQcId())){
            qw.eq(ActivityPlanInfoDo::getOrgQcId, wrapper.getOrgQcId());
        }
        if (StringUtils.isNotBlank(wrapper.getStoreNameLike())) {
            qw.like(ActivityPlanInfoDo::getStoreName, wrapper.getStoreNameLike());
        }
        qw.eq(ActivityPlanInfoDo::getIsDelete,Objects.nonNull(wrapper.getStatusType()) && wrapper.getStatusType().equals(StatusType.INVALID) ? StatusType.INVALID.getType() : StatusType.VALID.getType());
        qw.orderByDesc(ActivityPlanInfoDo::getDate)
                .orderByAsc(ActivityPlanInfoDo::getPlanStatus)
                .orderByDesc(ActivityPlanInfoDo::getClockInTime);
        if (Objects.nonNull(wrapper.getLimitNum())){
            qw.last(" limit "+wrapper.getLimitNum());
        }
        return qw;
    }

    private void errorMoreDate(List<Long> planIds){
        List<ActivityPlanInfoDo> todayPlans = activityPlanInfoMapper.selectTodayCount(planIds);
        // 今日数据存在时，判断是否执行、上班时间顺延1小时
        for (ActivityPlanInfoDo todayPlan : todayPlans) {
            if (PlanStatus.EXECUTION.getCode().equals(todayPlan.getPlanStatus())) {
                throw new FlowException(RCode.ACTIVITY_ID_IS_START.getCode(),String.format(RCode.ACTIVITY_ID_IS_START.getMsg(),todayPlan.getId()));
            }
            // 判断修改时间是否在促销员上班1小时内
            if (LocalDateTime.now().isAfter(todayPlan.getClockInTime().plusHours(1))){
                throw new FlowException(RCode.ACTIVITY_PLAN_ID_NOT_DELETE.getCode(),String.format(RCode.ACTIVITY_PLAN_ID_NOT_DELETE.getMsg(),todayPlan.getId()));
            }
        }
    }

}
