package com.wangxiaolu.promotion.domain.inspection.dao.impl;


import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.wangxiaolu.promotion.common.util.BeanUtils;
import com.wangxiaolu.promotion.domain.inspection.dao.InspectionInfoDao;
import com.wangxiaolu.promotion.domain.inspection.mapper.InspectionInfoMapper;
import com.wangxiaolu.promotion.domain.inspection.mapper.entity.InspectionInfoDO;
import com.wangxiaolu.promotion.domain.inspection.wrapper.InspectionInfoWrapper;
import com.wangxiaolu.promotion.exception.ParamException;
import com.wangxiaolu.promotion.pojo.PageInfo;
import com.wangxiaolu.promotion.pojo.activity.inspectionInfo.dto.InspectionInfoDto;
import com.wangxiaolu.promotion.result.basedata.RCode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.Objects;

/**
* @author DouXinYu
* @description 针对表【inspection_info(稽查信息表)】的数据库操作Service实现
* @createDate 2026-01-08 17:03:39
*/
@Service
public class InspectionInfoDaoImpl implements InspectionInfoDao {
    @Autowired
    private InspectionInfoMapper inspectionInfoMapper;

    /**
     * 根据ID查询稽查信息
     * @param id 稽核ID
     * @return 稽查信息DTO
     */
    @Override
    public InspectionInfoDto selectById(Long id) {
        InspectionInfoDO inspectionInfoDO = inspectionInfoMapper.selectById(id);
        return convertDoToDto(inspectionInfoDO);
    }

    /**
     * 根据计划ID查询稽查信息
     * @param planId 门店对应的计划ID
     * @return 稽查信息DTO
     */
    @Override
    public InspectionInfoDto selectByPlanId(Long planId) {
        InspectionInfoDO inspectionInfoDO = inspectionInfoMapper.selectByPlanId(planId);
        return convertDoToDto(inspectionInfoDO);
    }

    /**
     * 保存稽查信息
     * @param inspectionDto 稽查信息DTO
     * @return 保存后的稽查信息DTO
     */
    @Override
    public InspectionInfoDto save(InspectionInfoDto inspectionDto) {
        try {
            InspectionInfoDO inspectionDO = new InspectionInfoDO();
            // 拷贝基础属性
            BeanUtils.copyProperties(inspectionDto, inspectionDO);
            // 处理JSON数组类型字段（照片链接）
            convertArrayToString(inspectionDO, inspectionDto);
            // 插入数据库
            inspectionInfoMapper.insert(inspectionDO);
            // 查询插入后的完整数据并返回
            InspectionInfoDO savedDO = inspectionInfoMapper.selectById(inspectionDO.getId());
            return convertDoToDto(savedDO);
        } catch (Exception e) {
            String eMsg = e.getCause() != null ? e.getCause().getMessage() : "";
            // 处理plan_id唯一键冲突（对应数据库plan_id_unique索引）
            if (eMsg.contains("for key 'plan_id_unique'")) {
                throw new ParamException(RCode.DATA_HAVE_ERROR);
            }
            return null;
        }
    }

    /**
     * 根据ID更新稽查信息
     * @param inspectionDto 稽查信息DTO
     * @return 更新后的稽查信息DTO
     */
    @Override
    public InspectionInfoDto updateById(InspectionInfoDto inspectionDto) {
        InspectionInfoDO inspectionDO = new InspectionInfoDO();
        // 拷贝基础属性
        BeanUtils.copyProperties(inspectionDto, inspectionDO);
        // 处理JSON数组类型字段（照片链接）
        convertArrayToString(inspectionDO, inspectionDto);
        // 更新数据库
        inspectionInfoMapper.updateById(inspectionDO);
        // 查询更新后的完整数据并返回
        InspectionInfoDO updatedDO = inspectionInfoMapper.selectById(inspectionDO.getId());
        return convertDoToDto(updatedDO);
    }

    /**
     * 分页查询稽查信息
     * @param pageInfo 分页参数
     * @param wrapper  查询条件封装
     */
    @Override
    public void page(PageInfo pageInfo, InspectionInfoWrapper wrapper) {
        LambdaQueryWrapper<InspectionInfoDO> queryWrapper = buildQueryWrapper(wrapper);
        // 构建MyBatis-Plus分页对象
        Page<InspectionInfoDO> page = new Page<>(pageInfo.getPageNum(), pageInfo.getPageSize());
        // 执行分页查询
        Page<InspectionInfoDO> doPage = inspectionInfoMapper.selectPage(page, queryWrapper);
        // 转换分页结果
        pageInfo.pageCovert(doPage);
        pageInfo.setRecords(doPage.getRecords());
    }

    /**
     * 构建查询条件
     * @param wrapper 查询条件封装类
     * @return LambdaQueryWrapper
     */
    private LambdaQueryWrapper<InspectionInfoDO> buildQueryWrapper(InspectionInfoWrapper wrapper) {
        LambdaQueryWrapper<InspectionInfoDO> queryWrapper = new LambdaQueryWrapper<>();
        // 根据经销商ID查询
        if (Objects.nonNull(wrapper.getDealerId())) {
            queryWrapper.eq(InspectionInfoDO::getDealerId, wrapper.getDealerId());
        }
        // 根据门店编码查询
        if (Objects.nonNull(wrapper.getStoreCode())) {
            queryWrapper.eq(InspectionInfoDO::getStoreCode, wrapper.getStoreCode());
        }
        // 根据计划ID查询
        if (Objects.nonNull(wrapper.getPlanId())) {
            queryWrapper.eq(InspectionInfoDO::getPlanId, wrapper.getPlanId());
        }
        // 根据ID查询
        if (Objects.nonNull(wrapper.getId())){
            queryWrapper.eq(InspectionInfoDO::getId, wrapper.getId());
        }
        // 根据门店名称模糊查询
        if (!StringUtils.isEmpty(wrapper.getStoreName())){
            queryWrapper.like(InspectionInfoDO::getStoreName,wrapper.getStoreName());
        }

        return queryWrapper;
    }

    /**
     * 将DO转换为DTO（处理JSON字段）
     * @param inspectionInfoDO 数据库实体
     * @return 前端展示DTO
     */
    private InspectionInfoDto convertDoToDto(InspectionInfoDO inspectionInfoDO) {
        if (Objects.isNull(inspectionInfoDO)) {
            return null;
        }
        // 基础属性转换
        InspectionInfoDto dto = BeanUtils.transitionDto(inspectionInfoDO, InspectionInfoDto.class);
        // 处理JSON数组类型的照片字段（促销员在岗/离岗佐证照）
        dto.setTemWorkPhotos(Objects.isNull(inspectionInfoDO.getTemWorkPhotos())
                ? null : JSONArray.parseArray(inspectionInfoDO.getTemWorkPhotos()));
        return dto;
    }

    /**
     * 将DTO中的JSON数组转换为字符串（存入数据库）
     * @param inspectionDO 数据库实体
     * @param inspectionDto 前端传输DTO
     */
    private void convertArrayToString(InspectionInfoDO inspectionDO, InspectionInfoDto inspectionDto) {
        // 处理促销员在岗/离岗佐证照（JSONArray -> String）
        inspectionDO.setTemWorkPhotos(Objects.isNull(inspectionDto.getTemWorkPhotos())
                ? null : JSONObject.toJSONString(inspectionDto.getTemWorkPhotos()));
    }
}
