package com.wangxiaolu.promotion.utils;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.wangxiaolu.promotion.common.constant.Constants;
import com.wangxiaolu.promotion.common.redis.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

@Component
@Slf4j
public class WechatAccessTokenUtil {
    @Value("${wx.cx_miniapp.app_id}")
    private String appid;
    @Value("${wx.cx_miniapp.app_secret}")
    private String secret;
    private static final String TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=%s&secret=%s";
    private static final String STABLE_TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/stable_token";

    // 缓存access_token，避免频繁调用
    private String accessToken;
    private long expireTime;
    @Resource
    private WxMaService wxMaService ;


    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private RedisCache redisCache;



    // 获取有效的access_token
    public String getAccessToken() {
        String accessTokenRedisCache = redisCache.get(Constants.REDIS_ACCESS_TOKEN);
        // 检查是否过期，未过期直接返回
        if (ObjectUtil.isNotEmpty(accessTokenRedisCache)) {
            return accessTokenRedisCache;
        }
        // 过期则重新获取
        String url = String.format(TOKEN_URL, appid, secret);
        String response = restTemplate.getForObject(url, String.class);

        // string 转json
        JSONObject json = JSONObject.parseObject( response) ;
        accessToken = json.getString("access_token");
        // 有效期（秒）
        int expiresIn = json.getIntValue("expires_in");
        // 提前5分钟过期
        expireTime = System.currentTimeMillis() + (expiresIn - 300) * 1000;
        log.info("获取微信access_token,url:{},response:{}",url,response);

        redisCache.addToSeconds(Constants.REDIS_ACCESS_TOKEN,accessToken,expiresIn - 300);
        return accessToken;

    }

    public String getStableAccessToken() throws IOException {
        String accessTokenRedisCache = redisCache.get(Constants.REDIS_ACCESS_TOKEN);
        // 检查是否过期，未过期直接返回
        if (ObjectUtil.isNotEmpty(accessTokenRedisCache)) {
            return accessTokenRedisCache;
        }
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpPost httpPost = new HttpPost(STABLE_TOKEN_URL);

        // 2. 构建请求参数（普通模式，force_refresh=false）
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("grant_type", "client_credential");
        paramMap.put("appid", appid);
        paramMap.put("secret", secret);
        // 普通模式，避免频繁刷新
        paramMap.put("force_refresh", false);

        // 3. 设置请求头（JSON格式）
        httpPost.setHeader("Content-Type", "application/json;charset=UTF-8");
        httpPost.setEntity(new StringEntity(JSON.toJSONString(paramMap), "UTF-8"));

        // 4. 发送请求并解析响应
        String response = EntityUtils.toString(httpClient.execute(httpPost).getEntity());
        Map<String, Object> resultMap = JSON.parseObject(response, Map.class);

        // 5. 处理响应结果（成功则更新缓存，失败则抛异常）
        if (resultMap.get("errcode") != null && !"0".equals(resultMap.get("errcode").toString())) {
            throw new RuntimeException("获取 access_token 失败：" + resultMap.get("errmsg"));
        }
        accessToken = resultMap.get("access_token").toString();
        int expiresIn = Integer.parseInt(resultMap.get("expires_in").toString());
        log.info("获取微信access_token,url:{},response:{}",STABLE_TOKEN_URL,response);

        redisCache.addToSeconds(Constants.REDIS_ACCESS_TOKEN,accessToken,expiresIn - 300);
        return accessToken;

    }
}
