package com.wangxiaolu.promotion.service.activity.temporary.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.uuid.Generators;
import com.wangxiaolu.promotion.common.redis.RedisKeys;
import com.wangxiaolu.promotion.common.redis.service.RedisCache;
import com.wangxiaolu.promotion.domain.activity.dao.*;
import com.wangxiaolu.promotion.domain.user.dao.QinCeClienteleStoreDao;
import com.wangxiaolu.promotion.domain.user.dao.TemporaryInfoDao;
import com.wangxiaolu.promotion.domain.user.wrapperQo.StoreWrapper;
import com.wangxiaolu.promotion.enums.activity.ActivityPhotoType;
import com.wangxiaolu.promotion.enums.activity.ClockType;
import com.wangxiaolu.promotion.enums.activity.LogType;
import com.wangxiaolu.promotion.enums.activity.TemActApproveStatus;
import com.wangxiaolu.promotion.exception.DataException;
import com.wangxiaolu.promotion.exception.FlowException;
import com.wangxiaolu.promotion.exception.ParamException;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.PromotionStoreDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.TemporaryActivityReportedDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.TemporaryClockDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.vo.TemporaryActivityMarketCellVo;
import com.wangxiaolu.promotion.pojo.user.dto.QinCeClienteleStoreDto;
import com.wangxiaolu.promotion.pojo.user.dto.WxTemporaryInfoDto;
import com.wangxiaolu.promotion.result.basedata.RCode;
import com.wangxiaolu.promotion.service.activity.temporary.TemporaryActivityCoreService;
import com.wangxiaolu.promotion.utils.OkHttp;
import com.wangxiaolu.promotion.utils.QinCeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : liqiulin
 * @date : 2024-04-18 11
 * @describe :
 */
@Service
@Slf4j
public class TemporaryActivityCoreServiceImpl implements TemporaryActivityCoreService {
    @Autowired
    QinCeUtils qinCeUtils;
    @Autowired
    RedisCache redisCache;
    @Autowired
    private TemporaryActivityReportedDao temporaryActivityReportedDao;

    @Autowired
    private TemporaryActivityClockDao temporaryActivityClockDao;

    @Autowired
    private PromotionStoreDao promotionStoreDao;

    @Autowired
    private TemporaryActivityPhotoDao tempActivityPhotoDao;

    @Autowired
    private TemporaryActivityLogDao tempActivityLogDao;

    @Autowired
    private TemporaryInfoDao temporaryInfoDao;


    /**
     * 活动上报保存
     *
     * @return 生成id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public long activityDataReportedSave(TemporaryActivityReportedDto temActDto) {
        // 判断是否已存在保存数据
        TemporaryActivityReportedDto haveDate = temporaryActivityReportedDao.findOneByCurrentDate(temActDto.getTemporaryId());
        if (Objects.nonNull(haveDate)) {
            throw new FlowException(RCode.DATA_HAVE_ERROR, null);
        }

        // 查询上报人负责人
        WxTemporaryInfoDto wxTemporaryInfoDto = temporaryInfoDao.selectOneById(temActDto.getTemporaryId());

        // 初始化任务状态：已保存
        temActDto.setApproveStatus(TemActApproveStatus.APPROVED);
        temActDto.setApproverId(wxTemporaryInfoDto.getChargerQcId());
        temActDto.setApproveName(wxTemporaryInfoDto.getChargerName());
        temActDto.setApproveTime(new Date());

        // 根据storeQcId查询店铺信息，进行店铺信息补充
        PromotionStoreDto promotionStoreDto = promotionStoreDao.selectOneById(temActDto.getStoreId());
        temActDto.setStoreName(promotionStoreDto.getStoreName())
                .setStoreAddr(promotionStoreDto.getAddress())
                .setProvince(promotionStoreDto.getProvince())
                .setCity(promotionStoreDto.getCity());

        // 保存活动数据 temporary_activity_reported
        long reportedId = temporaryActivityReportedDao.activityDataSave(temActDto);
        temActDto.setId(reportedId);

        // 保存图片
        saveActivityPhoto(temActDto);

        // 日志保存
        tempActivityLogDao.save(temActDto.getTemporaryId(), temActDto.getTemporaryName(), LogType.t_2, temActDto.getId(), temActDto);
        return reportedId;
    }

    /**
     * 活动信息修改
     * 1、判断活动是否可以修改
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void activityDataReportedUpdate(TemporaryActivityReportedDto temActDto) {
        // 图片增量保存
        saveActivityPhoto(temActDto);
        temporaryActivityReportedDao.updateById(temActDto);
        // 日志保存
        tempActivityLogDao.save(temActDto.getTemporaryId(), temActDto.getTemporaryName(), LogType.t_2, temActDto.getId(), temActDto);
    }

    /**
     * 促销员当日打卡信息保存
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void clockInTodayActivity(TemporaryClockDto dto, Integer clockType) {
        log.info("促销员当日打卡[type-{}]信息保存：{}", clockType, dto.toString());
        if (Objects.isNull(dto.getId())) {
            temporaryActivityClockDao.save(dto);
        } else {
            temporaryActivityClockDao.updateById(dto);
        }
        // 根据ID保存图片
        saveClockPhoto(dto, clockType);
        // 日志保存
        tempActivityLogDao.save(dto.getTemporaryId(), dto.getTemporaryName(), LogType.t_1, dto.getId(), dto);
    }

    /**
     * 修改审批状态
     *
     * @param id 上报记录id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void activityReportedSubmit(Long id) {
        // 判断此id任务是否存在，并且是否满足提交审批条件
        TemporaryActivityReportedDto reportedDto = temporaryActivityReportedDao.findOneById(id);
        if (Objects.isNull(reportedDto)) {
            throw new ParamException(RCode.DATA_NOT_HAVE_ERROR, null);
        }
//        TemActApproveStatus approveStatus = reportedDto.getApproveStatus();
//        if (!TemActApproveStatus.SUBMITTED.name().equals(approveStatus.name()) && !TemActApproveStatus.SEND_BACK.name().equals(approveStatus.name())) {
//            throw new FlowException(RCode.STATUS_UPDATE_ERROR, null);
//        }

        // 查询促销员负责人，将人员补充到审批中
        WxTemporaryInfoDto wxTemporaryInfoDto = temporaryInfoDao.selectOneById(reportedDto.getTemporaryId());
        TemporaryActivityReportedDto updateDto = new TemporaryActivityReportedDto()
                .setId(id)
                .setApproveName(wxTemporaryInfoDto.getChargerName())
                .setApproveStatus(TemActApproveStatus.IN_APPROVAL);

        // 修改审批
        temporaryActivityReportedDao.updateById(updateDto);

        // 日志保存
        tempActivityLogDao.save(reportedDto.getTemporaryId(), reportedDto.getTemporaryName(), LogType.t_3, updateDto.getId(), updateDto);
    }

    @Override
    public void todayActivityMarketCellReported(TemporaryActivityMarketCellVo marketcellVo) {
        String key = temMarketCellRedisKey(marketcellVo.getTemporaryId());
        JSONArray marketCells = redisCache.getToJsonArray(key);
        if (Objects.isNull(marketCells)){
            marketCells = new JSONArray();
        }else {
            // 判断prdId是否已存在
            List<String> prdIdList = marketCells.stream().map(o -> ((JSONObject) o).getString("prdId")).collect(Collectors.toList());
            if (prdIdList.contains(marketcellVo.getPrdId())){
                throw new ParamException(RCode.PRODUCT_CLASS_HAS_ERROR,null);
            }
        }

        marketcellVo.setUuid(Generators.randomBasedGenerator().generate().toString());
        marketCells.add(marketcellVo);
        redisCache.addToJsonToMinute(key, marketCells, 30);
    }

    @Override
    public void todayActivityDeleteMarketCellReported(Integer temporaryId) {
        String key = temMarketCellRedisKey(temporaryId);
        redisCache.removeKey(key);
    }

    /**
     * 根据店铺或打卡记录中的店铺，判断打卡距离
     * 经度，范围为 -180~180，负数表示西经;纬度，范围为 -90~90，负数表示南纬
     *
     * @param storeQcId        店铺-勤策-id
     * @param id               打卡记录id
     * @param clockCoordinates 打卡经纬度
     */
    @Override
    public void clockStoreCalDistance(String storeQcId, Long id, String clockCoordinates) {
//        if (Objects.nonNull(id)) {
//            TemporaryClockDto clockDto = temporaryActivityClockDao.selectOne(new TemporaryClockWrapper().setId(id));
//            storeQcId = clockDto.getStoreQcId();
//        }
//        clockStoreCalDistanceByStoreQcId(storeQcId, clockCoordinates);
    }

    private void clockStoreCalDistanceByStoreQcId(String storeQcId, String clockCoordinates) {
        // 查询组织架构参数、创建url
        String[] clockCoordinateArr = clockCoordinates.split(",");
        List<Map> paramList = qinCeUtils.queryCalDistanceParam(storeQcId, clockCoordinateArr[0], clockCoordinateArr[1]);
        String url = qinCeUtils.builderUrlList(QinCeUtils.CAL_DISTANCE, paramList);

        // 发起请求、接收结果
        JSONObject resultJson = OkHttp.postQC(url, paramList);
        String rData = resultJson.getString("response_data");
        if (StringUtils.isBlank(rData) || !rData.contains(storeQcId)) {
            throw new ParamException(RCode.PRO_NOT_QC_STORE_CLOCK_DISTANCE, null);
        }
        Long distance = JSONObject.parseArray(rData).getJSONObject(0).getLong("distance");
        log.info("[打卡] 判断打卡距离，勤策-id:{}，打卡距离:{}，打卡经纬度:{}", storeQcId, distance, clockCoordinates);
//        todo 判断打卡距离
//        if (distance > 100L) {
//            throw new ParamException(RCode.PRO_FAR_QC_STORE_CLOCK_DISTANCE, null);
//        }
    }

    private String temMarketCellRedisKey(Integer temporaryId){
        return RedisKeys.TemporaryKeys.TEMPORARY_ACTIVITY_MARKET_CELL.getKey() + temporaryId + "_" + DateUtil.today();
    }

    /**
     * 推广活动照片保存
     *
     * @param temActDto 活动提交数据
     */
    private void saveActivityPhoto(TemporaryActivityReportedDto temActDto) {
        Integer temporaryId = temActDto.getTemporaryId();
        Long reportedId = temActDto.getId();
        // 推广试吃照片
        if (!CollectionUtils.isEmpty(temActDto.getTgscPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.TGSC.getType(), temActDto.getTgscPhotoUrls(), temActDto.getTgscChangePhotoUrls());
        }

        // 推广互动照片
        if (!CollectionUtils.isEmpty(temActDto.getTghdPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.TGHD.getType(), temActDto.getTghdPhotoUrls(), temActDto.getTghdChangePhotoUrls());
        }

        // 推广成交照片
        if (!CollectionUtils.isEmpty(temActDto.getTgcjPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.TGCJ.getType(), temActDto.getTgcjPhotoUrls(), temActDto.getTgcjChangePhotoUrls());
        }
        // POS机页面凭证
        if (!CollectionUtils.isEmpty(temActDto.getPsvPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.POS_SELL_VOUCHER.getType(), temActDto.getPsvPhotoUrls(), temActDto.getPsvChangePhotoUrls());
        }
        log.info("促销员今日活动上报-图片保存成功，关联活动上报数据id[{}]，数据信息：{}", temActDto.getId(), temActDto);
    }

    /**
     * 促销员上班打卡图片保存
     */
    private void saveClockPhoto(TemporaryClockDto dto, Integer clockType) {
        // 上班卡、午休下班卡、午休上班卡、下班卡
        if (ClockType.TEMPORARY_CLOCK_IN.equals(clockType)) {
            tempActivityPhotoDao.saveClockPhoto(dto.getTemporaryId(), dto.getId(), ActivityPhotoType.CLOCK_IN.getType(), dto.getClockInPhoto());
        } else if (ClockType.TEMPORARY_NOON_CLOCK_OUT.equals(clockType)) {
            tempActivityPhotoDao.saveClockPhoto(dto.getTemporaryId(), dto.getId(), ActivityPhotoType.NOON_CLOCK_OUT.getType(), dto.getNoonClockOutPhoto());
        } else if (ClockType.TEMPORARY_NOON_CLOCK_IN.equals(clockType)) {
            tempActivityPhotoDao.saveClockPhoto(dto.getTemporaryId(), dto.getId(), ActivityPhotoType.NOON_CLOCK_IN.getType(), dto.getNoonClockInPhoto());
        } else if (ClockType.TEMPORARY_CLOCK_OUT.equals(clockType)) {
            tempActivityPhotoDao.saveClockPhoto(dto.getTemporaryId(), dto.getId(), ActivityPhotoType.CLOCK_OUT.getType(), dto.getClockOutPhoto());
        }
    }
}
