package com.wangxiaolu.promotion.xxljobtask;

import com.wangxiaolu.promotion.pojo.activity.temporary.dto.WxTemporaryInfoDelayDtO;
import com.wangxiaolu.promotion.pojo.user.dto.WxTemporaryInfoDto;
import com.wangxiaolu.promotion.service.wechat.WeChatUserCoreService;
import com.wangxiaolu.promotion.service.user.WxTemporaryInfoDelayService;
import com.xxl.job.core.context.XxlJobHelper;
import com.xxl.job.core.handler.annotation.XxlJob;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * XXL-Job定时任务：每天00:01处理微信用户延迟更新的门店信息
 * @author: 自定义
 * @date: 2026-01-14
 */
@Slf4j
@Component
public class WxTemporaryInfoDelayHandler{

    @Resource
    private WxTemporaryInfoDelayService wxTemporaryInfoDelayService;

    @Resource
    private WeChatUserCoreService weChatUserCoreService;

    /**
     * 定时任务执行方法
     * 任务描述：每天00:01处理前一天未处理的用户门店信息更新
     * XXL-Job配置：cron表达式 0 1 0 * * ? （每天00:01执行）
     */
    @XxlJob("wxTemporaryInfoDelayHandler")
    public void execute() {
        // 1. 任务日志记录（XXL-Job控制台可见）
        XxlJobHelper.log("开始执行微信用户延迟信息更新任务，执行时间：{}", LocalDateTime.now());
        try {
            // 2. 构建时间范围：查询前一天00:00到当天00:00的未处理数据
            LocalDate yesterday = LocalDate.now().minusDays(1);
            LocalDateTime startTime = LocalDateTime.of(yesterday, LocalTime.MIN);
            LocalDateTime endTime = LocalDateTime.of(LocalDate.now(), LocalTime.MIN);

            // 3. 查询未处理的延迟数据
            List<WxTemporaryInfoDelayDtO> unprocessedList = wxTemporaryInfoDelayService.listUnprocessedData(startTime, endTime);
            if (unprocessedList == null || unprocessedList.isEmpty()) {
                XxlJobHelper.log("未查询到需要处理的延迟数据，任务结束");
                return;
            }
            XxlJobHelper.log("查询到待处理延迟数据共{}条", unprocessedList.size());

            // 4. 转换DTO并批量更新门店信息
            List<WxTemporaryInfoDto> updateList = new ArrayList<>();
            List<Long> idList = new ArrayList<>();
            for (WxTemporaryInfoDelayDtO delayDto : unprocessedList) {
                // 4.1 收集需要标记为已处理的ID
                idList.add(delayDto.getId());
                
                // 4.2 转换为业务层需要的DTO
                WxTemporaryInfoDto tempDto = new WxTemporaryInfoDto();
                BeanUtils.copyProperties(delayDto, tempDto);
                updateList.add(tempDto);
            }

            // 5. 批量更新用户门店信息
            boolean updateResult = weChatUserCoreService.batchSaveWxUserStoreInfo(updateList);
            if (!updateResult) {
                XxlJobHelper.log("批量更新用户门店信息失败，数据量：{}", updateList.size());
                XxlJobHelper.handleFail("批量更新用户门店信息失败");
                return;
            }
            XxlJobHelper.log("批量更新用户门店信息成功，数据量：{}", updateList.size());

            // 6. 标记延迟表数据为已处理（1=已处理）
            boolean markResult = wxTemporaryInfoDelayService.batchUpdateProcessedStatus(idList, 1);
            if (!markResult) {
                XxlJobHelper.log("标记延迟数据为已处理失败，ID列表：{}", idList);
                XxlJobHelper.handleFail("标记延迟数据为已处理失败");
                return;
            }
            XxlJobHelper.log("标记{}条延迟数据为已处理成功", idList.size());

            // 7. 任务执行成功
            XxlJobHelper.handleSuccess("微信用户延迟信息更新任务执行成功，共处理" + unprocessedList.size() + "条数据");
        } catch (Exception e) {
            // 8. 异常处理
            log.error("微信用户延迟信息更新任务执行异常", e);
            XxlJobHelper.log("任务执行异常：{}", e.getMessage());
            XxlJobHelper.handleFail("任务执行异常：" + e.getMessage());
        }
    }
}