package com.wangxiaolu.promotion.service.activity.temporary.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.uuid.Generators;
import com.wangxiaolu.promotion.common.redis.RedisKeys;
import com.wangxiaolu.promotion.common.redis.service.RedisCache;
import com.wangxiaolu.promotion.domain.activity.dao.*;
import com.wangxiaolu.promotion.domain.user.dao.TemporaryInfoDao;
import com.wangxiaolu.promotion.enums.activity.ActivityPhotoType;
import com.wangxiaolu.promotion.enums.activity.LogType;
import com.wangxiaolu.promotion.enums.activity.TemActApproveStatus;
import com.wangxiaolu.promotion.exception.FlowException;
import com.wangxiaolu.promotion.exception.ParamException;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.PromotionStoreDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.TemporaryActivityMarketCellDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.TemporaryActivityReportedDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.vo.TemporaryActivityMarketCellVo;
import com.wangxiaolu.promotion.pojo.user.dto.WxTemporaryInfoDto;
import com.wangxiaolu.promotion.result.basedata.RCode;
import com.wangxiaolu.promotion.service.activity.temporary.TemporaryActivityCoreService;
import com.wangxiaolu.promotion.utils.QinCeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author : liqiulin
 * @date : 2024-04-18 11
 * @describe :
 */
@Service
@Slf4j
public class TemporaryActivityCoreServiceImpl implements TemporaryActivityCoreService {
    @Autowired
    QinCeUtils qinCeUtils;
    @Autowired
    RedisCache redisCache;
    @Autowired
    private TemporaryActivityReportedDao temporaryActivityReportedDao;
    @Autowired
    private PromotionStoreDao promotionStoreDao;
    @Autowired
    private TemporaryActivityPhotoDao tempActivityPhotoDao;
    @Autowired
    private TemporaryActivityLogDao tempActivityLogDao;
    @Autowired
    private TemporaryInfoDao temporaryInfoDao;
    @Autowired
    private TemporaryActivityMarketCellDao temporaryActivityMarketCellDao;


    /**
     * 活动上报保存
     *
     * @return 生成id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public long activityDataReportedSave(TemporaryActivityReportedDto temActDto) {
        // 判断是否已存在保存数据
        TemporaryActivityReportedDto haveDate = temporaryActivityReportedDao.findOneByCurrentDate(temActDto.getTemporaryId());
        if (Objects.nonNull(haveDate)) {
            throw new FlowException(RCode.DATA_HAVE_ERROR);
        }

        // 返回生成的活动id
        long reportedId = temporaryActivityReportedDao.activityDataSave(temActDto);
        temActDto.setId(reportedId);

        // 保存图片
        saveActivityPhoto(temActDto);

        // 售卖单元新增保存
//        String key = temMarketCellRedisKey(temActDto.getTemporaryId());
//        temporaryActivityMarketCellDao.saveList(reportedId,redisCache.getToJsonArray(key));
//        redisCache.removeKey(key);

//        tempActivityLogDao.save(temActDto.getTemporaryId(), temActDto.getTemporaryName(), LogType.t_2, temActDto.getId(), temActDto);
        return reportedId;
    }

    /**
     * 活动信息修改
     * 1、判断活动是否可以修改
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void activityDataReportedUpdate(TemporaryActivityReportedDto temActDto) {
        // 图片增量保存
        saveActivityPhoto(temActDto);
        temporaryActivityReportedDao.updateById(temActDto);

        // 售卖单元新增保存
//        String key = temMarketCellRedisKey(temActDto.getTemporaryId());
//        temporaryActivityMarketCellDao.saveList(temActDto.getId(),redisCache.getToJsonArray(key));
//        redisCache.removeKey(key);
        // 日志保存
//        tempActivityLogDao.save(temActDto.getTemporaryId(), temActDto.getTemporaryName(), LogType.t_2, temActDto.getId(), temActDto);
    }

    /**
     * 修改审批状态
     *
     * @param id 上报记录id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void activityReportedSubmit(Long id) {
        // 判断此id任务是否存在，并且是否满足提交审批条件
        TemporaryActivityReportedDto reportedDto = temporaryActivityReportedDao.findOneById(id);
        if (Objects.isNull(reportedDto)) {
            throw new ParamException(RCode.DATA_NOT_HAVE_ERROR, null);
        }

        // 查询促销员负责人，将人员补充到审批中
        TemporaryActivityReportedDto updateDto = new TemporaryActivityReportedDto()
                .setId(id)
                .setApproveStatus(TemActApproveStatus.APPROVED)
                .setApproveTime(new Date());

        // 修改审批
        temporaryActivityReportedDao.updateById(updateDto);

        // 日志保存
        tempActivityLogDao.save(reportedDto.getTemporaryId(), reportedDto.getTemporaryName(), LogType.t_3, updateDto.getId(), updateDto);
    }

    @Override
    public void todayActivityMarketCellReported(TemporaryActivityMarketCellVo marketcellVo) {
        String key = temMarketCellRedisKey(marketcellVo.getTemporaryId());
        JSONArray marketCells = redisCache.getToJsonArray(key);
        if (Objects.isNull(marketCells)) {
            marketCells = new JSONArray();
        } else {
            // 判断prdId是否已存在
            List<String> prdIdList = marketCells.stream().map(o -> ((JSONObject) o).getString("prdId")).collect(Collectors.toList());
            if (prdIdList.contains(marketcellVo.getPrdId())) {
                throw new ParamException(RCode.PRODUCT_CLASS_HAS_ERROR, null);
            }
        }

        marketcellVo.setUuid(Generators.randomBasedGenerator().generate().toString());
        marketCells.add(marketcellVo);
        redisCache.addToJsonToMinute(key, marketCells, 30);
    }

    @Override
    public void todayActivityDeleteMarketCellReported(Integer temporaryId) {
        String key = temMarketCellRedisKey(temporaryId);
        redisCache.removeKey(key);
    }

    @Override
    public void todayUpdateMarketCellOne(TemporaryActivityMarketCellVo marketcellVo) {
        // 修改/删除 缓存中的数据
        if (StringUtils.isNotBlank(marketcellVo.getUuid())) {
            todayUpdateMarketCellOneToRedis(marketcellVo);
        }

        //  修改/删除 DB库中的数据
        if (Objects.nonNull(marketcellVo.getMarketCellId())){
            todayUpdateMarketCellOneToDb(marketcellVo);
        }
    }

    private void todayUpdateMarketCellOneToDb(TemporaryActivityMarketCellVo marketcellVo) {
        if (marketcellVo.operateIsDelete()){
            temporaryActivityMarketCellDao.deleteById(marketcellVo.getMarketCellId());
        }

        if (marketcellVo.operateIsUpdate()){
            TemporaryActivityMarketCellDto dto = new TemporaryActivityMarketCellDto()
                    .setId(marketcellVo.getMarketCellId())
                    .setBag(marketcellVo.getBag())
                    .setBox(marketcellVo.getBox());
            temporaryActivityMarketCellDao.updateById(dto);
        }

    }

    private void todayUpdateMarketCellOneToRedis(TemporaryActivityMarketCellVo marketcellVo) {
        String key = temMarketCellRedisKey(marketcellVo.getTemporaryId());
        JSONArray marketCells = redisCache.getToJsonArray(key);
        if (Objects.isNull(marketCells)) {
            throw new ParamException(RCode.DATA_NOT_HAVE_ERROR, null);
        }

        String uuid = marketcellVo.getUuid();

        if (marketcellVo.operateIsUpdate()) {
            marketCells.stream().forEach(jo -> {
                JSONObject mj = (JSONObject) jo;
                String uuidR = mj.getString("uuid");
                if (uuidR.equals(uuid)) {
                    mj.put("box", marketcellVo.getBox());
                    mj.put("bag", marketcellVo.getBag());
                }
            });
        }

        if (marketcellVo.operateIsDelete()) {
            for (int i = 0; i < marketCells.size(); i++) {
                JSONObject mj = (JSONObject) marketCells.get(i);
                String uuidR = mj.getString("uuid");
                if (uuidR.equals(uuid)) {
                    marketCells.remove(i);
                    break;
                }
            }
        }

        redisCache.removeKey(key);
        if (!CollectionUtils.isEmpty(marketCells)) {
            redisCache.addToJsonToMinute(key, marketCells, 30);
        }
    }

    private String temMarketCellRedisKey(Integer temporaryId) {
        return RedisKeys.TemporaryKeys.TEMPORARY_ACTIVITY_MARKET_CELL.getKey() + temporaryId + "_" + DateUtil.today();
    }

    /**
     * 推广活动照片保存
     *
     * @param temActDto 活动提交数据
     */
    private void saveActivityPhoto(TemporaryActivityReportedDto temActDto) {
        Integer temporaryId = temActDto.getTemporaryId();
        Long reportedId = temActDto.getId();
        // 推广试吃照片
        if (!CollectionUtils.isEmpty(temActDto.getTgscPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.TGSC.getType(), temActDto.getTgscPhotoUrls(), temActDto.getTgscChangePhotoUrls());
        }

        // 推广互动照片
        if (!CollectionUtils.isEmpty(temActDto.getTghdPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.TGHD.getType(), temActDto.getTghdPhotoUrls(), temActDto.getTghdChangePhotoUrls());
        }

        // 推广成交照片
        if (!CollectionUtils.isEmpty(temActDto.getTgcjPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.TGCJ.getType(), temActDto.getTgcjPhotoUrls(), temActDto.getTgcjChangePhotoUrls());
        }
        // POS机页面凭证
        if (!CollectionUtils.isEmpty(temActDto.getPsvPhotoUrls())) {
            tempActivityPhotoDao.saveReportedList(temporaryId, reportedId, ActivityPhotoType.POS_SELL_VOUCHER.getType(), temActDto.getPsvPhotoUrls(), temActDto.getPsvChangePhotoUrls());
        }
    }
}
