package com.wangxiaolu.promotion.domain.activity.dao.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.wangxiaolu.promotion.domain.activity.dao.PromotionStoreDao;
import com.wangxiaolu.promotion.domain.activity.mapper.PromotionStoreMapper;
import com.wangxiaolu.promotion.domain.activity.mapper.entity.PromotionStoreDO;
import com.wangxiaolu.promotion.domain.activity.wrapperQo.PromotionStoreWrapper;
import com.wangxiaolu.promotion.exception.ParamException;
import com.wangxiaolu.promotion.pojo.PageInfo;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.PromotionStoreDto;
import com.wangxiaolu.promotion.result.basedata.RCode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author : liqiulin
 * @date : 2024-06-18 13
 * @describe :
 */
@Service
@Slf4j
public class PromotionStoreDaoImpl implements PromotionStoreDao {

    @Autowired
    PromotionStoreMapper promotionStoreMapper;

    /**
     * 保存店铺
     */
    @Override
    public void save(PromotionStoreDto promotionStoreDto) {
        // 查询是否重名
        PromotionStoreWrapper psw = new PromotionStoreWrapper()
                .setStoreName(promotionStoreDto.getStoreName());
        int coud = findCoud(psw);
        if (coud > 0 ){
            throw new ParamException(RCode.PROMOTION_STORE_HAS_NAME,null);
        }

        PromotionStoreDO storeDo = new PromotionStoreDO();
        BeanUtils.copyProperties(promotionStoreDto, storeDo);
        promotionStoreMapper.insert(storeDo);
    }

    @Override
    public List<PromotionStoreDto> findList(PromotionStoreWrapper promotionStoreWrapper) {
        LambdaQueryWrapper<PromotionStoreDO> qw = buildWrapper(promotionStoreWrapper);
        qw.select(PromotionStoreDO::getStoreName,PromotionStoreDO::getId).last("limit 200");

        List<PromotionStoreDO> promotionStoreDOS = promotionStoreMapper.selectList(qw);
        return transitionDtos(promotionStoreDOS);
    }

    @Override
    public PromotionStoreDto selectOneById(Long id) {
        PromotionStoreDO promotionStoreDO = promotionStoreMapper.selectById(id);
        return transitionDto(promotionStoreDO);
    }

    @Override
    public void qinceShopDetailAllTask(JSONArray responseDatas) {
        LambdaQueryWrapper<PromotionStoreDO> qw = new LambdaQueryWrapper<>();

        for (Object responseData : responseDatas) {
            qw.clear();

            try {
                JSONObject jo = JSONObject.parseObject(JSONObject.toJSONString(responseData));
                PromotionStoreDO storeDo = new PromotionStoreDO();
                storeDo.setQcId(jo.getString("id"))
                        .setStoreName(jo.getString("store_name"))
                        .setProvince(jo.getString("store_mss_province"))
                        .setCity(jo.getString("store_mss_city"))
                        .setArea(jo.getString("store_mss_area"))
                        .setStreet(jo.getString("store_mss_street"))
                        .setAddress(jo.getString("store_addr"))
                        .setTemporaryId(0)
                        .setTemporaryName("系统同步")
                        .setStoreManager(jo.getString("store_manager"))
                        .setStoreManagerWaiqin365Id(jo.getString("store_manager_waiqin365_id"));

                if (StringUtils.isBlank(storeDo.getQcId())) {
                    continue;
                }

                JSONArray dealersJR = jo.getJSONArray("dealers");
                if (!CollectionUtils.isEmpty(dealersJR)) {
                    JSONObject dealersJ = dealersJR.getJSONObject(0);
                    storeDo.setDealersName(dealersJ.getString("dealer_name"))
                            .setWaiqin365DealersId(dealersJ.getString("waiqin365_dealer_id"));
                }

                // 判断qc_id是否存在；存在则修改，不存在则新增
                qw.eq(PromotionStoreDO::getQcId, storeDo.getQcId());
                PromotionStoreDO doExist = promotionStoreMapper.selectOne(qw);

                if (Objects.isNull(doExist)) {
                    promotionStoreMapper.insert(storeDo);
                } else {
                    storeDo.setId(doExist.getId());
                    promotionStoreMapper.updateById(storeDo);
                }


            } catch (Exception e) {
                log.error("勤策-同步终端到促销店铺数据异常，异常数据：{}", responseData);
                log.error("勤策-同步终端到促销店铺数据异常\n{}", e.getMessage());
            }
        }
    }

    @Override
    public void findPage(PromotionStoreWrapper wrap, PageInfo pageInfo) {
        LambdaQueryWrapper<PromotionStoreDO> qw = buildWrapper(wrap);
        Page<PromotionStoreDO> page = new Page<>(pageInfo.getPageNum(), pageInfo.getPageSize());
        Page<PromotionStoreDO> storePage = promotionStoreMapper.selectPage(page, qw);
        pageInfo.pageCovert(storePage);
        pageInfo.setRecords(transitionDtos(storePage.getRecords()));
    }

    private LambdaQueryWrapper<PromotionStoreDO> buildWrapper(PromotionStoreWrapper promotionStoreWrapper){
        LambdaQueryWrapper<PromotionStoreDO> lqw = new LambdaQueryWrapper<>();
        if (StringUtils.isNotBlank(promotionStoreWrapper.getProvince())){
            lqw.eq(PromotionStoreDO::getProvince,promotionStoreWrapper.getProvince());
        }
        if (StringUtils.isNotBlank(promotionStoreWrapper.getCity())){
            lqw.eq(PromotionStoreDO::getCity,promotionStoreWrapper.getCity());
        }
        if (StringUtils.isNotBlank(promotionStoreWrapper.getArea())){
            lqw.eq(PromotionStoreDO::getArea,promotionStoreWrapper.getArea());
        }
        if (StringUtils.isNotBlank(promotionStoreWrapper.getStoreNameLike())){
            lqw.like(PromotionStoreDO::getStoreName,promotionStoreWrapper.getStoreNameLike());
        }
        if (StringUtils.isNotBlank(promotionStoreWrapper.getStoreName())){
            lqw.eq(PromotionStoreDO::getStoreName,promotionStoreWrapper.getStoreName());
        }
        if (Objects.nonNull(promotionStoreWrapper.getQcIdIsBlank())){
            if (promotionStoreWrapper.getQcIdIsBlank()){
                lqw.eq(PromotionStoreDO::getQcId,"");
            }else {
                lqw.ne(PromotionStoreDO::getQcId,"");
            }
        }
        // 所属战区
        if (StringUtils.isNotBlank(promotionStoreWrapper.getDeptQcId())){
            lqw.eq(PromotionStoreDO::getDeptQcId,promotionStoreWrapper.getDeptQcId());
        }
        // 所属负责人
        if (StringUtils.isNotBlank(promotionStoreWrapper.getChargerQcId())){
            lqw.eq(PromotionStoreDO::getChargerQcId,promotionStoreWrapper.getChargerQcId());
        }
        return lqw;
    }

    private int findCoud(PromotionStoreWrapper promotionStoreWrapper){
        LambdaQueryWrapper<PromotionStoreDO> qw = buildWrapper(promotionStoreWrapper);
        Integer count = promotionStoreMapper.selectCount(qw);
        return count;
    }


    /**
     * DO to DTO (单个对象)
     *
     * @param storeDos DO对象List
     * @return DTO对象
     */
    private List<PromotionStoreDto> transitionDtos(List<PromotionStoreDO> storeDos) {
        if (CollectionUtils.isEmpty(storeDos)) {
            return new ArrayList<>();
        }

        List<PromotionStoreDto> dtos = new ArrayList<>(storeDos.size() * 2);
        for (PromotionStoreDO storeDo : storeDos) {
            dtos.add(transitionDto(storeDo));
        }
        return dtos;
    }

    private PromotionStoreDto transitionDto(PromotionStoreDO storeDo) {
        PromotionStoreDto dto = null;
        if (Objects.isNull(storeDo)) {
            return dto;
        }
        dto = new PromotionStoreDto();
        BeanUtils.copyProperties(storeDo, dto);
        return dto;
    }

}
