package com.wangxiaolu.promotion.service.activityplanv2.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.wangxiaolu.promotion.common.excel.ReadExcelUtils;
import com.wangxiaolu.promotion.common.redis.RedisKeys;
import com.wangxiaolu.promotion.common.redis.service.RedisCache;
import com.wangxiaolu.promotion.domain.activityplanv2.dao.ActivityPlanInfoDao;
import com.wangxiaolu.promotion.domain.activityplanv2.dao.ActivityPlanRecordDao;
import com.wangxiaolu.promotion.domain.user.dao.QinCeClienteleStoreDao;
import com.wangxiaolu.promotion.domain.user.wrapperQo.StoreWrapper;
import com.wangxiaolu.promotion.exception.DataException;
import com.wangxiaolu.promotion.pojo.activity.manage.dto.ActivityPlanInfoDto;
import com.wangxiaolu.promotion.pojo.activity.manage.dto.ActivityPlanRecordDto;
import com.wangxiaolu.promotion.pojo.activity.manage.vo.ActivityPlanVo;
import com.wangxiaolu.promotion.pojo.user.dto.QinCeClienteleStoreDto;
import com.wangxiaolu.promotion.result.basedata.RCode;
import com.wangxiaolu.promotion.service.activityplanv2.PromPlanCoreService;
import com.wangxiaolu.promotion.utils.DateUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.*;
import java.util.*;

/**
 * @author : liqiulin
 * @date : 2025-01-21 13
 * @describe :
 */
@Service
public class PromPlanCoreServiceImpl implements PromPlanCoreService {

    @Autowired
    private QinCeClienteleStoreDao qinCeClienteleStoreDao;
    @Autowired
    private ActivityPlanRecordDao activityPlanRecordDao;
    @Autowired
    private ActivityPlanInfoDao activityPlanInfoDao;
    @Autowired
    private RedisCache redisCache;

    @Override
    public Map<String, Object> selfPlanUp(ActivityPlanVo planVo, String filePath) throws DataException {
        // 1、解析活动计划文件是否符合规范
        Map<String, Object> map = readSheetByCSJLUp(planVo, filePath);
        if (!map.containsKey("uuid")){
            return map;
        }

        // 2、生成uuid并关联上传记录
        ActivityPlanRecordDto record = new ActivityPlanRecordDto()
                .setEmployeeId(planVo.getEmployeeId())
                .setEmployeeName(planVo.getEmployeeName())
                .setEmployeeNo(planVo.getEmployeeNo())
                .setExcelUrl(planVo.getExcelUrl())
                .setExcelFiledId(planVo.getExcelId());
        map.put("record", record);
        // 保存到缓存（uuid）
        redisCache.addToJsonToMinute(RedisKeys.Manage.ACTIVITY_PLAN_UP.getKey()+map.get("uuid").toString(),map,30);
        // 返回结果
        return map;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void selfPlanAf(String planUuid) {
        JSONObject mapJson = redisCache.getToJson(RedisKeys.Manage.ACTIVITY_PLAN_UP.getKey() + planUuid);
        if (Objects.isNull(mapJson)){
            throw new DataException(RCode.REDIS_PUSH_DATA_NOT_EXIT);
        }
        JSONObject record = mapJson.getJSONObject("record");
        JSONArray table = mapJson.getJSONArray("table");

        // 保存上传记录
        Long recordId = activityPlanRecordDao.save(record);
        activityPlanInfoDao.saveList(table,recordId);
    }

    private Map<String, Object> readSheetByCSJLUp(ActivityPlanVo planVo, String filePath) throws DataException {
        ReadExcelUtils readExcelUtils = new ReadExcelUtils(filePath);
        String[] headers = readExcelUtils.readTitle();
        Map<Integer, List<Object>> rows = readExcelUtils.readContent();

        if (headers.length != 10) {
            throw new DataException(RCode.ACTIVITY_PLAN_TEM_ERROR);
        }
        if (CollectionUtil.isEmpty(rows)) {
            throw new DataException(RCode.DATA_NOT_HAVE_ERROR);
        }

        boolean isEmpty = false;
        ArrayList<Integer> indexList = new ArrayList<>();
        for (Map.Entry<Integer, List<Object>> row : rows.entrySet()) {
            List<Object> cells = row.getValue();
            // 判断当前行是否有任一数据为空，如果有则报错，不进行解析
            isEmpty = (cells.size() != 10) || cells.stream().anyMatch(cell -> ObjectUtil.isNull(cell) || StringUtils.isBlank(cell.toString()));
            if (isEmpty) {
                indexList.add(row.getKey() + 1);
            }
        }
        if (isEmpty) {
            throw new DataException(RCode.ACTIVITY_PLAN_CELL_NOT_NULL, indexList);
        }

        Map<String, Object> rMap = new HashMap<>();
        rMap.put("uuid", UUID.randomUUID().toString());
        List<ActivityPlanInfoDto> rDtos = new ArrayList<>(rows.size() * 2);
        for (Map.Entry<Integer, List<Object>> row : rows.entrySet()) {
            List<ActivityPlanInfoDto> infoDtos = getRowByCSJLUp(planVo, row.getValue(),rMap);
            rDtos.addAll(infoDtos);
        }
        rMap.put("table", rDtos);
        return rMap;
    }

    /**
     * 城市经理上传的计划工号只能是自己
     *
     * @param row 表中的每一行数据
     * @return
     */
    private List<ActivityPlanInfoDto> getRowByCSJLUp(ActivityPlanVo planVo, List<Object> row,Map<String, Object> rMap) {
        List<ActivityPlanInfoDto> dtos = new ArrayList<>();
        /**
         * 5：多日期
         */
        String days = row.get(5).toString();
        String[] dayArr = days.split("、");
        for (int i = 0; i < dayArr.length; i++) {
            ActivityPlanInfoDto dto = new ActivityPlanInfoDto();
            try {
                String dayStr = dayArr[i];
                int day = Integer.parseInt(dayStr);

                /**
                 * 0：工号
                 */
                Object empNoO = row.get(0);
                if (!planVo.getEmployeeNo().equals(empNoO.toString())) {
                    dto.setErrorMsg("请填写您自己的工号；");
                } else {
                    dto.setEmployeeId(planVo.getEmployeeId());
                    dto.setEmployeeNo(planVo.getEmployeeNo());
                    dto.setEmployeeName(planVo.getEmployeeName());
                }

                /**
                 *  1：门店编码
                 */
                String sc = row.get(1).toString();
                StoreWrapper storeWrap = new StoreWrapper().setStoreCode(sc);
                QinCeClienteleStoreDto storeDto = qinCeClienteleStoreDao.getOneStore(storeWrap);
                if (ObjectUtil.isEmpty(storeDto)) {
                    dto.setErrorMsg("门店编码错误；");
                } else {
                    // todo 门店一定有系统名称？
                    dto.setLineName(storeDto.getLineName());
                    dto.setStoreCode(sc);
                    dto.setStoreName(storeDto.getStoreName());
                    dto.setDealerId(storeDto.getDealerId());
                    dto.setDealerName(storeDto.getDealersName());
                    dto.setProvince(storeDto.getStoreMssProvince());
                    dto.setCity(storeDto.getStoreMssCity());
                    dto.setArea(storeDto.getStoreMssArea());
                    dto.setAddr(storeDto.getStoreAddr());
                }

                /**
                 * 3：活动模式
                 */
                String pattern = row.get(3).toString();
                if ("单点CP,常规MINI秀,校园活动".contains(pattern)) {
                    dto.setPattern(pattern);
                } else {
                    dto.setErrorMsg("活动模式分为：单点CP、常规MINI秀、校园活动；");
                }


                /**
                 * 4：月份
                 * 年份 todo 以后判断是否跨年
                 */
                int year = DateUtil.thisYear();
                Month month = Month.of(Integer.parseInt(row.get(4).toString()));
                LocalDate planDate = LocalDate.of(year, month, day);
                dto.setYear(year).setMonth(planDate.getMonthValue()).setDate(DateUtils.parseDateBylocalDate(planDate));

                if (planDate.isBefore(LocalDate.now())) {
                    dto.setErrorMsg("不能是过去日期；");
                }

                /**
                 * 6：促销员上班时间
                 */
                LocalTime inLocalTime = DateUtils.parseLocalTimeByEmdtime(row.get(6).toString());

                /**
                 * 7：促销员下班时间
                 */
                LocalTime outLocalTime = DateUtils.parseLocalTimeByEmdtime(row.get(7).toString());

                if (ObjectUtils.anyNull(inLocalTime, outLocalTime)) {
                    dto.setErrorMsg("上下班时间必填；");
                }

                if (ObjectUtils.allNotNull(inLocalTime, outLocalTime) && inLocalTime.isAfter(outLocalTime)) {
                    dto.setErrorMsg("下班时间需晚于上班时间；");
                } else {
                    dto.setClockInTime(LocalDateTime.of(planDate, inLocalTime));
                    dto.setClockOutTime(LocalDateTime.of(planDate, outLocalTime));
                }

                // 8：工资
                dto.setSalary(new BigDecimal(row.get(8).toString()));
                // 9：杂费
                dto.setIncidentals(new BigDecimal(row.get(9).toString()));

            } catch (DateTimeException e) {
                dto.setErrorMsg("月份、日期、时间需要调整；");
            }
            if (StringUtils.isNotBlank(dto.getErrorMsg())){
                rMap.remove("uuid");
            }
            dtos.add(dto);
        }

        return dtos;
    }

}
