package com.wangxiaolu.promotion.controller.activity.temporary;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.wangxiaolu.promotion.common.redis.RedisKeys;
import com.wangxiaolu.promotion.common.redis.service.RedisCache;
import com.wangxiaolu.promotion.enums.activity.ClockType;
import com.wangxiaolu.promotion.exception.FlowException;
import com.wangxiaolu.promotion.exception.ParamException;
import com.wangxiaolu.promotion.pojo.activity.temporary.dto.TemporaryClockDto;
import com.wangxiaolu.promotion.pojo.activity.temporary.vo.TemporaryClockVo;
import com.wangxiaolu.promotion.result.basedata.R;
import com.wangxiaolu.promotion.result.basedata.RCode;
import com.wangxiaolu.promotion.service.activity.temporary.TemporaryActivityClockCoreService;
import com.wangxiaolu.promotion.service.activity.temporary.TemporaryActivityCoreService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.Objects;

/**
 * @author : liqiulin
 * @date : 2024-04-17 19
 * @describe : 促销员打卡信息保存
 */
@Slf4j
@RestController
@RequestMapping("/activity/temporary/core")
public class TemporaryActivityClockCoreController {

    @Autowired
    private TemporaryActivityClockCoreService tempActivityClockCoreService;
    @Autowired
    private RedisCache redisCache;

    /**
     * 促销员当日打卡信息保存
     */
    @PostMapping("/today/clock")
    public R clockInTodayActivity(@RequestBody TemporaryClockVo clockVo) {
        clockVo.validate();
        Integer clockType = clockVo.getClockType();
        boolean isClockIn = ClockType.TEMPORARY_CLOCK_IN.equals(clockType);
        // 上班卡必需有店铺id
        if (isClockIn && (Objects.isNull(clockVo.getStoreId()) || clockVo.getStoreId() <= 0)) {
            throw new ParamException(RCode.NOT_CLOCK_STORE_ERROR, null);
        }
        // 非上班卡必需有打卡记录ID
        if (!isClockIn && Objects.isNull((clockVo.getId()))) {
            throw new ParamException(RCode.CLOCK_DETAIL_ERROR, null);
        }

        // 根据打卡经纬度判断与打卡店铺的距离，距离超过100米不能打卡
//        tempActivityClockCoreService.clockStoreCalDistance(clockVo.getStoreQcId(),clockVo.getId(),clockVo.getClockCoordinates());

        // 限制重复提交
        repetitiveClock(clockVo);

        Date clockTime = new Date();
        TemporaryClockDto dto = new TemporaryClockDto(clockVo.getClockType(), clockVo.getId(), clockVo.getTemporaryId(), clockVo.getTemporaryName(), clockVo.getClockProvince(), clockVo.getClockCity());
        // 上班卡、午休下班卡、午休上班卡、下班卡
        if (isClockIn) {
            builderClockInData(clockVo, dto, clockTime);
        } else if (ClockType.TEMPORARY_NOON_CLOCK_OUT.equals(clockType)) {
            builderNoonClockOutData(clockVo, dto, clockTime);
        } else if (ClockType.TEMPORARY_NOON_CLOCK_IN.equals(clockType)) {
            builderNoonClockInData(clockVo, dto, clockTime);
        } else if (ClockType.TEMPORARY_CLOCK_OUT.equals(clockType)) {
            builderClockOutData(clockVo, dto, clockTime);
        }
        tempActivityClockCoreService.clockInTodayActivity(dto, clockType);
        return R.success();
    }

    /**
     * 促销员-打卡（上班卡）
     * @param clockVo
     * @return
     */

    @PostMapping("/today/v2/clock")
    public R clockInTodayActivityV2(@RequestBody TemporaryClockVo clockVo) {
        clockVo.validate();
        Integer clockType = clockVo.getClockType();
        boolean isClockIn = ClockType.TEMPORARY_CLOCK_IN.equals(clockType);
        // 上班卡必需有促销计划ID
        if (isClockIn && Objects.isNull(clockVo.getPlanId())) {
            throw new ParamException(RCode.NOT_CLOCK_STORE_ERROR, null);
        }
        // 非上班卡必需有打卡记录ID
        if (!isClockIn && Objects.isNull((clockVo.getId()))) {
            throw new ParamException(RCode.CLOCK_DETAIL_ERROR, null);
        }

        // 根据打卡经纬度判断与打卡店铺的距离，距离超过100米不能打卡
//        tempActivityClockCoreService.clockStoreCalDistance(clockVo.getStoreQcId(),clockVo.getId(),clockVo.getClockCoordinates());

        // 限制重复提交
        repetitiveClock(clockVo);

        Date clockTime = new Date();
        TemporaryClockDto dto = new TemporaryClockDto(clockVo.getClockType(), clockVo.getId(), clockVo.getTemporaryId(), clockVo.getTemporaryName(), clockVo.getClockProvince(), clockVo.getClockCity());
        // 上班卡、午休下班卡、午休上班卡、下班卡
        if (isClockIn) {
            builderClockInData(clockVo, dto, clockTime);
        } else if (ClockType.TEMPORARY_NOON_CLOCK_OUT.equals(clockType)) {
            builderNoonClockOutData(clockVo, dto, clockTime);
        } else if (ClockType.TEMPORARY_NOON_CLOCK_IN.equals(clockType)) {
            builderNoonClockInData(clockVo, dto, clockTime);
        } else if (ClockType.TEMPORARY_CLOCK_OUT.equals(clockType)) {
            builderClockOutData(clockVo, dto, clockTime);
        }
        tempActivityClockCoreService.clockInTodayPlan(dto, clockType);
        return R.success(dto);
    }

    /**
     * 打卡照片更换
     *
     * @param clockVo 更换照片信息
     * @return 是否成功
     */
    @PutMapping("/today/clock/update")
    public R updateClockPhoto(@RequestBody TemporaryClockVo clockVo) {
        if (Objects.isNull(clockVo.getTemporaryId()) || clockVo.getTemporaryId() <= 0) {
            throw new ParamException(RCode.LOGIN_PARAM_ERROR, null);
        }

        if (StringUtils.isBlank(clockVo.getClockPhoto())) {
            throw new ParamException(RCode.NOT_CLOCK_PHOTO_ERROR, null);
        }

        long minuteBetween = DateUtil.between(clockVo.getLastClockTime(), new Date(), DateUnit.MINUTE);
        if (minuteBetween > 30) {
            throw new ParamException(RCode.UPDATE_CLOCK_PHOTO_TIME_LONG_ERROR, null);
        }

        TemporaryClockDto dto = new TemporaryClockDto(clockVo.getId(), clockVo.getTemporaryId(), clockVo.getClockPhoto(), clockVo.getPhotoType());
        tempActivityClockCoreService.updateClockPhoto(dto);
        return R.success();
    }

    /**
     * redis打卡-限制重复提交
     */
    private void repetitiveClock(TemporaryClockVo clockVo) {
        String clockRecordKey = RedisKeys.TemporaryKeys.TEMPORARY_CLOCK_RECIRD.getKey() + clockVo.getTemporaryId() + "_" + clockVo.getClockType();
        String record = redisCache.get(clockRecordKey);
        if (Objects.nonNull(record)) {
            throw new FlowException(RCode.USER_REPETITIVE_CLOCK_MINUTE.getCode(), String.format(RCode.USER_REPETITIVE_CLOCK_MINUTE.getMsg(), "2"));
        }
        redisCache.addToJsonToMinute(clockRecordKey, clockVo.getTemporaryName(), 2);
    }


    // 上班打卡
    private void builderClockInData(TemporaryClockVo clockVo, TemporaryClockDto clockDto, Date dateTime) {
        clockDto.setPlanId(clockVo.getPlanId())
                .setStoreId(clockVo.getStoreId())
                .setStoreName(clockVo.getStoreName())
                .setClockInAddress(clockVo.getClockAddress())
                .setClockInCoordinates(clockVo.getClockCoordinates())
                .setClockInPhoto(clockVo.getClockPhoto())
                .setClockInTime(dateTime);
    }

    // 午休下班卡
    private void builderNoonClockOutData(TemporaryClockVo clockVo, TemporaryClockDto clockDto, Date dateTime) {
        clockDto.setId(clockVo.getId())
                .setNoonClockOutAddress(clockVo.getClockAddress())
                .setNoonClockOutCoordinates(clockVo.getClockCoordinates())
                .setNoonClockOutPhoto(clockVo.getClockPhoto())
                .setNoonClockOutTime(dateTime);

    }

    // 午休上班卡
    private void builderNoonClockInData(TemporaryClockVo clockVo, TemporaryClockDto clockDto, Date dateTime) {
        clockDto.setId(clockVo.getId())
                .setNoonClockInAddress(clockVo.getClockAddress())
                .setNoonClockInCoordinates(clockVo.getClockCoordinates())
                .setNoonClockInPhoto(clockVo.getClockPhoto())
                .setNoonClockInTime(dateTime);
    }

    // 下班卡
    private void builderClockOutData(TemporaryClockVo clockVo, TemporaryClockDto clockDto, Date dateTime) {
        clockDto.setId(clockVo.getId())
                .setClockOutAddress(clockVo.getClockAddress())
                .setClockOutCoordinates(clockVo.getClockCoordinates())
                .setClockOutPhoto(clockVo.getClockPhoto())
                .setClockOutTime(dateTime);
    }


}
