package com.wangxiaolu.promotion.websocket;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.TextWebSocketHandler;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.*;


@Slf4j
@Component
public class TemporaryActivityTaskClockSocketHandler extends TextWebSocketHandler {

    // 存储在线会话（用户ID -> 会话）
    private static final Map<String, WebSocketSession> sessions = new ConcurrentHashMap<>();
    // 心跳检测线程池
    private final ScheduledExecutorService heartBeatExecutor = new ScheduledThreadPoolExecutor(
    1,
    r -> {
        Thread t = new Thread(r, "TemporaryActivityTaskClock-HeartBeat");
        t.setDaemon(false);
        return t;
    },
    new ThreadPoolExecutor.AbortPolicy()
);

    private final ObjectMapper objectMapper = new ObjectMapper();

    public TemporaryActivityTaskClockSocketHandler() {
        // 初始化心跳检测（每30秒发送一次ping）
        heartBeatExecutor.scheduleAtFixedRate(this::sendHeartBeat, 30, 30, TimeUnit.SECONDS);
    }

    /**
     * 连接建立后调用
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) throws Exception {
        // 从握手属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        sessions.put(userId, session);
        log.info("用户[" + userId + "]连接成功，当前在线：" + sessions.size());
        // 发送连接成功消息
        session.sendMessage(new TextMessage("连接成功，用户ID：" + userId));
    }

    /**
     * 收到前端消息时调用
     */
    @Override
    protected void handleTextMessage(WebSocketSession session, TextMessage message) throws Exception {
        String userId = (String) session.getAttributes().get("userId");
        String payload = message.getPayload();
        log.info("收到用户[" + userId + "]的消息：" + payload);

        // 处理心跳响应（前端收到ping后返回pong）
        if ("pong".equals(payload)) {
            log.info("用户[" + userId + "]心跳正常");
            return;
        }
        // 业务处理
        MessageBean messageBean = objectMapper.readValue(payload, MessageBean.class);
        handleMessageType(messageBean);
        // 业务消息处理（示例：广播消息）
        broadcast("用户[" + userId + "]：" + payload);
    }
    private void handleMessageType(MessageBean messageBean) {
        switch (messageBean.getType()) {
            case "ACTIVITY_START":
//                startActivity(messageBean.getContent());
                break;
            case "ACTIVITY_END":
//                endActivity(messageBean.getContent());
                break;
            default:
                log.warn("未知消息类型: {}", messageBean.getType());
        }
    }
    /**
     * 连接关闭时调用
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) throws Exception {
        String userId = (String) session.getAttributes().get("userId");
        sessions.remove(userId);
        log.info("用户[" + userId + "]断开连接，当前在线：" + sessions.size());
    }

    /**
     * 发送心跳（ping）
     */
    private void sendHeartBeat() {
        for (Map.Entry<String, WebSocketSession> entry : sessions.entrySet()) {
            WebSocketSession session = entry.getValue();
            String userId = entry.getKey();
            if (session.isOpen()) {
                try {
                    session.sendMessage(new TextMessage("ping"));
                } catch (IOException e) {
                    log.error("用户[" + userId + "]心跳发送失败，强制断开");
                    try {
                        session.close();
                    } catch (IOException ex) {
                        ex.printStackTrace();
                    }
                    sessions.remove(userId);
                }
            }
        }
    }

    /**
     * 广播消息给所有在线用户
     */
    public void broadcast(String message) throws IOException {
        for (WebSocketSession session : sessions.values()) {
            if (session.isOpen()) {
                session.sendMessage(new TextMessage(message));
            }
        }
    }

    /**
     * 向指定用户发送消息
     */
    public void sendToUser(String userId, String message) throws IOException {
        WebSocketSession session = sessions.get(userId);
        if (session != null && session.isOpen()) {
            session.sendMessage(new TextMessage(message));
        }
    }
}
