package com.wangxiaolu.promotion.websocket;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.wangxiaolu.promotion.websocket.pojo.MessageBean;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.*;


@Slf4j
//@Component
public class TemporaryActivityTaskClockSocketHandler extends TextWebSocketHandler {

    // 存储在线会话（用户ID -> 会话）
    private static final ConcurrentHashMap<String, WebSocketSession> sessions = new ConcurrentHashMap<>();
    // 心跳检测线程池
    private final ScheduledExecutorService heartBeatExecutor = new ScheduledThreadPoolExecutor(
    1,
    r -> {
        Thread t = new Thread(r, "TemporaryActivityTaskClock-HeartBeat");
        t.setDaemon(false);
        return t;
    },
    new ThreadPoolExecutor.AbortPolicy()
);

    private final ObjectMapper objectMapper = new ObjectMapper();

    public TemporaryActivityTaskClockSocketHandler() {
        log.info("TemporaryActivityTaskClockSocketHandler 初始化");
        // 初始化心跳检测（每30秒发送一次ping）
        heartBeatExecutor.scheduleAtFixedRate(this::sendHeartBeat, 30, 30, TimeUnit.SECONDS);
    }

    /**
     * 连接建立后调用
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) throws Exception {
        // 从握手属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        sessions.put(userId, session);
        log.info("用户[" + userId + "]连接成功，当前在线：" + sessions.size());
        // 发送连接成功消息
        session.sendMessage(new TextMessage("连接成功，用户ID：" + userId));
    }

    /**
     * 收到前端消息时调用
     */
    @Override
    protected void handleTextMessage(WebSocketSession session, TextMessage message)   {
        String userId = (String) session.getAttributes().get("userId");
        String payload = message.getPayload();
        log.info("收到用户[" + userId + "]的消息：" + payload);

        // 处理心跳响应（前端收到ping后返回pong）
        if ("pong".equals(payload)) {
            log.info("用户[" + userId + "]心跳正常");
            return;
        }
        // 判断payload 是否是json
        if (payload.startsWith("{")) {
            MessageBean messageBean = null;
            try {
                messageBean = objectMapper.readValue(payload, MessageBean.class);
            } catch (JsonProcessingException e) {
                 log.error("用户[" + userId + "]消息格式错误：" + payload);
                 sendToUser(userId, "用户[" + userId + "]消息格式错误：" );
            }
//            handleMessageType(messageBean);
        }
    }
    private void handleMessageType(MessageBean messageBean) {
        switch (messageBean.getMsgType()) {
            case "ACTIVITY_START":
//                startActivity(messageBean.getContent());
                break;
            case "ACTIVITY_END":
//                endActivity(messageBean.getContent());
                break;
            default:
                log.warn("未知消息类型: {}", messageBean.getMsgType());
        }
    }
    /**
     * 连接关闭时调用
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) throws Exception {
        String userId = (String) session.getAttributes().get("userId");
        sessions.remove(userId);
        log.info("用户[" + userId + "]断开连接，当前在线：" + sessions.size());
    }

    /**
     * 发送心跳（ping）
     */
    private  void sendHeartBeat() {
        for (Map.Entry<String, WebSocketSession> entry : sessions.entrySet()) {
            WebSocketSession session = entry.getValue();
            String userId = entry.getKey();
            if (session.isOpen()) {
                try {
                    session.sendMessage(new TextMessage("ping"));
                } catch (IOException e) {
                    log.error("用户[" + userId + "]心跳发送失败，强制断开");
                    try {
                        session.close();
                    } catch (IOException ex) {
                        log.error("关闭会话失败", ex);
                        ex.printStackTrace();
                    }
                    sessions.remove(userId);
                }
            }
        }
    }

    /**
     * 广播消息给所有在线用户
     */
    public void broadcast(String message) throws IOException {
        for (WebSocketSession session : sessions.values()) {
            if (session.isOpen()) {
                session.sendMessage(new TextMessage(message));
            }
        }
    }

    /**
     * 向指定用户发送消息
     */
    public Integer sendToUser(String userId, String message)   {
        WebSocketSession session = sessions.get(userId);
        if (session != null && session.isOpen()) {
            try {
                session.sendMessage(new TextMessage(message));
            } catch (IOException e) {
                log.error("用户[" + userId + "]发送字符消息失败：" + message);
                 return 0;
            }
            return 1;
        }
        return 0;
    }

    public Integer sendToUser(String userId, MessageBean messageBean)  {
        WebSocketSession session = sessions.get(userId);
        if (session != null && session.isOpen()) {
            try {
                String jsonMessage = objectMapper.writeValueAsString(messageBean);
                session.sendMessage(new TextMessage(jsonMessage));
            } catch (IOException e) {
                log.error("用户[" + userId + "]发送Bean消息失败：" + e.getMessage());
                return 0;
            }
            return 1;
        }else{
            log.warn("发送信息失败：用户[" + userId + "]不存在链接" );
            return 0;
        }
    }

    public void handleTransportError(WebSocketSession session, Throwable exception) throws Exception {
        log.error("用户[" + session.getAttributes().get("userId") + "]异常断开");

    }
}
