package com.wangxiaolu.promotion.utils;

import com.google.gson.Gson;
import okhttp3.ConnectionPool;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 腾讯地图工具类（整合：逆地理编码 + 省市区三级嵌套列表）
 */
@Component
public class TencentMapUtil {
    // 你的腾讯API Key
    private static final String API_KEY = "UTEBZ-UJ3KG-OORQO-QT3PT-JDWU7-YRBZA";

    // 1. 逆地理编码接口
    private static final String GEOCODER_URL = "https://apis.map.qq.com/ws/geocoder/v1/?location=%f,%f&key=%s&get_poi=0";
    // 2. 省市区三级嵌套接口（新版结构）
    private static final String DISTRICT_TRIPLE_URL = "https://apis.map.qq.com/ws/district/v1/list?key=%s&struct_type=1";

    // OkHttp客户端
    private static final OkHttpClient OK_HTTP_CLIENT = new OkHttpClient.Builder()
            .connectTimeout(10, TimeUnit.SECONDS)
            .readTimeout(10, TimeUnit.SECONDS)
            .connectionPool(new ConnectionPool(10, 30, TimeUnit.SECONDS))
            .retryOnConnectionFailure(true)
            .build();
    private static final Gson GSON = new Gson();

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    // ==================== 1. 逆地理编码实体（保留原有） ====================
    public static class AddressInfo {
        private String province;    // 省份
        private String city;        // 城市
        private String district;    // 区县
        private String street;      // 街道
        private String streetNumber;// 门牌号
        private String fullAddress; // 完整地址
        private double lng;         // 经度
        private double lat;         // 纬度

        // Getter & Setter
        public String getProvince() { return province; }
        public void setProvince(String province) { this.province = province; }
        public String getCity() { return city; }
        public void setCity(String city) { this.city = city; }
        public String getDistrict() { return district; }
        public void setDistrict(String district) { this.district = district; }
        public String getStreet() { return street; }
        public void setStreet(String street) { this.street = street; }
        public String getStreetNumber() { return streetNumber; }
        public void setStreetNumber(String streetNumber) { this.streetNumber = streetNumber; }
        public String getFullAddress() { return fullAddress; }
        public void setFullAddress(String fullAddress) { this.fullAddress = fullAddress; }
        public double getLng() { return lng; }
        public void setLng(double lng) { this.lng = lng; }
        public double getLat() { return lat; }
        public void setLat(double lat) { this.lat = lat; }
    }

    // ==================== 2. 三级嵌套列表实体（匹配前端） ====================
    public static class RegionNode {
        private String text;       // 全称（如“北京市”）
        private String value;      // adcode（如“110000”）
        private List<RegionNode> children; // 下级节点

        public RegionNode() {
            this.children = new ArrayList<>();
        }

        public RegionNode(String text, String value) {
            this.text = text;
            this.value = value;
            this.children = new ArrayList<>();
        }

        // Getter & Setter
        public String getText() { return text; }
        public void setText(String text) { this.text = text; }
        public String getValue() { return value; }
        public void setValue(String value) { this.value = value; }
        public List<RegionNode> getChildren() { return children; }
        public void setChildren(List<RegionNode> children) { this.children = children; }
    }

    // ==================== 腾讯API内部解析实体 ====================
    // 逆地理编码响应
    private static class GeoResponse {
        private int status;
        private String message;
        private GeoResult result;

        // Getter & Setter
        public int getStatus() { return status; }
        public void setStatus(int status) { this.status = status; }
        public String getMessage() { return message; }
        public void setMessage(String message) { this.message = message; }
        public GeoResult getResult() { return result; }
        public void setResult(GeoResult result) { this.result = result; }
    }

    private static class GeoResult {
        private String formatted_address;
        private AddressComponent address_component;

        // Getter & Setter
        public String getFormatted_address() { return formatted_address; }
        public void setFormatted_address(String formatted_address) { this.formatted_address = formatted_address; }
        public AddressComponent getAddress_component() { return address_component; }
        public void setAddress_component(AddressComponent address_component) { this.address_component = address_component; }
    }

    private static class AddressComponent {
        private String province;
        private String city;
        private String district;
        private String street;
        private String street_number;

        // Getter & Setter
        public String getProvince() { return province; }
        public void setProvince(String province) { this.province = province; }
        public String getCity() { return city; }
        public void setCity(String city) { this.city = city; }
        public String getDistrict() { return district; }
        public void setDistrict(String district) { this.district = district; }
        public String getStreet() { return street; }
        public void setStreet(String street) { this.street = street; }
        public String getStreet_number() { return street_number; }
        public void setStreet_number(String street_number) { this.street_number = street_number; }
    }

    // 三级嵌套响应
    private static class DistrictResponse {
        private int status;
        private String message;
        private List<ProvinceItem> result;

        // Getter & Setter
        public int getStatus() { return status; }
        public void setStatus(int status) { this.status = status; }
        public String getMessage() { return message; }
        public void setMessage(String message) { this.message = message; }
        public List<ProvinceItem> getResult() { return result; }
        public void setResult(List<ProvinceItem> result) { this.result = result; }
    }

    private static class ProvinceItem {
        private String id;
        private String fullname;
        private List<CityItem> districts;

        // Getter & Setter
        public String getId() { return id; }
        public void setId(String id) { this.id = id; }
        public String getFullname() { return fullname; }
        public void setFullname(String fullname) { this.fullname = fullname; }
        public List<CityItem> getDistricts() { return districts; }
        public void setDistricts(List<CityItem> districts) { this.districts = districts; }
    }

    private static class CityItem {
        private String id;
        private String fullname;
        private List<DistrictItem> districts;

        // Getter & Setter
        public String getId() { return id; }
        public void setId(String id) { this.id = id; }
        public String getFullname() { return fullname; }
        public void setFullname(String fullname) { this.fullname = fullname; }
        public List<DistrictItem> getDistricts() { return districts; }
        public void setDistricts(List<DistrictItem> districts) { this.districts = districts; }
    }

    private static class DistrictItem {
        private String id;
        private String fullname;

        // Getter & Setter
        public String getId() { return id; }
        public void setId(String id) { this.id = id; }
        public String getFullname() { return fullname; }
        public void setFullname(String fullname) { this.fullname = fullname; }
    }

    // ==================== 核心功能方法 ====================
    /**
     * 1. 逆地理编码：经纬度转地址（保留原有功能）
     */
    public AddressInfo getAddressByLngLat(double lng, double lat) {
        AddressInfo addressInfo = new AddressInfo();
        try {
            // 拼接请求地址（注意：腾讯接口是 lat,lng 顺序）
            String requestUrl = String.format(GEOCODER_URL, lat, lng, API_KEY);
            Request request = new Request.Builder().url(requestUrl).get().build();

            try (Response response = OK_HTTP_CLIENT.newCall(request).execute()) {
                if (!response.isSuccessful()) {
                    throw new IOException("逆地理编码请求失败，响应码：" + response.code());
                }

                GeoResponse resp = GSON.fromJson(response.body().string(), GeoResponse.class);
                if (resp.getStatus() != 0) {
                    throw new IOException("逆地理编码API失败：" + resp.getMessage());
                }

                // 解析地址信息
                GeoResult result = resp.getResult();
                AddressComponent component = result.getAddress_component();

                addressInfo.setProvince(component.getProvince());
                addressInfo.setCity(component.getCity());
                addressInfo.setDistrict(component.getDistrict());
                addressInfo.setStreet(component.getStreet());
                addressInfo.setStreetNumber(component.getStreet_number());
                addressInfo.setFullAddress(result.getFormatted_address());
                addressInfo.setLng(lng);
                addressInfo.setLat(lat);
            }
        } catch (Exception e) {
            System.err.println("逆地理编码失败：" + e.getMessage());
            e.printStackTrace();
        }
        return addressInfo;
    }

    /**
     * 2. 获取省→市→区三级嵌套列表（匹配前端）
     */
    public List<RegionNode> getTripleLevelRegion() {
        List<RegionNode> regionList = new ArrayList<>();
        try {
            String requestUrl = String.format(DISTRICT_TRIPLE_URL, API_KEY);
            Request request = new Request.Builder().url(requestUrl).get().build();

            try (Response response = OK_HTTP_CLIENT.newCall(request).execute()) {
                if (!response.isSuccessful()) {
                    throw new IOException("三级区划请求失败，响应码：" + response.code());
                }

                DistrictResponse resp = GSON.fromJson(response.body().string(), DistrictResponse.class);
                if (resp.getStatus() != 0) {
                    throw new IOException("三级区划API失败：" + resp.getMessage());
                }

                // 转换为前端结构
                if (resp.getResult() != null && !resp.getResult().isEmpty()) {
                    for (ProvinceItem province : resp.getResult()) {
                        RegionNode provinceNode = new RegionNode(province.getFullname(), province.getId());

                        if (province.getDistricts() != null && !province.getDistricts().isEmpty()) {
                            for (CityItem city : province.getDistricts()) {
                                RegionNode cityNode = new RegionNode(city.getFullname(), city.getId());

                                if (city.getDistricts() != null && !city.getDistricts().isEmpty()) {
                                    for (DistrictItem district : city.getDistricts()) {
                                        cityNode.getChildren().add(new RegionNode(district.getFullname(), district.getId()));
                                    }
                                }
                                provinceNode.getChildren().add(cityNode);
                            }
                        }
                        regionList.add(provinceNode);
                    }
                }
            }
        } catch (Exception e) {
            System.err.println("生成三级区划失败：" + e.getMessage());
            e.printStackTrace();
        }
        return regionList;
    }

    // ==================== 缓存预热 ====================
    @PostConstruct
    public void preloadCache() {
        // 预热三级区划缓存
        new Thread(() -> {
            try {
                List<RegionNode> regionList = getTripleLevelRegion();
                if (!regionList.isEmpty()) {
                    stringRedisTemplate.opsForValue().set(
                            "promotion:district:triple_level",
                            GSON.toJson(regionList),
                            86400L,
                            TimeUnit.SECONDS
                    );
                }
            } catch (Exception e) {
                System.err.println("三级区划缓存预热失败：" + e.getMessage());
            }
        }).start();
    }
}