package com.promotion.gateway.filter;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.google.gson.Gson;
import com.promotion.gateway.data.UserDataService;
import com.wangxiaolu.promotion.common.redis.RedisKeys;
import com.wangxiaolu.promotion.result.basedata.R;
import com.wangxiaolu.promotion.result.basedata.RCode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.core.Ordered;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.http.server.reactive.ServerHttpRequestDecorator;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * @author : liqiulin
 * @date : 2024-05-23 16
 * @describe : 权限校验过滤器
 */
@Slf4j
@Component
public class AuthGlobalFilter implements GlobalFilter, Ordered {

    // 用户注册、用户登录、发送手机号验证码
    private final List<String> whiteUrls = Arrays.asList("/login", "/enroll", "/sms/send/ver_code");

    @Autowired
    UserDataService userDataService;

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, GatewayFilterChain chain) {
        long millis = System.currentTimeMillis();
        ServerHttpRequest request = exchange.getRequest();
        URI URIPath = request.getURI();
        String method = request.getMethodValue();
        HttpHeaders header = request.getHeaders();
        log.info("------------- 开始 -------------");
        log.info("\n请求request信息：\nURI : {}\nheader = {}", URIPath, header);
        String url = exchange.getRequest().getURI().getPath();

        // 1、判断url是否放行
        boolean isIgnore = whiteUrls.stream().anyMatch(url::contains);
        if (isIgnore) {
            log.info("请求放行，URL：{}", url);
            return chain.filter(exchange);
        }

        String token = exchange.getRequest().getHeaders().getFirst("Authorization");
        // 2、未登录请求跳转登录
        if (StringUtils.isBlank(token)) {
            return loginError(exchange);
        }

        // 3、验证token'
        JSONObject userJson = userDataService.getUserByToken(RedisKeys.UserKeys.TEMPORARY_TOKEN.getKey() + token);
        if (Objects.isNull(userJson)) {
            log.info("token登录错误：{}", token);
            return loginError(exchange);
        }

        if ("POST".equals(method)) {
            return DataBufferUtils.join(exchange.getRequest().getBody())
                    .flatMap(dataBuffer -> {
                        byte[] bytes = new byte[dataBuffer.readableByteCount()];
                        dataBuffer.read(bytes);
                        try {
                            String bodyString = new String(bytes, "utf-8");
                            log.info("\n请求参数：\n" + bodyString);
                            exchange.getAttributes().put("POST_BODY", bodyString);
                        } catch (UnsupportedEncodingException e) {
                            e.printStackTrace();
                        }
                        DataBufferUtils.release(dataBuffer);
                        Flux<DataBuffer> cachedFlux = Flux.defer(() -> {
                            DataBuffer buffer = exchange.getResponse().bufferFactory()
                                    .wrap(bytes);
                            return Mono.just(buffer);
                        });

                        ServerHttpRequest mutatedRequest = new ServerHttpRequestDecorator(
                                exchange.getRequest()) {
                            @Override
                            public Flux<DataBuffer> getBody() {
                                return cachedFlux;
                            }
                        };
                        log.info("------------- 结束-------------", System.currentTimeMillis() - millis);
                        return chain.filter(exchange.mutate().request(mutatedRequest)
                                .build());
                    });
        }

        return chain.filter(exchange);
    }

    @Override
    public int getOrder() {
        return -1;
    }

    private Mono<Void> loginError(ServerWebExchange exchange) {
        ServerHttpResponse response = exchange.getResponse();
        response.setStatusCode(HttpStatus.OK);
        response.getHeaders().add("Content-Type", "application/json;charset=UTF-8");
        R res = new R(RCode.NOT_LOGIN_ERROR);
        DataBuffer buffer = response.bufferFactory().wrap(new Gson().toJson(res).getBytes(StandardCharsets.UTF_8));
        log.info("------------- 结束 -------------");
        return response.writeWith(Flux.just(buffer));
    }

    private void logSave() {

    }

}
