package com.promotion.gateway.filter;

import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.google.gson.Gson;
import com.promotion.gateway.util.JwtTokenUtils;
import com.promotion.gateway.constant.RCode;
import com.promotion.gateway.constant.StatusCode;
import com.promotion.gateway.data.UserDataService;
import com.wangxiaolu.promotion.result.basedata.R;
import io.jsonwebtoken.Claims;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.core.Ordered;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.http.HttpStatus;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;

/**
 * @author : liqiulin
 * @date : 2024-05-23 16
 * @describe : 权限校验过滤器
 */
@Slf4j
@Component
public class AuthGlobalFilter implements GlobalFilter, Ordered {

    /**
     * 放行URL
     * 用户注册、用户登录、发送手机号验证码
     */
    private final List<String> whiteUrls = Arrays.asList("/login", "/enroll", "/sms/send/ver_code","/user/aliyun/signature","/wechat/miniapp/wxtoken","/wechat/miniapp/phone","/wechat/miniapp/openid","/temporary/findStoreList");

    @Autowired
    private UserDataService userDataService;

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, GatewayFilterChain chain) {
        String url = exchange.getRequest().getURI().getPath();

        // 1、判断url是否放行
        boolean isIgnore = whiteUrls.stream().anyMatch(url::contains);
        if (isIgnore) {
            return chain.filter(exchange);
        }

        // 2、未登录请求跳转登录
        String token = exchange.getRequest().getHeaders().getFirst("Authorization");
        if (StringUtils.isBlank(token)) {
            return loginError(exchange, RCode.NOT_LOGIN_ERROR);
        }

        // 3、验证token
        Claims claims = JwtTokenUtils.parseToken(token);
        if (claims == null)
        {
            return loginError(exchange,RCode.NOT_LOGIN_TIMEOUT_ERROR);
        }
        String userkey = JwtTokenUtils.getUserKey(claims);
        boolean islogin = userDataService.hasKeyToken(userkey);
        if (!islogin)
        {
            return loginError(exchange,RCode.NOT_LOGIN_TIMEOUT_ERROR);
        }
        String userid = JwtTokenUtils.getUserId(claims);
        String username = JwtTokenUtils.getUserName(claims);
        if (StringUtils.isBlank(userid) || StringUtils.isBlank(username))
        {
            log.error("鉴权失败,登录已过期："+token);
            return loginError(exchange,RCode.NOT_LOGIN_TIMEOUT_ERROR);
        }

        return chain.filter(exchange);
    }

    @Override
    public int getOrder() {
        return -2;
    }

    private Mono<Void> loginError(ServerWebExchange exchange, StatusCode statusCode) {
        ServerHttpResponse response = exchange.getResponse();
        response.setStatusCode(HttpStatus.OK);
        response.getHeaders().add("Content-Type", "application/json;charset=UTF-8");
        R res = new R(statusCode);
        DataBuffer buffer = response.bufferFactory().wrap(new Gson().toJson(res).getBytes(StandardCharsets.UTF_8));
        return response.writeWith(Flux.just(buffer));
    }

//    private Mono<Void> loginError(ServerWebExchange exchange, RCode statusCode) {
//        ServerHttpResponse response = exchange.getResponse();
//        response.setStatusCode(HttpStatus.OK);
//        response.getHeaders().add("Content-Type", "application/json;charset=UTF-8");
//        R res = new R(statusCode);
//        DataBuffer buffer = response.bufferFactory().wrap(new Gson().toJson(res).getBytes(StandardCharsets.UTF_8));
//        return response.writeWith(Flux.just(buffer));
//    }

}
